import time
from packaging import version

from si_tests import settings
from si_tests.utils import utils
from si_tests.deployments.utils import (extra_context, file_utils, wait_utils,
                                        commons, kubectl_utils)
from si_tests.deployments.utils.namespace import NAMESPACE
import fileinput


def is_tf_deployment():
    context = file_utils.get_osdpl_content()

    if not context["spec"]["preset"] == "compute-tf":
        return False
    return True


@utils.log_method_time()
def deploy_tungsten_fabric(os_manager, timeout):
    kubectl = kubectl_utils.Kubectl(NAMESPACE.tf)
    context = extra_context.ExtraContext()
    tf_version = settings.OPENSTACK_DEPLOY_TF_VERSION
    tf_cr_api_v2 = settings.TF_OPERATOR_API_V2
    commons.LOG.info("Wait for tungsten fabric operator to be ready")
    wait = wait_utils.Waiter(os_manager, timeout, NAMESPACE.tf)
    wait.deployment("tungstenfabric-operator")

    tf_example = "tf/tungstenfabric_v2.yaml" if tf_cr_api_v2 else \
        "tf/tungstenfabric.yaml"
    tf_cr_yaml = file_utils.get_example(tf_example)
    cr_data = file_utils.get_yaml_content(tf_cr_yaml)

    tf_cr_context = context.tf_operator
    if tf_cr_context:
        commons.LOG.info("Update tf custom resource yaml")
        if tf_cr_context.get("override") and tf_cr_context.get("spec"):
            cr_data["spec"] = tf_cr_context["spec"]
        else:
            tf_cr_context.pop("override", None)
            utils.merge_dicts(cr_data, context.tf_operator)

    cr_tf_version = cr_data["spec"].get("tfVersion") if tf_cr_api_v2 else (
        cr_data["spec"].get("settings", {}).get("tfVersion", None))
    if cr_tf_version:
        commons.LOG.info(f"TF version (from extra context): {cr_tf_version}")
    elif tf_version != "":
        commons.LOG.info(f"Set TF version (from settings): {tf_version}")
        if tf_cr_api_v2:
            cr_data["spec"]["tfVersion"] = tf_version
        else:
            cr_data["spec"]["settings"]["tfVersion"] = tf_version

    new_cr_yaml = file_utils.get_new_name(tf_cr_yaml)
    commons.LOG.info("Save new tf custom resource to %s", new_cr_yaml)
    file_utils.save_to_yaml(cr_data, new_cr_yaml)

    commons.LOG.info("Apply tf custom resource")
    kubectl.apply(new_cr_yaml)
    os_manager.tf_manager.detect_api_version()
    tfoperator_version = os_manager.tf_manager.tfoperator_version()
    commons.LOG.info(f"TF Operator version: {tfoperator_version}")
    analytics = os_manager.tf_manager.is_analytics_enabled()
    commons.LOG.info(f"TF Analytics enabled: {analytics}")
    if not analytics and tf_cr_api_v2:
        deployments = ["cassandra-operator", "rabbitmq-operator",
                       "redis-operator", "tungstenfabric-operator",
                       "zookeeper-operator"]
    else:
        # For case when we have v1 and disabled analytics we
        # still have kafka operator
        deployments = ["cassandra-operator", "rabbitmq-operator",
                       "redis-operator", "tungstenfabric-operator",
                       "zookeeper-operator", "kafka-operator"]

    for deployment in deployments:
        commons.LOG.info("Wait for %s deployment", deployment)
        wait.deployment(deployment)

    statefulsets = [("redis-tf-redis", 3), ("tf-rabbitmq", 1),
                    ("tf-cassandra-config-dc1-rack1", 1), ("tf-zookeeper", 3)]

    if analytics:
        statefulsets.extend([("tf-kafka", 3), ("tf-zookeeper-nal", 3),
                             ("tf-cassandra-analytics-dc1-rack1", 1)])

    for (statefulset, replicas) in statefulsets:
        commons.LOG.info("Wait for %s statefulset", statefulset)
        wait.statefulset(statefulset, replicas=replicas)

    def get_node_names(label):
        return [node.read().metadata.name
                for node in os_manager.api.nodes.list(
                label_selector=label)]

    analytics_nodes = get_node_names("tfanalytics=enabled")
    config_nodes = get_node_names("tfconfig=enabled")
    control_nodes = get_node_names("tfcontrol=enabled")
    webui_nodes = get_node_names("tfwebui=enabled")
    configdb_nodes = get_node_names("tfconfigdb=enabled")
    analyticsdb_nodes = get_node_names("tfanalyticsdb=enabled")
    vrouter_nodes = get_node_names("tfvrouter=enabled")
    vrouter_dpdk_nodes = get_node_names("tfvrouter-dpdk=enabled")

    daemonsets = [("tf-config", len(config_nodes)),
                  ("tf-config-db", len(configdb_nodes)),
                  ("tf-control", len(control_nodes)),
                  ("tf-webui", len(webui_nodes))]
    if analytics:
        daemonsets.extend([("tf-analytics", len(analytics_nodes)),
                          ("tf-analytics-alarm", len(analytics_nodes)),
                          ("tf-analytics-database", len(analyticsdb_nodes)),
                          ("tf-analytics-snmp", len(analytics_nodes))])
    if version.parse(tfoperator_version) >= version.parse("0.11") and not \
            os_manager.tf_manager.is_vgw_disabled():
        vrouter_vgw_nodes = get_node_names("openstack-gateway=enabled,"
                                           "tfvrouter=enabled")
        daemonsets.append(("tf-vrouter-vgw", len(vrouter_vgw_nodes)))
    commons.LOG.info("ALL expected %s daemonsets", daemonsets)
    depl = os_manager.api.deployments.get(
        'tungstenfabric-operator', NAMESPACE.tf)
    operator_labels = depl.read().metadata.labels
    commons.LOG.info(f"TFOperator labels: {operator_labels}")
    for (daemonset, replicas) in daemonsets:
        commons.LOG.info("Wait for %s daemonset", daemonset)
        wait.daemonset(daemonset, replicas=replicas)
    commons.LOG.info("Wait for tf-vrouter-agent daemonsets")
    wait.tf_vrouter_daemonsets("agent", len(vrouter_nodes))
    if vrouter_dpdk_nodes:
        wait.tf_vrouter_daemonsets("agent-dpdk", len(vrouter_dpdk_nodes))

    if version.parse(tfoperator_version) >= version.parse("0.16"):
        wait.tf_vrouter_daemonsets("provisioner", len(vrouter_nodes))
        if vrouter_dpdk_nodes:
            wait.tf_vrouter_daemonsets("provisioner-dpdk", len(vrouter_dpdk_nodes))

    commons.LOG.info(
        "Wait for openstack to be deployed to register vsrx router")
    start_time = time.time()
    os_manager.wait_openstackdeployment_health_status(timeout=wait.timeout)
    wait.update_timeout(time.time() - start_time)

    commons.LOG.info("Register vsrx router in TFconfigDB")
    vsrx_file = file_utils.get_context("tools/tf/register_resources.py")

    # Custom BGP vsrx router
    if settings.OPENSTACK_TF_VSRX_ROUTER:
        commons.LOG.info("Custom BGP vsrx router {}"
                         .format(settings.OPENSTACK_TF_VSRX_ROUTER))
        with fileinput.FileInput(vsrx_file,
                                 inplace=True, backup='.bak') as vfile:
            for line in vfile:
                print(line.replace('10.199.199.199',
                                   settings.OPENSTACK_TF_VSRX_ROUTER), end='')
    kubectl.create("configmap",
                   "register-net-objects --from-file {}".format(vsrx_file))

    commons.LOG.info("Get image for register-net-objects job")
    tf_config_pod = [x for x in os_manager.api.pods.list_starts_with(
        "tf-config", NAMESPACE.tf) if not x.name.startswith('tf-config-db')][0]

    reg_image = None

    for container in tf_config_pod.read().spec.containers:
        if container.name == "provisioner":
            reg_image = container.image
            break

    if not reg_image:
        commons.LOG.info("Image for vsrx not found")
        raise ValueError("Image for vsrx not found")

    register_job_yaml = file_utils.get_context(
        "tools/tf/register_net_objects.yaml")

    register_job_data = file_utils.get_yaml_content(register_job_yaml)
    container = register_job_data['spec']['template']['spec']['containers'][0]
    container['image'] = reg_image
    if tf_version == "24.1":
        container['command'][0] = 'python3'

    new_yaml = file_utils.get_new_name(register_job_yaml)
    file_utils.save_to_yaml(register_job_data, new_yaml)

    commons.LOG.info("Apply register_net_objects.yaml")
    kubectl.apply(new_yaml, options="-n {}".format(NAMESPACE.tf))
    commons.LOG.info("Wait for register-net-objects job")
    kubectl.wait(
        "jobs/register-net-objects --for=condition=complete --timeout=30s")

    """
    # for versions < 2.3
    commons.LOG.info("Create tungstenfabric endpoint")
    tf_config_api_name = kubectl.get(
        "service",
        "-l app=tf-config,tungstenfabric=config,"
        "component=api -o jsonpath='{.items[0].metadata.name}'",
        namespace=NAMESPACE.tf, string=True)
    tf_config_api_port = kubectl.get(
        "service",
        "-l app=tf-config,tungstenfabric=config,"
        "component=api -o jsonpath='{.items[0].spec.ports[0].port}'",
        namespace=NAMESPACE.tf, string=True)
    keystone_client_name = os_manager.api.pods.list_starts_with(
        "keystone-client", NAMESPACE.openstack)[0].read().metadata.name
    kubectl.exec(
        keystone_client_name,
        "bash -c 'openstack service create sdn --name tungstenfabric"
        " && openstack endpoint create tungstenfabric internal "
        f"http://{tf_config_api_name}.tf.svc.cluster.local:"
        f"{tf_config_api_port} --region RegionOne'",
        namespace=NAMESPACE.openstack)
    """
    commons.LOG.info("Tungsten Fabric successfully deployed")
