#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

import os

import yaml

from si_tests import logger
from si_tests import settings
from si_tests.utils import utils, waiters

LOG = logger.logger


class K0sBotstrapManager(object):
    """
    Simple k0s-installer on localhost
    """

    def __init__(self, k0s_config=None):
        self.bs_dir = os.path.join(os.environ['HOME'], 'ksi_k0s_bootstrap')
        self.k0s_kubeadm = os.path.join(self.bs_dir, 'kubeconfig.yaml')
        self.k0s_kubeadm_art = os.path.join(settings.ARTIFACTS_DIR, 'bootstrap_kubeconfig.yaml')
        self.k0s_config_dict = k0s_config or dict()
        self.k0s_config_target = '/etc/k0s/k0s.yaml'
        # ugly way to ensure folder
        self.k0s_ctd_path = '/etc/k0s/containerd.d/'
        self.executor = utils.get_local_executor()
        self.k0s_apply_registry_hack = settings.KSI_K0S_APPLY_REGISTRY_HACK
        self.k0s_version = settings.KSI_K0S_VERSION,

    def get_kubeconfig_path(self):
        return self.k0s_kubeadm

    def install_k0s_if_not(self):
        if not self._is_k0s_nodes():
            self.install_k0s()
        else:
            LOG.info('looks like k0s already installed')
        return True

    def install_k0s(self):
        """
        Install k0s, using flow from  https://get.k0s.sh
        :return:
        """
        _id = utils.get_timestamp()
        _k0s_ins = os.path.join(self.bs_dir, f'k0s_install_{_id}.sh')
        run_envs = {
            "K0S_VERSION": self.k0s_version,
            'DEBUG': 'True',
        }
        envs_string = utils.make_export_env_strting(run_envs)
        LOG.info('Attempt to install k0s')
        if not os.path.exists(self.bs_dir):
            self.executor.check_call(f"mkdir -p {self.bs_dir}", verbose=True)
        if os.path.isfile(self.k0s_kubeadm):
            LOG.warn(f'File already exists {self.k0s_kubeadm}')
            self.executor.check_call(f"mv -v {self.k0s_kubeadm} {self.k0s_kubeadm}_{_id}", verbose=True)

        LOG.info("Downloading k0s installer")
        self.executor.check_call(f"curl --proto '=https' --tlsv1.2 -sSf https://get.k0s.sh --output {_k0s_ins}",
                                 verbose=True)
        LOG.info("Install k0s")
        self.executor.check_call(f"chmod 0755 {_k0s_ins}", verbose=True)
        self.executor.check_call(f"{envs_string}; sudo {_k0s_ins}", verbose=True)
        if not os.path.exists('/etc/k0s/containerd.d'):
            self.executor.check_call(f"sudo mkdir -vp {self.k0s_ctd_path}")
        if self.k0s_apply_registry_hack:
            self.apply_registry_hack()
        self.executor.check_call(["sudo", "bash", "-c", f"k0s config create > {self.k0s_config_target}"],
                                 verbose=True)
        if self.k0s_config_dict:
            LOG.info(f"k0s merge-to-config detected:\n{self.k0s_config_dict}")
            with open(self.k0s_config_target, 'r') as f:
                _config = yaml.load(f.read(), Loader=yaml.SafeLoader)
            _config = utils.merge_dicts(_config, self.k0s_config_dict)
            self._write_file_with_sudo(path=self.k0s_config_target,
                                       content=yaml.safe_dump(_config,
                                                              default_flow_style=False,
                                                              sort_keys=False))
            LOG.info(f'Merged config saved at {self.k0s_config_target}')

        self.executor.check_call(
            f"{envs_string}; sudo k0s install controller "
            f"-c {self.k0s_config_target} "
            f"--single "
            f"--force",
            verbose=True)
        self.executor.check_call(f"{envs_string}; sudo systemctl daemon-reload", verbose=True)
        # TODO: service start=>enable?
        self.executor.check_call(f"{envs_string}; sudo k0s start", verbose=True)

        _timeout = 10 * 60
        LOG.info(f"Waiting for k0s warm-up Timeout:{_timeout}sec...")
        waiters.wait(lambda: self._is_k0s_nodes(),
                     timeout=_timeout,
                     interval=5,
                     timeout_msg="Timeout for waiting expected '_is_k0s_nodes' alive")
        LOG.info(f'dumping k0s kubeconfig to {self.k0s_kubeadm}')
        self.executor.check_call(f"sudo k0s kubeconfig admin >  {self.k0s_kubeadm}; chmod 0600 {self.k0s_kubeadm}",
                                 verbose=True)

    def _is_k0s_nodes(self):
        ret = self.executor.execute("sudo k0s kubectl get nodes", verbose=True)
        # refactor(alexz-kh): will we always have it? might be there is a better way?
        ret1 = self.executor.execute("sudo k0s kubectl get serviceaccount default", verbose=True)
        if ret.exit_code != 0 or ret1.exit_code != 0:
            return False
        return True

    def _write_file_with_sudo(self, path, content):
        """
        that must be removed, in favor executor.with_sudo_somehow logic
        :param path:
        :param content:
        :return:
        """
        cmd = [
            "sudo", "bash", "-c",
            f"cat <<'EOF' > {path}\n{content}\nEOF"
        ]
        self.executor.check_call(cmd,
                                 verbose=True)

    def save_bootstrap_kubeconfig_to_arts(self):
        LOG.info(f'Copy {self.k0s_kubeadm} -> {self.k0s_kubeadm_art}')
        self.executor.check_call(['cp', '-va', self.k0s_kubeadm, self.k0s_kubeadm_art])

    def apply_registry_hack(self):
        """
        Quick hack, to deal with dockerhub rate-limits
        :return:
        """
        LOG.warn('apply_registry_hack to k0s initiated')
        content = '''
[plugins."io.containerd.grpc.v1.cri".registry]
  [plugins."io.containerd.grpc.v1.cri".registry.mirrors]

    [plugins."io.containerd.grpc.v1.cri".registry.mirrors."docker.io"]
      endpoint = ["https://dockerproxy.artifactory-eu.mcp.mirantis.net/v2/"]

    [plugins."io.containerd.grpc.v1.cri".registry.mirrors."gcr.io"]
      endpoint = ["https://dockerproxy.artifactory-eu.mcp.mirantis.net/v2/"]

    [plugins."io.containerd.grpc.v1.cri".registry.mirrors."k8s.gcr.io"]
      endpoint = ["https://dockerproxy.artifactory-eu.mcp.mirantis.net/v2/"]

    [plugins."io.containerd.grpc.v1.cri".registry.mirrors."quay.io"]
      endpoint = ["https://dockerproxy.artifactory-eu.mcp.mirantis.net/v2/"]

    [plugins."io.containerd.grpc.v1.cri".registry.mirrors."registry.k8s.io"]
      endpoint = ["https://dockerproxy.artifactory-eu.mcp.mirantis.net/v2/"]
        '''
        self._write_file_with_sudo(path='/etc/k0s/containerd.d/registry.toml', content=content)
