import os
import yaml

from si_tests import settings
from si_tests import logger
import si_tests.utils.templates as template_utils

LOG = logger.logger


def test_add_helm_repository(kcm_manager, show_step):
    """Create a HelmRepository or GitRepository from the specified template

    Scenario:
        1. Read the repository template
        2. (optional) Read the Secret template
        3. (optional) Create the Secret from the template
        4. Create the repository from the template
    """
    repo_template_path = settings.CUSTOM_HELM_REPOSITORY_TEMPLATE_PATH
    secret_template_path = settings.CUSTOM_HELM_REPOSITORY_SECRET_TEMPLATE_PATH

    show_step(1)
    if not os.path.isfile(repo_template_path):
        raise Exception(f"Repository template file not found: '{settings.CUSTOM_HELM_REPOSITORY_TEMPLATE_PATH}'")

    options = {}
    repo_template = template_utils.render_template(
        file_path=repo_template_path,
        options=options
    )
    repo_body = yaml.load(repo_template, Loader=yaml.SafeLoader)
    repo_kind = repo_body['kind']
    repo_name = repo_body['metadata']['name']
    repo_namespace = repo_body['metadata']['namespace']
    # Possible reference to a Secret object, if specified
    expected_secret_name = (repo_body['spec'].get('secretRef', {}).get('name') or
                            repo_body['spec'].get('certSecretRef', {}).get('name'))
    kcm_manager.get_or_create_namespace(repo_namespace)

    # Get API of the supported repositories
    if repo_kind == 'HelmRepository':
        repo_api = kcm_manager.api.helmrepositories
    elif repo_kind == 'GitRepository':
        repo_api = kcm_manager.api.gitrepositories
    else:
        raise NotImplementedError(f"Repository with 'kind: {repo_kind}' is unsupported in this test. "
                                  f"Please consider to make a commit to support it.")

    show_step(2)
    if secret_template_path and os.path.isfile(secret_template_path):
        options = {}
        secret_template = template_utils.render_template(
            file_path=secret_template_path,
            options=options
        )
        # secret_template can be empty, if no username/password were specified for default template
        if secret_template:
            secret_body = yaml.load(secret_template, Loader=yaml.SafeLoader)
            secret_name = secret_body['metadata']['name']
            secret_namespace = secret_body['metadata']['namespace']
            # Check that the secret is created in the same namespace as the repository object
            assert secret_namespace == repo_namespace, (
                f"Namespaces for the repository and for the Secret are not the same. "
                f"Repo namespace is '{repo_namespace}' from the template '{repo_template_path}'. "
                f"Secret namespace is 'secret_namespace' from the template '{secret_template_path}'.")

            # Check that the secret reference name in the repository object matches the Secret object name
            assert secret_name == expected_secret_name, (
                f"Secret name '{secret_name}' from template '{secret_template_path}' doesn't match "
                f"the repository reference name to the secret '{expected_secret_name}' "
                f"in the template '{repo_template_path}'")

            show_step(3)
            if kcm_manager.api.secrets.present(name=secret_name, namespace=secret_namespace):
                if settings.CUSTOM_HELM_REPOSITORY_OVERWRITE:
                    kcm_manager.api.secrets.get(name=secret_name, namespace=secret_namespace).delete()
                else:
                    raise Exception(f"Secret '{secret_namespace}/{secret_name}' already exists")

            kcm_manager.api.secrets.create(namespace=secret_namespace, body=secret_body)
        else:
            if expected_secret_name:
                raise Exception(f"Repository template '{repo_template_path}' has a reference "
                                f"to the secret '{expected_secret_name}', "
                                f"but the Secret template file '{secret_template_path}' is empty")
            else:
                LOG.info("Secret template is empty, proceed without the secret")
    else:
        if expected_secret_name:
            raise Exception(f"Repository template '{repo_template_path}' has a reference "
                            f"to the secret '{expected_secret_name}', "
                            f"but the Secret template file '{secret_template_path}' not found")
        else:
            LOG.info("Secret template file was not provided, proceed without the secret")

    show_step(4)
    if repo_api.present(name=repo_name, namespace=repo_namespace):
        if settings.CUSTOM_HELM_REPOSITORY_OVERWRITE:
            repo_api.get(name=repo_name, namespace=repo_namespace).delete()
        else:
            raise Exception(f"Repository '{repo_kind}' '{repo_namespace}/{repo_name}' already exists")

    repo_api.create(repo_name, namespace=repo_namespace, body=repo_body)
