#    Copyright 2024 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
import os
import yaml

from si_tests import settings
from si_tests import logger
from si_tests.managers.kaas_manager import Manager

LOG = logger.logger


def get_upgrade_path(supported_crs, update_versions, path=None):
    path = path or []
    if update_versions:
        for update_version in update_versions:
            # availableUpgrades may have several options,
            # but supported_crs may not have some of them due to provider limitations
            new_data = supported_crs.get(update_version, {})
            if not new_data:
                continue
            new_path = path.copy()
            new_path.append(new_data['name'])
            new_update_versions = [
                ver['version']
                for ver in new_data.get('availableUpgrades', [])]
            for path in get_upgrade_path(supported_crs,
                                         new_update_versions,
                                         new_path):
                yield path
    else:
        yield path


def _get_kaasreleases_path(kaas_manager, cr_current, supported_crs):
    """
    Gen path for upgrade based on available upgrades in kaas release
    This method generates path according to information stored in kaas release object
    based on availableUpgrades key.
    - availableUpgrades:
      - version: 10.3.0
      name: mke-10-0-0
      providers:
        supported:
        - byo
      version: 10.0.0
    - name: mke-10-3-0
      providers:
        supported:
        - byo
      version: 10.3.0
    Returns list of lists with possible paths.
    Based on example above [[mke-10-3-0]] will be returned because we will
    search for release name where version is equal to version that described
    in version under availableUpgrades
    """
    current_versions = [data for data in supported_crs.values() if data['name'] == cr_current]
    LOG.debug(f"Found next version for update: {current_versions}")

    assert current_versions, (f"clusterrelease {cr_current} is unsupported in the current kaasrelease")
    assert len(current_versions) <= 1, (f"There is more than one record for the release '{cr_current}' "
                                        f"in kaasreleases supportedClusterReleases")
    update_versions = [ver['version'] for ver in current_versions[0].get('availableUpgrades', [])]
    if not update_versions:
        return [[cr_current]]
    paths = list(get_upgrade_path(supported_crs, update_versions))
    if paths and paths[0]:
        return paths
    else:
        return [[cr_current]]


def _get_path_for_release_from_yaml(release_name, target_release_name, kaas_manager, provider,
                                    file_dir=settings.UPGRADE_PATHS_YAML_FILE_DIR):
    path = []
    directory = os.path.dirname(file_dir)
    kaas_release = kaas_manager.get_active_kaasrelease()
    kaas_version = kaas_release.data.get('spec', {}).get('version', '')
    assert kaas_version, "Cannot get release version from the active KaaSRelease object"
    kaas_version = kaas_version.replace('-rc', '')
    filename = f"{kaas_version}" + ".yaml"
    actual_file_path = os.path.join(directory, filename)
    assert os.path.exists(actual_file_path), f"File with name {filename} is not existed in {directory}"
    LOG.info(f"Using file {actual_file_path} for getting upgrade path")

    if not target_release_name:
        LOG.warning(f"Target release is not set. Will determine max "
                    f"available version for current release as target. Current release: {release_name}")
        cluster_prefix = release_name.split('-')[0]
        max_versions = kaas_manager.si_config.version_generators.gen_latest_supported_clusterreleases(
            provider, cluster_prefix)
        if len(max_versions) == 1:
            target_release_name = max_versions[0]
        else:
            current_release_minor = '-'.join(release_name.split('-')[0:3])
            for ver in max_versions:
                if ver.startswith(current_release_minor):
                    target_release_name = ver
                    break
        assert target_release_name, f"Target release is not found for {release_name}"
        LOG.info(f"Found {target_release_name} as a target for {release_name}")

    with open(actual_file_path, 'r') as paths_file:
        file_data = yaml.safe_load(paths_file)
        LOG.debug(f"Using next data from file for getting upgrade path:\n{yaml.dump(file_data)}")
        release_name_prefix = release_name.split('-')[0]
        data = file_data['upgrade_data'][release_name_prefix]
        assert data, f"No data exists in file {actual_file_path} for {release_name_prefix} clusterreleases"

        for i in data:
            if (i.get('RELEASE') == release_name and i.get('TARGET_RELEASE') == target_release_name
                    and not i.get('OPTIONAL')):
                upgrade_path_data = [u['versions'] for u in i.get('UPGRADE_PATH') if not u['optional']][0]
                for info in upgrade_path_data:
                    r_name = info['release_name']
                    path.append(r_name)

    return path


def _generete_update_release_names_byo(kaas_manager, child_cluster):
    """
    This method checks that path for upgrade generated based on ucp tags is the same
    as path generated based on kaasrelease object. It is required to not miss
    situation when wrong versions for upgrade are passed to kaasrelease object.
    """
    cr_current = child_cluster.clusterrelease_version
    supported_crs = kaas_manager.get_supported_clusterreleases(provider=settings.BYO_PROVIDER_NAME)
    supported_crs_byo = {k: v for k, v in supported_crs.items()
                         if 'byo' in v.get('providers', {}).get('supported', [])}
    kaasreleases_paths = _get_kaasreleases_path(kaas_manager, cr_current, supported_crs=supported_crs_byo)
    assert len(kaasreleases_paths) == 1, f"More than one path for upgrade found: {kaasreleases_paths}"
    kaasreleases_path = kaasreleases_paths[0]
    for release_name in kaasreleases_path:
        yield release_name


def _generete_update_release_names_mke(kaas_manager, child_cluster):
    cr_current = child_cluster.clusterrelease_version
    target_release_name = settings.KAAS_CHILD_CLUSTER_UPDATE_RELEASE_NAME
    provider_name = child_cluster.provider.provider_name
    available_clusterreleases = kaas_manager.get_clusterrelease_names()
    predefined_upgrade_path = settings.KAAS_CHILD_CLUSTER_UPGRADE_PATH

    if predefined_upgrade_path:
        LOG.info(f"Using predefined upgrade path {predefined_upgrade_path}")
        if not target_release_name:
            target_release_name = predefined_upgrade_path[-1]
            LOG.info(f"Target release name was not set. Using {target_release_name} from given path as latest")
        if cr_current == target_release_name and not settings.KAAS_CHILD_CLUSTER_UPDATE_INFO_PATH:
            msg = (f"Skipping update: requested clusterrelease version {target_release_name} "
                   f"is the same as the current clusterrelease version {cr_current}. "
                   f"To continue with ClusterUpdatePlan steps, please specify KAAS_CHILD_CLUSTER_UPDATE_INFO_PATH")
            LOG.banner(msg)
            return

        for release_name in predefined_upgrade_path:
            assert release_name in available_clusterreleases, (
                f"Release with name {release_name} from predefined path {predefined_upgrade_path} is "
                f"not existed in available clusterreleases. "
                f"Available clusterreleases:\n{yaml.dump(available_clusterreleases)}")
            yield release_name
            if release_name == target_release_name:
                break

    elif settings.USE_UPGRADE_PATH_FROM_YAML:
        if cr_current == target_release_name and not settings.KAAS_CHILD_CLUSTER_UPDATE_INFO_PATH:
            msg = (f"Skipping update: requested clusterrelease version {target_release_name} "
                   f"is the same as the current clusterrelease version {cr_current}. "
                   f"To continue with ClusterUpdatePlan steps, please specify KAAS_CHILD_CLUSTER_UPDATE_INFO_PATH")
            LOG.banner(msg)
            return
        path = _get_path_for_release_from_yaml(cr_current, target_release_name,
                                               kaas_manager=kaas_manager,
                                               provider=provider_name)
        assert path, f"No path found for upgrading cluster {cr_current} to {target_release_name}. Chck file data"
        LOG.info(f"Select the following upgrade path to reach the release "
                 f"{target_release_name}:\n{path}")
        for release_name in path:
            assert release_name in available_clusterreleases, (
                f"Release with name {release_name} is not existed in available clusterreleases. "
                f"Available clusterreleases:\n{yaml.dump(available_clusterreleases)}")
            yield release_name
            if release_name == target_release_name:
                break

    else:
        if not target_release_name:
            LOG.info("KAAS_CHILD_CLUSTER_UPDATE_RELEASE_NAME wasn't set, trying "
                     "to acquire available clusterrelease version for upgrade")
            cr_available = child_cluster.available_clusterrelease_version

            if cr_available is None:
                LOG.warning("Cluster doesn't have available version for update in status.providerStatus.releaseRefs")
                # Check that the current cluster version is actually the latest available version in the current MCC
                supported_crs_names = kaas_manager.get_supported_clusterrelease_names(provider=provider_name)

                # Filter the release names, leaving mosk- releases for mosk, and mke- releases for mke,
                # depending on the current cluster release.
                child_version_prefix = cr_current[:3]
                supported_crs_names = [name for name in supported_crs_names if name.startswith(child_version_prefix)]

                LOG.info(f"Supported version names: {supported_crs_names}")

                latest_supported_cr = supported_crs_names[-1]
                assert latest_supported_cr == cr_current, (
                    f"Version for upgrade wasn't found: Cluster '{child_cluster.namespace}/{child_cluster.name}' "
                    f"current version is '{cr_current}' , latest available clusterrelease version in MCC "
                    f"for the provider '{provider_name}' is '{latest_supported_cr}' , "
                    f"but the Cluster don't have available version for upgrade in status.providerStatus.releaseRefs")
                cr_available = latest_supported_cr

            LOG.info("Cluster release version {0} will be used for upgrade"
                     .format(cr_available))
            target_release_name = cr_available

        if cr_current == target_release_name and not settings.KAAS_CHILD_CLUSTER_UPDATE_INFO_PATH:
            if settings.KAAS_CHILD_CLUSTER_RELEASE_MUST_NOT_BE_SAME:
                raise Exception(f"Clusterrelease '{child_cluster.namespace}/{child_cluster.name}'"
                                f"already equal with to be updated:"
                                f"{cr_current} == {target_release_name}"
                                f"That's unexpected situation")
            msg = (f"Skipping update: requested clusterrelease version {target_release_name} "
                   f"is the same as the current clusterrelease version {cr_current}. "
                   f"To continue with ClusterUpdatePlan steps, please specify KAAS_CHILD_CLUSTER_UPDATE_INFO_PATH")
            LOG.banner(msg)
            return

        supported_crs = kaas_manager.get_supported_clusterreleases(provider=provider_name)
        LOG.debug(f"Supported releases:\n{supported_crs}")
        assert len([data for _, data in supported_crs.items()
                    if data['name'] == cr_current]) == 1, (
            f"More than one clusterrelease with name {cr_current} found in "
            f"kaasrelease object spec.supportedClusterReleases:\n"
            f"{yaml.dump(supported_crs)}"
        )

        update_versions = [
            [
                ver['version'] for ver in data.get('availableUpgrades', [])
            ]
            for _, data in supported_crs.items() if data['name'] == cr_current
        ][0]

        if not update_versions:
            raise Exception(f"Clusterrelease {cr_current} don't have any "
                            f"'availableUpgrades' to reach the specified "
                            f"version {target_release_name}")

        paths = list(get_upgrade_path(supported_crs, update_versions))
        paths_str = "\n".join([str(p) for p in paths])
        LOG.info(f"All available upgrade paths for {cr_current}:\n{paths_str}")

        for path in paths:
            if target_release_name in path:
                break
        LOG.info(f"Select the following upgrade path to reach the release "
                 f"{target_release_name}:\n{path}")
        for release_name in path:
            yield release_name
            if release_name == target_release_name:
                break


def generate_update_release_names():
    kaas_manager = Manager(kubeconfig=settings.KUBECONFIG_PATH)
    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    child_cluster = ns.get_cluster(settings.TARGET_CLUSTER)
    lcm_type = child_cluster.lcm_type
    if lcm_type == 'byo':
        for release_name in _generete_update_release_names_byo(
                kaas_manager=kaas_manager, child_cluster=child_cluster):
            yield release_name
    else:
        for release_name in _generete_update_release_names_mke(
                kaas_manager=kaas_manager, child_cluster=child_cluster):
            yield release_name
