#!/usr/bin/env python3

import argparse
import os
import sys
import yaml

import exec_helpers

sys.path.append(os.getcwd())
try:
    from si_tests import settings
    from si_tests.managers.kaas_manager import Manager
    from si_tests import logger
    from si_tests.utils import utils
except ImportError:
    print("ImportError: Run the application from the si-tests directory or "
          "set the PYTHONPATH environment variable to directory which contains"
          " ./si_tests")
    sys.exit(1)


LOG = logger.logger


def get_child_kubeconfig_ssh(artifacts_dir,
                             child_kubeconfig_name):
    """Get child cluster kubeconfig via ssh."""
    child_kubeconfig_path = os.path.join(
        artifacts_dir, child_kubeconfig_name)

    kaas_manager = Manager(settings.KUBECONFIG_PATH)
    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    child_cluster = ns.get_cluster(settings.TARGET_CLUSTER)
    try:
        LOG.info("Try to save child cluster kubeconfig "
                 "via management cluster.")
        _, child_kubeconfig = child_cluster.get_kubeconfig_from_secret()
        LOG.info("Saving child cluster kubeconfig to:"
                 " {}".format(child_kubeconfig_path))
        with open(child_kubeconfig_path, 'w') as f:
            f.write(child_kubeconfig)
        return child_kubeconfig_path
    except Exception as e:
        LOG.error('Saving child kubeconfig failed with:' + str(e))

    LOG.info('Save child cluster kubeconfig via ssh from master node.')
    external_ip = child_cluster.get_controller_ip()
    assert external_ip, \
        "Master node not found for child cluster {cluster}".format(
            cluster=settings.TARGET_CLUSTER)

    keys = utils.load_keyfile(settings.KAAS_CHILD_CLUSTER_PRIVATE_KEY_FILE)
    pkey = utils.get_rsa_key(keys['private'])
    username = settings.KAAS_CHILD_CLUSTER_SSH_LOGIN

    auth = exec_helpers.SSHAuth(username=username, password='', key=pkey)
    master_remote = exec_helpers.SSHClient(
        host=external_ip,
        port=22,
        auth=auth,
        )

    home_path = '/home/{}'.format(username)
    LOG.info("Copy kubeconfig from from root user to mcc-user")
    master_remote.check_call(
        "sudo install -o {username} /root/.kube/config"
        " {homepath}/kubeconfig".format(username=username,
                                        homepath=home_path))

    # Download child kubeconfig from master node
    LOG.info("Downloading kubeconfig from {ip} to "
             "directory: {kubeconf_path}".format(
                 ip=external_ip,
                 kubeconf_path=artifacts_dir))
    master_remote.download(
        "{homepath}/kubeconfig".format(homepath=home_path),
        "{kubeconfig_path}".format(kubeconfig_path=child_kubeconfig_path))

    with open(child_kubeconfig_path) as kubeconfig_file:
        child_kubeconfig = yaml.load(kubeconfig_file, Loader=yaml.SafeLoader)

    api_url = 'https://{external_ip}:443'.format(external_ip=external_ip)
    LOG.info("Replace public endpoint to %s", api_url)
    child_kubeconfig['clusters'][0]['cluster']['server'] = api_url

    LOG.info("Removing certificate-authority-data from kubeconfig")
    if "certificate-authority-data" in child_kubeconfig[
            'clusters'][0]['cluster']:
        del child_kubeconfig['clusters'][0]['cluster'][
            'certificate-authority-data']

    # Set insecure-skip-tls-verify to true for child kubeconfig
    child_kubeconfig['clusters'][0]['cluster'][
        'insecure-skip-tls-verify'] = True

    LOG.info("Save child cluster kubeconfig to '{path}'".format(
        path=child_kubeconfig_path))
    with open(child_kubeconfig_path, 'w') as kubeconfig_file:
        yaml.dump(child_kubeconfig, kubeconfig_file)
    return child_kubeconfig_path


def load_params():
    """
    Parse CLI arguments and environment variables

    :return: ArgumentParser instance
    """
    parser = argparse.ArgumentParser(description=(
        'Get child cluster kubeconfig via ssh to'
        'local directory'
    ), formatter_class=argparse.RawTextHelpFormatter)
    parser.add_argument('--artifacts-directory',
                        help=('Local directory where should be placed child'
                              'cluster kubeconfig'),
                        default=settings.ARTIFACTS_DIR,
                        type=str)
    parser.add_argument('--child-kubeconfig-name',
                        help=('Local name of child cluster kubeconfig'),
                        default='child_kubeconfig',
                        type=str)
    return parser


def main():
    parser = load_params()
    opts = parser.parse_args()

    if not settings.KUBECONFIG_PATH or not os.path.isfile(
            settings.KUBECONFIG_PATH):
        LOG.error("Please set KUBECONFIG environment variable with "
                  "the path to the kubeconfig file for KaaS management "
                  "cluster")
        return 11

    if not settings.TARGET_CLUSTER and settings.TARGET_NAMESPACE:
        LOG.error("Please set TARGET_CLUSTER and TARGET_NAMESPACE environment "
                  "variables with the name of target child cluster and "
                  "namespace")
        return 11

    get_child_kubeconfig_ssh(artifacts_dir=opts.artifacts_directory,
                             child_kubeconfig_name=opts.child_kubeconfig_name)


if __name__ == '__main__':
    sys.exit(main())
