#    Copyright 2022 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
import base64
from datetime import datetime

import pytest
import yaml

from si_tests import logger
from si_tests import settings
from si_tests.managers import bootstrap_manager
from si_tests.utils import utils
from si_tests.utils import waiters
from si_tests.utils.certs import CertManager

LOG = logger.logger


@pytest.mark.usefixtures("introspect_distribution_not_changed")
def test_set_child_custom_cert(kaas_manager, show_step):
    """Add custom cert to the child cluster.

    Scenario:
        1. Collecting env data
        2. Init bootstrap using SI_CONFIG
        3. Get old cert from lb and kubeconfig data
        4. Generate and set new cert
        5. Wait for update data in kubeconfig - WR for PRODX-27128
        6. Check and validate cert

    """

    def cluster_kubeconfig_contains_ca(cluster, ca):
        _, kubeconfig_yaml = cluster.get_kubeconfig_from_secret()
        kubeconfig = yaml.load(kubeconfig_yaml, Loader=yaml.SafeLoader)
        cert_auth_data = kubeconfig['clusters'][0]['cluster']['certificate-authority-data']
        cert_auth_data_decoded = base64.b64decode(cert_auth_data).decode("utf-8")

        if ca in cert_auth_data_decoded:
            return True
        else:
            return False

    show_step(1)
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)

    show_step(2)
    bootstrap = bootstrap_manager.BootstrapManager.get_si_config_bootstrap_manager()

    host = cluster.get_load_balancer_address()
    ips = None
    dns_names = None
    if utils.is_valid_ip(host):
        ips = [host]
    else:
        dns_names = [host]
    show_step(3)
    cluster_data = cluster.get_lcm_cluster(name=cluster_name, namespace=namespace_name).read()
    if cluster.provider.name != 'byo':
        control_nodes = cluster_data.spec['machineTypes']['control']
        deploy_params = [x for x in control_nodes if x['name'] == 'deploy'][0]['params']

        # Note
        # ucp_kube_apiserver_port used on providers with lb on nodes
        # kube_apiserver_port used on providers with external lb

        port = deploy_params.get('ucp_kube_apiserver_port') or deploy_params.get('kube_apiserver_port')
        assert port, "Kube API port not found in LCMCluster task 'deploy' for 'control' nodes"
    else:
        port = 443

    show_step(4)
    app = "mke"
    ca_dir_suffix = datetime.now().strftime('%d-%-H-%-M')
    ca_path = f'{app}-{cluster.name}-{ca_dir_suffix}'
    cert_dir = cluster.name

    cert_pem, key_pem, ca_pem = bootstrap.generate_cert(ips=ips, dns_names=dns_names,
                                                        ca_path=ca_path, cert_dir=cert_dir)
    cert_manager = CertManager(kaas_manager=kaas_manager)
    cert_manager.apply_cert_for_app(app=app, cluster=cluster, hostname=host,
                                    cert_pem=cert_pem, key_pem=key_pem, ca_pem=ca_pem)

    show_step(5)
    # WR for PRODX-27128
    waiters.wait(
        lambda: cluster_kubeconfig_contains_ca(cluster, ca=ca_pem),
        interval=10, timeout=300,
        timeout_msg="Kubeconfig was not updated with new cert data.")

    show_step(6)
    # renew cluster obj with updated cert
    cluster.check.check_cert_conversation(host)
    cluster.check.check_cert_equal(port, cert=cert_pem)
    cluster.store_k8s_artifacts()
