import base64

import pytest
import yaml

from si_tests import logger
from si_tests import settings
from si_tests.utils import waiters
from si_tests.utils.utils import Provider

LOG = logger.logger


def get_proxy_setting(machines, expected_proxy):
    result = {}
    for machine in machines:
        logs = machine.exec_pod_cmd(
            "cat /etc/lcm/environment | grep http_proxy || true")['logs']
        result[machine.name] = expected_proxy in logs
    return result


@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.TARGET_CLUSTER}"])
def test_crud_child_proxy(kaas_manager, proxy_manager, show_step, _):
    """Create/Update/Delete child cluster proxy

    Scenario:
            1. Check that all needed data were passed into test
            2. Check if child cluster has a proxy
            3. If it has, delete proxy, verify deletion
            4. Create new proxy
            5. Update child cluster with new proxy
            6. Verify cluster
            7. Update cluster with no proxy
            8. Check /etc/lcm/environment on each machine
            9. Delete proxy (if not a BM with proxy before test)
            10. Check cluster
    """
    show_step(1)
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    child_cluster = ns.get_cluster(cluster_name)
    proxy_child_name = child_cluster.cluster_proxy_name
    machines = child_cluster.get_machines()
    region = kaas_manager.get_mgmt_cluster().region_name

    assert settings.KAAS_EXTERNAL_PROXY_ACCESS_STR, (
        "value KAAS_EXTERNAL_PROXY_ACCESS_STR is not defined")

    assert settings.KAAS_EXTERNAL_PROXY_ACCESS_STR_FOR_UPDATE, (
        "value KAAS_EXTERNAL_PROXY_ACCESS_STR_FOR_UPDATE is not defined")
    child_cluster.check.check_machines_status()
    child_cluster.check.check_cluster_readiness()
    child_cluster.check.check_k8s_nodes()
    LOG.info('Preflight check passed')
    show_step(2)
    # Check if cluster has a proxy
    pr_spec = child_cluster.spec.get('providerSpec').get('value')
    LOG.info(f"Check if cluster has proxy {pr_spec}")
    proxy_exist_flag = bool(pr_spec.get('proxy'))
    if proxy_exist_flag:
        show_step(3)
        LOG.info("Update cluster spec with No proxy")
        child_cluster.update_cluster_proxy()
        child_cluster.check.check_any_machine_exp_status(
            expected_status='Deploy')
        child_cluster.check.check_machines_status()
        child_cluster.check.check_cluster_nodes(timeout=3600)
        child_cluster.check.check_k8s_nodes()
        LOG.info(f"Cluster has been updated {child_cluster.name}")
        proxy_settings = get_proxy_setting(
            machines, settings.KAAS_EXTERNAL_PROXY_ACCESS_STR)
        LOG.info(f"Current proxy setting {proxy_settings}")
        proxy_object = kaas_manager.get_proxy_by_name(pr_spec.get('proxy'),
                                                      namespace_name)
        LOG.info(f"Current proxy object {proxy_object}")
        proxy_object[0].delete()
        waiters.wait(lambda: not kaas_manager.api.kaas_proxies.present(
            name=proxy_child_name, namespace=namespace_name
        ), timeout_msg="Proxy {proxy_child_name} was not deleted successfully")
        child_cluster.check.check_cluster_readiness()
        child_cluster.check.check_diagnostic_cluster_status()
        child_cluster.check.check_deploy_stage_success()

    # Check proxy at nodes
    LOG.info("Check proxy setting for each machine")
    for machine in machines:
        logs = machine.exec_pod_cmd(
            "cat /etc/lcm/environment | grep http_proxy || true")['logs']
        logs_data = yaml.load(logs.replace('=', ': '), Loader=yaml.SafeLoader)
        assert 'http_proxy' in logs_data and logs_data['http_proxy'] == ''

    # Create proxy and add it to cluster
    show_step(4)
    LOG.info("Create proxy object")
    if settings.KAAS_SSL_PROXY_CERTIFICATE_FILE:
        with open(settings.KAAS_SSL_PROXY_CERTIFICATE_FILE, 'r') as f:
            proxy_cert_content = base64.b64encode(f.read().encode("utf-8")).decode()
            LOG.info("Using proxy with custom certificate")
    else:
        proxy_cert_content = None
    proxy_object = ns.create_proxyobject(
        name=f"{cluster_name}-{settings.KAAS_PROXYOBJECT_NAME}",
        proxy_str=settings.KAAS_EXTERNAL_PROXY_ACCESS_STR,
        region=region,
        ca_cert=proxy_cert_content)
    proxy_child_name = proxy_object.data['metadata']['name']
    LOG.info(f"Proxy object {proxy_child_name} created")

    # Add proxy
    show_step(5)
    LOG.info("Add proxy to child")
    child_cluster.update_cluster_proxy(proxy_child_name)

    # Waiting for the transition of any node to the Deploy status
    show_step(6)
    child_cluster.check.check_any_machine_exp_status(
        expected_status='Deploy')
    child_cluster.check.check_machines_status()
    child_cluster.check.check_cluster_nodes(timeout=3600)
    child_cluster.check.check_cluster_readiness()
    child_cluster.check.check_k8s_nodes()
    child_cluster.check.check_diagnostic_cluster_status()
    child_cluster.check.check_deploy_stage_success()
    LOG.info(f"Cluster update {child_cluster.name}")

    # Check proxy at nodes
    LOG.info("Check proxy setting for each machine after test add it")
    proxy_settings = get_proxy_setting(
        machines, settings.KAAS_EXTERNAL_PROXY_ACCESS_STR)
    assert all([v for _, v in proxy_settings.items()]), (
        f"Some machines don't have proxy"
        f"'{settings.KAAS_EXTERNAL_PROXY_ACCESS_STR}' "
        f"in /etc/lcm/environment after adding proxy: {proxy_settings}")

    # Edit proxy
    show_step(7)
    LOG.info("Starting to update proxy")
    child_cluster.update_proxy(
        proxy_child_name,
        settings.KAAS_EXTERNAL_PROXY_ACCESS_STR_FOR_UPDATE,
        ca_cert=None
    )
    child_cluster.check.check_any_machine_exp_status(
        expected_status='Deploy')
    child_cluster.check.check_machines_status()
    child_cluster.check.check_cluster_nodes(timeout=3600)
    child_cluster.check.check_cluster_readiness()
    child_cluster.check.check_k8s_nodes()
    child_cluster.check.check_diagnostic_cluster_status()
    child_cluster.check.check_deploy_stage_success()
    LOG.info(f'Cluster update "{child_cluster.name}"')

    # Check proxy at nodes
    show_step(8)
    LOG.info("Check proxy setting on each machine after updating")
    proxy_settings = get_proxy_setting(
        machines, settings.KAAS_EXTERNAL_PROXY_ACCESS_STR)
    assert not any([v for _, v in proxy_settings.items()]), (
        f"Some machines still have proxy"
        f"'{settings.KAAS_EXTERNAL_PROXY_ACCESS_STR}' "
        f"in /etc/lcm/environment after proxy deletion: {proxy_settings}"
    )
    proxy_settings = get_proxy_setting(
        machines, settings.KAAS_EXTERNAL_PROXY_ACCESS_STR_FOR_UPDATE)
    assert all([v for _, v in proxy_settings.items()]), (
        f"Some machines don't have proxy"
        f"'{settings.KAAS_EXTERNAL_PROXY_ACCESS_STR_FOR_UPDATE}' "
        f"in /etc/lcm/environment after adding proxy: {proxy_settings}")

    # If bm env goes before test (for HA), keep proxy at the end
    if child_cluster.provider is not Provider.baremetal or not proxy_exist_flag:
        show_step(9)
        # Delete proxy
        LOG.info("Update cluster to not use proxy")
        child_cluster.update_cluster_proxy()
        child_cluster.check.check_any_machine_exp_status(
            expected_status='Deploy')
        show_step(10)
        child_cluster.check.check_machines_status()
        child_cluster.check.check_cluster_nodes(timeout=3600)
        child_cluster.check.check_cluster_readiness()
        child_cluster.check.check_k8s_nodes()
        child_cluster.check.check_diagnostic_cluster_status()
        LOG.info(f'Cluster has been updated"{child_cluster.name}"')

        # Check proxy at nodes
        proxy_settings = get_proxy_setting(
            machines, settings.KAAS_EXTERNAL_PROXY_ACCESS_STR_FOR_UPDATE)
        assert not any([v for _, v in proxy_settings.items()]), (
            f"Some machines still have proxy"
            f"'{settings.KAAS_EXTERNAL_PROXY_ACCESS_STR_FOR_UPDATE}' "
            f"in /etc/lcm/environment after proxy deletion: {proxy_settings}"
        )
        for machine in machines:
            logs = machine.exec_pod_cmd(
                "cat /etc/lcm/environment | grep http_proxy || true")['logs']
            logs_data = yaml.load(logs.replace('=', ': '),
                                  Loader=yaml.SafeLoader)
            assert 'http_proxy' in logs_data and logs_data['http_proxy'] == ''

        # Delete proxy
        LOG.info("Delete proxy")
        proxy_object.delete()
        waiters.wait(lambda: not kaas_manager.api.kaas_proxies.present(
            name=proxy_child_name, namespace=namespace_name
        ), timeout_msg="Proxy {proxy_child_name} was not deleted successfully")
        child_cluster.check.check_cluster_readiness()
