#!/usr/bin/env python
"""Utilty script to clean up Equinix environment."""
from argparse import Namespace
from typing import List

import argparse
import sys
import re

import packet
from packet.IPAddress import IPAddress
from packet.Vlan import Vlan
from packet.baseapi import ResponseError


def run_cli(args):
    """Define and read CLI arguments."""
    cli = argparse.ArgumentParser(
        prog="Equinix environment cleanup",
        description="Command line tool for generate summary report")

    cli.add_argument(
        '--api-key',
        help="Define project api key",
        dest="api_key",
        required=True)
    cli.add_argument(
        '--project-id',
        help="Define project api key",
        dest="project_id",
        required=True)

    commands = cli.add_subparsers(
        title="Operation commands")
    list_servers = commands.add_parser(
        "list-servers",
        help="List of available Servers",
        description="Show servers which are available in project")
    list_servers.add_argument(
        '--tags',
        help="Use tags for filter servers",
        nargs="+",
        default=None)
    output_group = list_servers.add_mutually_exclusive_group()
    output_group.add_argument(
        '--show-name',
        help="Show names of servers", default=True,
        action="store_true")
    output_group.add_argument(
        '--show-ips',
        help="Show attaced ips of servers",
        default=False,
        action="store_true")
    output_group.add_argument(
        '--show-id',
        help="Show ids of servers",
        default=False,
        action="store_true")
    list_servers.set_defaults(func=run_list_servers)

    list_ips = commands.add_parser(
        "list-ips",
        help="List of project IPs",
        description="Show ips which are requested in project")
    list_ips.add_argument(
        '--tag',
        help="Use tags for filter ips",
        nargs="+",
        dest='tags',
        default=None)
    output_group = list_ips.add_mutually_exclusive_group()
    output_group.add_argument(
        '--show-cidr',
        help="Show cidr", default=True,
        action="store_true")
    output_group.add_argument(
        '--show-id',
        help="Show ids of ip object",
        default=False,
        action="store_true")
    list_ips.set_defaults(func=run_list_ips)

    list_vlans = commands.add_parser(
        "list-vlans",
        help="List of project Vlans",
        description="Show vlans which are configured in project")
    list_vlans.add_argument(
        '--description',
        help="Use descriptions for filter vlans",
        nargs="+",
        dest='descriptions',
        default=None)
    list_vlans.add_argument(
        '--show-id',
        help="Show ids of vlan object",
        default=False,
        action="store_true")
    list_vlans.set_defaults(func=run_list_vlans)

    delete_servers = commands.add_parser(
        "delete-servers",
        help="Delete selected or all servers",
        description="Delete servers in project"
    )
    delete_servers.add_argument(
        "--ids",
        nargs="+",
        help="Define servers IDs to delete")
    delete_servers.add_argument(
        '--tag',
        help="Use tags for filter servers",
        nargs="+",
        dest='tags',
        default=None)
    delete_servers.add_argument(
        "--delete-all",
        action="store_true",
        default=False,
        help="Delete all servers")
    delete_servers.set_defaults(func=run_delete_servers)

    delete_ips = commands.add_parser(
        "delete-ips",
        help="Delete selected ips",
        description="Delete ips in project"
    )
    delete_ips.add_argument(
        "--ids",
        nargs="+",
        help="Define ips IDs to delete")
    delete_ips.add_argument(
        '--tag',
        help="Use tags for filter ips",
        nargs="+",
        dest='tags',
        default=None)
    delete_ips.add_argument(
        "--delete-all",
        action="store_true",
        default=False,
        help="Delete all elastic ips")
    delete_ips.set_defaults(func=run_delete_ips)

    delete_vlans = commands.add_parser(
        "delete-vlans",
        help="Delete selected vlans",
        description="Delete vlans in project"
    )
    delete_vlans.add_argument(
        "--ids",
        nargs="+",
        help="Define vlans IDs to delete")
    delete_vlans.add_argument(
        '--description',
        help="Use description for filter vlans",
        nargs="+",
        dest='descriptions',
        default=None)
    delete_vlans.add_argument(
        "--delete-all",
        action="store_true",
        default=False,
        help="Delete all vlans")
    delete_vlans.set_defaults(func=run_delete_vlans)

    return cli.parse_args(args=args)


def get_client(token: str) -> packet.Manager:
    """Create Equinix client instance."""
    return packet.Manager(auth_token=token)


def get_servers(
        client: packet.Manager,
        project_id: str) -> List[packet.Device]:
    """Get list of provisioned servers."""
    params = {
        'per_page': 100
    }
    devices = client.list_devices(project_id=project_id, params=params)
    return devices


def filter_servers_by_tags(
        servers: List[packet.Device],
        tags: List[str]) -> List[packet.Device]:
    """Filter input devices list by tags."""
    tags = set(tags)
    return [one for one in servers if tags <= set(one.tags)]


def delete_server(device: packet.Device) -> None:
    """Delete the one server."""
    if device.state == 'queued':
        print(f"Skipping server {device.hostname} ({device.id}): in queued state")
    else:
        print(f"Deleting server {device.hostname} ({device.id})")
        device.delete()


def get_ips(
        client: packet.Manager,
        project_id: str) -> List[IPAddress]:
    """Get list of IPs in the project."""
    ips = client.list_project_ips(project_id)
    ips = [ip for ip in ips if ip.global_ip is False]
    return ips


def filter_ips_by_tags(
        ips: List[IPAddress],
        tags: List[str]) -> List[IPAddress]:
    """Filter input list of IPs by tags."""
    tags = set(tags)
    return [one for one in ips if tags <= set(one.tags)]


def delete_ip(ip: IPAddress) -> None:
    print(f"Deleting ip {ip.address}/{ip.cidr} ({ip.id})")
    ip.delete()


def get_vlans(
        client: packet.Manager,
        project_id: str) -> List[Vlan]:
    try:
        return client.list_vlans(project_id)
    except ResponseError as ex:
        if ex.response.status_code == 404:
            print(f"No vlans in the project {project_id}")
            return []


def filter_vlans_by_ids(
        vlans: List[Vlan],
        ids: List[str]) -> List[Vlan]:
    return [one for one in vlans if one.id in ids]


def delete_vlan(vlan: Vlan) -> None:
    print(f"Deleting vlan {vlan.description} {vlan.vxlan} {vlan.id}")
    vlan.delete()


def filter_vlans_by_descriptions(
        vlans: List[Vlan],
        descriptions: List[str]) -> List[Vlan]:
    """Filter input list of Vlans by descriptions.
    Match description as regular expression since ending part of description
    'mcc_<CLUSTER_NAME>_<CLUSTER_IP>' CLUSTER_IP is unknown to script"""
    descriptions = set(descriptions)
    out = []
    for vlan in vlans:
        for descr in descriptions:
            if re.search(f"^{descr}.*", vlan.description):
                out += [vlan]
    return out


def run_delete_servers(args: Namespace) -> None:
    """Operate destroy servers."""
    api_key = args.api_key
    ids = args.ids or list()
    tags = args.tags or list()
    project_id = args.project_id
    delete_all = args.delete_all
    client = get_client(api_key)

    if delete_all:
        devices = get_servers(client, project_id)
        for d in devices:
            delete_server(d)

        return None

    for one in ids:
        device = client.get_device(one)
        delete_server(device)

    if tags:
        devices = get_servers(client, project_id)
        devices = filter_servers_by_tags(devices, tags)
        for d in devices:
            delete_server(d)


def run_delete_ips(args: Namespace) -> None:
    """Operate destroy servers."""
    api_key = args.api_key
    ids = args.ids or list()
    tags = args.tags or list()
    project_id = args.project_id
    delete_all = args.delete_all
    client = get_client(api_key)

    if delete_all:
        ips = get_ips(client, project_id)
        for one in ips:
            delete_ip(one)

        return

    for one in ids:
        ip = client.get_ip(one)
        delete_ip(ip)

    if tags:
        ips = get_ips(client, project_id)
        ips = filter_ips_by_tags(ips, tags)
        for one in ips:
            delete_ip(one)


def run_delete_vlans(args: Namespace) -> None:
    """delete vlans"""
    api_key = args.api_key
    ids = args.ids or list()
    descriptions = args.descriptions or list()
    project_id = args.project_id
    delete_all = args.delete_all
    client = get_client(api_key)
    vlans = get_vlans(client, project_id)

    if delete_all:
        for one in vlans:
            delete_vlan(one)

        return

    if ids:
        to_delete = filter_vlans_by_ids(vlans, ids)
        for vlan in to_delete:
            delete_vlan(vlan)

    if descriptions:
        vlans = get_vlans(client, project_id)
        to_delete = filter_vlans_by_descriptions(vlans, descriptions)
        for one in to_delete:
            delete_vlan(one)


def run_list_servers(args: Namespace) -> None:
    """Operate list of servers."""
    api_key = args.api_key
    project_id = args.project_id
    show_id = args.show_id
    show_ips = args.show_ips
    tags = args.tags

    client = get_client(api_key)
    devices = get_servers(client, project_id)
    if tags:
        devices = filter_servers_by_tags(devices, tags)

    for d in devices:
        if show_id:
            print(d.id)
        elif show_ips:
            print(d, [f"{addr['address']}/{addr['cidr']}({addr['id']})"
                      for addr in d.ip_addresses])
        else:
            print(d)


def run_list_ips(args: Namespace) -> None:
    """Operate list of servers."""
    api_key = args.api_key
    project_id = args.project_id
    show_id = args.show_id
    tags = args.tags

    client = get_client(api_key)
    ips = get_ips(client, project_id)
    if tags:
        ips = filter_ips_by_tags(ips, tags)

    for ip in ips:
        if show_id:
            print(ip.id)
        else:
            print(f"{ip.address}/{ip.cidr}", ip.tags)


def run_list_vlans(args: Namespace) -> None:
    """Operate list of vlans."""
    api_key = args.api_key
    project_id = args.project_id
    show_id = args.show_id
    descriptions = args.descriptions

    client = get_client(api_key)
    vlans = get_vlans(client, project_id)
    if descriptions:
        vlans = filter_vlans_by_descriptions(vlans, descriptions)

    for vlan in vlans:
        if show_id:
            print(vlan.id)
        else:
            print(f"{vlan.vxlan} {vlan.description}")


def main(cli_args: List[str]):
    """Run script."""
    cli = run_cli(cli_args)
    return cli.func(cli)


if __name__ == '__main__':
    sys.exit(
        main(sys.argv[1:])
    )
