import shlex
import subprocess
import tarfile
import os
import yaml
import tempfile
import xml.etree.ElementTree as ET

from kubernetes.client.rest import ApiException

from si_tests.deployments.utils import commons, file_utils, kubectl_utils
from si_tests.deployments.utils.namespace import NAMESPACE
from si_tests.utils import utils, waiters
from si_tests import settings

OS_NAMESPACE = NAMESPACE.openstack


def is_junit_xml(file_path):
    try:
        tree = ET.parse(file_path)
        root = tree.getroot()

        # Check if root element is <testsuite> or <testsuites>
        if root.tag not in ('testsuite', 'testsuites'):
            return False

        if root.tag == 'testsuites':
            for child in root.iter():
                if (child.tag == "testsuite"
                        and ('tests' in child.attrib and ('failures' in child.attrib or 'errors' in child.attrib))):
                    return True
        elif (root.tag == "testsuite"
              and ('tests' in root.attrib and ('failures' in root.attrib or 'errors' in root.attrib))):
            return True

        return False

    except ET.ParseError:
        return False


def turn_off_service(os_manager, service):
    osdpl = os_manager.get_osdpl_deployment()
    osdpl_data = osdpl.read()
    osdpl_services = osdpl_data.spec["features"].get("services", [])
    services = list(filter(lambda x: not x == service, osdpl_services))

    def _wait_fingerprint_updated(original):
        osdpl = os_manager.get_osdpl_deployment()
        osdpl_data = osdpl.read()
        current = osdpl_data.status["fingerprint"]
        if original != current:
            return True
        commons.LOG.info("Osdpl fingerprint still is not updated %s", current)
        return False

    if service in osdpl_services:
        original = osdpl_data.status["fingerprint"]
        commons.LOG.info("Turn off old %s", service)
        osdpl.patch({"spec": {"features": {"services": services}}})
        waiters.wait(lambda: _wait_fingerprint_updated(original), interval=30, timeout=900)

    if osdpl_data.spec.get("services", {}).get(service):
        # reread fresh osdpl
        osdpl = os_manager.get_osdpl_deployment()
        osdpl_data = osdpl.read()
        original = osdpl_data.status["fingerprint"]
        commons.LOG.info(f"Delete old {service} config")
        osdpl.patch({"spec": {"services": {service: None}}})
        waiters.wait(lambda: _wait_fingerprint_updated(original), interval=30, timeout=900)

    # Wait for kubernetes to delete old service
    def _wait_service_absence():
        helmbunle_name = 'openstack-{}'.format(service)
        hc_list = os_manager.os_helm_manager.list()
        hc_exist = list(filter(
            lambda x: x["name"] == helmbunle_name,
            hc_list))
        if not hc_exist:
            return True
        commons.LOG.info("Still exist %s", service)
        return False

    waiters.wait(_wait_service_absence, interval=10, timeout=600)
    return services


def _is_test_service_node_exists(os_manager, service):
    nodes = os_manager.api.nodes.list_raw().items
    for node in nodes:
        labels = node.metadata.labels
        if labels.get(service, 'disabled') == 'enabled':
            return True
    return False


def run_tests(os_manager, report_name, run_pod, service, get_run_command, extra_conf_options=None):
    run_command = get_run_command(report_name)

    services = turn_off_service(os_manager, service)

    osdpl = os_manager.get_osdpl_deployment()

    svcs_body = {}
    custom_flavor = settings.TEMPEST_CUSTOM_FLAVOR
    custom_image = settings.TEMPEST_CUSTOM_IMAGE
    custom_image_alt = settings.TEMPEST_CUSTOM_IMAGE_ALT
    custom_public_net = settings.TEMPEST_CUSTOM_PUBLIC_NET
    custom_tempest_parameters = settings.TEMPEST_CUSTOM_PARAMETERS
    custom_tempest_config = yaml.safe_load(custom_tempest_parameters)
    override_stepler_docker_image = settings.STEPLER_DOCKER_IMAGE

    if override_stepler_docker_image:
        commons.LOG.info("Override default stepler docker image")
        svcs_body = utils.merge_dicts(svcs_body, {"spec": {"services": {
            service: {service: {"values": {"images": {"tags": {
                "stepler_run_tests": override_stepler_docker_image}}}}}}}})

    if _is_test_service_node_exists(os_manager, service):
        svcs_body = utils.merge_dicts(svcs_body, {"spec": {"services": {
            service: {service: {"values": {"labels": {"job": {
                        "node_selector_key": service,
                        "node_selector_value": "enabled"
                        }}}}}}}})

    if custom_flavor:
        svcs_body = utils.merge_dicts(svcs_body, {"spec": {"services": {
            service: {service: {"values": {"conf": {"convert_to_uuid": {
                "compute": {"flavor_ref": custom_flavor,
                            'flavor_ref_alt': custom_flavor}}}}}}}}})
    if custom_image:
        svcs_body = utils.merge_dicts(svcs_body, {"spec": {"services": {
            service: {service: {"values": {"conf": {"convert_to_uuid": {
                "compute": {"image_ref": custom_image}}}}}}}}})
    if custom_image_alt:
        svcs_body = utils.merge_dicts(svcs_body, {"spec": {"services": {
            service: {service: {"values": {"conf": {"convert_to_uuid": {
                "compute": {"image_ref_alt": custom_image_alt}}}}}}}}})

    if custom_public_net:
        svcs_body = utils.merge_dicts(svcs_body, {"spec": {"services": {
            service: {service: {"values": {"conf": {"convert_to_uuid": {
                "network": {"public_network_id": custom_public_net}}}}}}}}})

    if custom_tempest_parameters:
        svcs_body = utils.merge_dicts(svcs_body, {"spec": {"services": {
            service: {service: {"values": {"conf": {
                "tempest": custom_tempest_config}}}}}}})

    if run_command:
        commons.LOG.info(f"Explicitly set {service} command")
        conf = {"script": run_command}
        if extra_conf_options is not None:
            conf.update(yaml.safe_load(extra_conf_options))
        svcs_body = utils.merge_dicts(svcs_body, {"spec": {"services": {service: {service: {"values": {
                "conf": conf}}}}}})
        osdpl.patch(svcs_body)
    else:
        msg = "TestScheme was not defined or incorrect. " \
              "Tests will be run with default script from Chart."
        commons.LOG.warning(msg)
        if svcs_body:
            osdpl.patch(svcs_body)

    commons.LOG.info("Turn on %s", service)
    services.append(service)
    osdpl.patch({"spec": {"features": {"services": services}}})

    def _wait_job():
        job = os_manager.api.jobs.get(
            run_pod, namespace=OS_NAMESPACE)
        data = job.read()
        commons.LOG.info(data.status)
        if data.status.failed:
            raise ValueError(
                "Job {} failed to complete".format(job.name))
        assert data.status.succeeded == 1

    try:
        waiters.wait_pass(
            _wait_job, interval=60, timeout=8 * 3600,
            expected=(AssertionError, ApiException))
    finally:
        jobs = os_manager.api.pods.list(
            name_prefix=run_pod, namespace=OS_NAMESPACE)
        if not jobs:
            commons.LOG.warning("No pods %s found", run_pod)
            return

        job = jobs[0]
        logs = job._manager.api.read_namespaced_pod_log(
            name=job.name, namespace=OS_NAMESPACE)
        commons.LOG.info(logs)


def _get_archived_artifact(results_pod, path, source):
    commons.LOG.info("Archiving big file: {}{}".format(path, source))
    kubectl = kubectl_utils.Kubectl()
    tmp_path = "/tmp/"
    tgz_filename = "{}.tgz".format(source)
    target_path = os.path.join(tmp_path, tgz_filename)
    cmd = "tar -zcvf {} --directory {} {}".format(target_path, path, source)
    try:
        kubectl.exec(results_pod, cmd, namespace=OS_NAMESPACE)
        return tmp_path, tgz_filename
    except Exception:
        commons.LOG.exception(
            "Failed to create archive '{}', skipping".format(
                target_path
            )
        )
        return None, None


def _get_file_size(results_pod, source):
    kubectl = kubectl_utils.Kubectl()
    try:
        return int(
            kubectl.exec(
                results_pod,
                "stat --format=%s {}".format(source),
                namespace=OS_NAMESPACE
            ).result_str
        )
    except Exception:
        commons.LOG.exception("Could not determine file size, will copy as is")
        return -1


def save_report(os_manager, report_name, results_pod, volume_log_dir,
                log_name=None, kubeconfig=None, service='tempest'):

    commons.LOG.info("Collect artifacts from pvc")
    kubectl = kubectl_utils.Kubectl(kubeconfig=kubeconfig)

    commons.LOG.info("Create pod to collect artifacts from PVC")
    basepath = os.path.dirname(os.path.abspath(__file__))
    pod_yaml = file_utils.join(
        basepath, "../yamls/{}.yaml".format(results_pod))

    with open(pod_yaml, 'r') as f:
        pod_data = yaml.safe_load(f)

    if _is_test_service_node_exists(os_manager, service):
        pod_data["spec"]["nodeSelector"] = {service: "enabled"}

    if settings.KAAS_OFFLINE_DEPLOYMENT:
        for container in pod_data['spec']['containers']:
            container["image"] = container["image"].replace('mirantis.azurecr.io', '127.0.0.1:44301')
    with tempfile.NamedTemporaryFile(mode="w") as f:
        yaml.dump(pod_data, f)
        kubectl.apply(f.name)

    def _wait_pod():
        result_pod = os_manager.api.pods.get(
            results_pod, namespace=OS_NAMESPACE)
        assert result_pod.read().status.phase == "Running"
        return result_pod

    waiters.wait_pass(_wait_pod, interval=5, timeout=420)

    artifacts = [report_name]
    if log_name:
        artifacts.append(log_name)

    failed_artifacts = []
    for artifact in artifacts:
        commons.LOG.info("Inspecting %s on %s", artifact, results_pod)
        artifact_size = _get_file_size(results_pod, os.path.join(volume_log_dir, artifact))
        # if artifact is >1M, archive it
        if artifact_size < 0 or artifact_size < 1024*1024:
            is_archived = False
            used_artifact = artifact
            used_path = volume_log_dir
        else:
            is_archived = True
            used_path, used_artifact = _get_archived_artifact(results_pod, volume_log_dir, artifact)
            # Fallback to copy in case of error
            if not used_path or not used_artifact:
                is_archived = False
                used_artifact = artifact
                used_path = volume_log_dir

        commons.LOG.info("Copy %s to local machine", used_artifact)
        source = "{}/{}:{}{}".format(
            OS_NAMESPACE, results_pod, used_path, used_artifact)
        destination = file_utils.get_artifact(
            used_artifact if is_archived else artifact
        )
        try:
            kubectl.cp(source, destination)
            if is_archived:
                commons.LOG.info("Extracting copied file")
                subprocess.run(
                    shlex.split(
                        "tar -xvf {} --directory {}".format(
                            destination,
                            os.path.dirname(file_utils.get_artifact(artifact))
                        )
                    )
                )
        except Exception:
            commons.LOG.exception(f"Failed to get artifact {artifact}")
            failed_artifacts.append(artifact)

    commons.LOG.info("Delete results pod")
    kubectl.delete("", "-f {}".format(pod_yaml), OS_NAMESPACE, cwd=".")
    assert report_name not in failed_artifacts, \
        f"Report {report_name} was not downloaded, all failed artifacts: {failed_artifacts}"


def convert_results_to_xml(report_name):
    report_path = file_utils.get_artifact(report_name)
    if not is_junit_xml(report_path):
        with open("{}.xml".format(report_path), "w") as f:
            p = subprocess.Popen(
                shlex.split("subunit2junitxml {}".format(report_path)),
                stdout=f)
            p.wait(timeout=120)
        commons.LOG.info("Convert report from subunit to junit xml")
        subprocess.run(shlex.split("rm {}".format(report_path)))
    else:
        commons.LOG.info("Report file is junit xml format")
        subprocess.run(shlex.split(f'mv {report_path} {report_path}.xml'))


def archive_log(log_name):
    log_path = file_utils.get_artifact(log_name)
    if os.path.exists(file_utils.get_artifact(log_name)):
        commons.LOG.info("Archive %s", log_name)
        archive = file_utils.get_artifact("{}.tar.gz".format(log_name))
        with tarfile.open(archive, "w:gz") as tar:
            tar.add(log_path)
    else:
        commons.LOG.info("%s not present in artifacts", log_name)
