import os
import ipaddress
import yaml

from si_tests import settings
from . import file_utils
from si_tests import logger
from si_tests.utils import utils, locks
from si_tests.managers.kaas_manager import Manager

LOG = logger.logger


class ExtraContext:
    def __init__(self):
        context_file = settings.OPENSTACK_DEPLOY_EXTRA_CONTEXT_FILE

        if os.path.exists(context_file):
            self._data = file_utils.render_yaml_content(context_file) or {}
            LOG.info(f"Extra context updated from {context_file}")
        else:
            self._data = {}
            LOG.info("Extra context is empty")

        self.kaas_manager = None
        self.kaas_child_manager = Manager()
        if settings.KUBECONFIG_MGMT_PATH:
            self.kaas_manager = Manager(kubeconfig=settings.KUBECONFIG_MGMT_PATH)

    def _get_child_cluster(self):
        k8s_node = self.kaas_child_manager.api.nodes.list()[0].name
        child_cluster = None
        for cluster in self.kaas_manager.get_clusters():
            for machine in cluster.get_machines():
                if machine.get_k8s_node_name() == k8s_node:
                    child_cluster = cluster
                    break
        return child_cluster

    @property
    def dynamic_network_opts(self):
        if self.kaas_manager is None:
            return {}
        cluster = self._get_child_cluster()
        if cluster.provider is utils.Provider.equinixmetalv2:
            nm = locks.NetworkLockManager(
                f"{cluster.name}-lock",
                cluster.namespace,
                self.kaas_manager,
            )
            facility = cluster.spec["providerSpec"]["value"].get("facility")
            metro = utils.get_equinix_metro_by_facility(facility)
            nets = {}
            for net_type in ["pxe", "floating", "tunnel", "migration"]:
                nets[net_type] = nm.pick_net_by_tag(f"{cluster.name}-{net_type}", metro)

            lmi_vid = nets["migration"] or nets["pxe"]["vlan_id"]
            live_migration_interface = f"bond0.{lmi_vid}"

            tunnel_vid = nets["tunnel"] or nets["pxe"]["vlan_id"]
            tunnel_interface = f"bond0.{tunnel_vid}"

            floating_net = nets["floating"] or nets["pxe"]
            floating_vid = floating_net["vlan_id"]
            floating_interface = f"bond0.{floating_vid}"
            floating_network = ipaddress.IPv4Network(f"{floating_net['subnet']}/{floating_net['mask']}")
            floating_router = floating_net["router_addr"]
            floating_pool_start = floating_network[2]
            floating_pool_end = floating_network[254]
            floating_range = str(floating_network)

            network_template = f"""
            spec:
              features:
                nova:
                  live_migration_interface: {live_migration_interface}
                neutron:
                  tunnel_interface: {tunnel_interface}
                  floating_network:
                    subnet:
                      gateway: {floating_router}
                      pool_end: {floating_pool_end}
                      pool_start: {floating_pool_start}
                      range: {floating_range}
                  external_networks:
                    - physnet: physnet1
                      interface: {floating_interface}
                      bridge: br-ex
                      network_types:
                       - flat
                      vlan_ranges: null
                      mtu: null
            """
            return yaml.safe_load(network_template)
        else:
            return {}

    @property
    def data(self):
        return self._data

    @property
    def third_party(self):
        return self.data.get("3rd-party", {})

    @property
    def metallb(self):
        return self.third_party.get("metallb", {})

    @property
    def ceph(self):
        return self.third_party.get("ceph", {})

    @property
    def openstack_cluster(self):
        return self.data.get("openstack", {})

    @property
    def openstack_controller(self):
        return self.data.get("operator", {})

    @property
    def tf_helm(self):
        return self.data.get("tf-helm", {})

    @property
    def tf_operator(self):
        return self.data.get("tf-operator", {})

    @property
    def openstack_opts(self):
        return self.data.get("openstack_opts", {})
