import pytest

from si_tests import logger
from si_tests import settings
from si_tests.utils import waiters

LOG = logger.logger


@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.TARGET_CLUSTER}"])
def test_worker_machinepools_actions(kaas_manager, show_step, _):
    """Create/Update/Delete machinepool with worker node type

    Scenario:
            1. Create Machinepool with 1 replicas
            2. Wait for machine spawn & became ready
            3. Set replicas to 2
            4. Wait for machine spawned & became ready
            5. Remove machine (replicas remains unchanged)
            6. Wait for machine spawned & became ready
            7. Decrease replicas to 1
            8. Remove machine belongs to machinepool
            9. Remove machinepool (should be removed with it's machines)

    """
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    child_cluster = ns.get_cluster(cluster_name)
    region = child_cluster.region_name
    initial_cluster_machines_count = len(child_cluster.get_machines())

    worker_node = child_cluster.get_machines(machine_type='worker')[0]
    data = child_cluster.get_existing_machine_spec(worker_node)

    child_cluster.check.check_machines_status(timeout=1800)
    child_cluster.check.check_cluster_nodes()
    child_cluster.check.check_k8s_nodes()
    show_step(1)
    machinepool = child_cluster.create_machinepool(1, 'node', region, **data)
    show_step(2)
    waiters.wait(lambda: len(child_cluster.get_machines()) == initial_cluster_machines_count + 1,
                 interval=10, timeout=60, timeout_msg="Machinepool-created machine was not created")
    LOG.info('Machines count changed.')
    machinepool_controlled_machines = child_cluster.get_machinepool_controlled_machines(machinepool.name)
    assert len(machinepool_controlled_machines) == machinepool.replicas(), \
        "Machines count controlled by machinepool {0} is {1} while machine replicas set to {2}".format(
            machinepool.name, machinepool_controlled_machines, machinepool.replicas())
    # We are expecting that we have only one mp-controller machine at this time because we created machinepool with
    # only one replica
    first_mp_machine = machinepool_controlled_machines[0]
    child_cluster.check.wait_machine_status_by_name(machine_name=first_mp_machine.name, expected_status='Ready')
    # Checking that status fields works properly
    assert len(machinepool_controlled_machines) == machinepool.ready_replicas(), \
        "Machine {0} has status ready, but ready replicas count at machinepool is {1} while expected {2}".format(
            first_mp_machine.name, machinepool.ready_replicas(), machinepool_controlled_machines)

    child_cluster.check.check_machines_status(timeout=1800)
    child_cluster.check.check_cluster_nodes()
    child_cluster.check.check_k8s_nodes()
    show_step(3)
    machinepool.set_replicas(2)
    show_step(4)
    waiters.wait(lambda: len(child_cluster.get_machines()) == initial_cluster_machines_count + 2,
                 interval=10, timeout=60, timeout_msg="Machinepool-controlled machine was not created. Expected "
                                                      "machine count is {0}, but actual is {1}".format(
            initial_cluster_machines_count + 2, child_cluster.get_machines()
        ))
    machinepool_controlled_machines = child_cluster.get_machinepool_controlled_machines(machinepool.name)
    # Assuming that only 2 machines in mp
    second_mp_machine = [machine for machine in machinepool_controlled_machines
                         if machine.name != first_mp_machine.name][0]
    child_cluster.check.wait_machine_status_by_name(machine_name=second_mp_machine.name, expected_status='Ready')
    assert len(machinepool_controlled_machines) == machinepool.ready_replicas(), \
        "Machine {0} has status ready, but ready replicas count at machinepool is {1} while expected {2}".format(
            second_mp_machine.name, machinepool.ready_replicas(), machinepool_controlled_machines)

    child_cluster.check.check_machines_status(timeout=1800)
    child_cluster.check.check_cluster_nodes()
    child_cluster.check.check_k8s_nodes()
    show_step(5)
    second_mp_machine.delete()
    child_cluster.wait_machine_deletion(second_mp_machine.name)
    assert machinepool.replicas() - 1 == machinepool.ready_replicas(), \
        "Machinepool's ready replicas does not changed after machine was removed. Expected ready replicas is {0}, " \
        "actual is {1}".format(machinepool.replicas() - 1, machinepool.ready_replicas())
    show_step(6)
    # Check that machine is recreated
    waiters.wait(lambda: len(child_cluster.get_machines()) == initial_cluster_machines_count + 2,
                 interval=10, timeout=60, timeout_msg="Machinepool-created machine was not created")
    machinepool_controlled_machines = child_cluster.get_machinepool_controlled_machines(machinepool.name)
    third_mp_machine = [machine for machine in machinepool_controlled_machines
                        if machine.name != first_mp_machine.name][0]
    child_cluster.check.wait_machine_status_by_name(machine_name=third_mp_machine.name, expected_status='Ready')

    child_cluster.check.check_machines_status(timeout=1800)
    child_cluster.check.check_cluster_nodes()
    child_cluster.check.check_k8s_nodes()
    show_step(7)
    machinepool.set_replicas(1)
    # Due deletePolicy = never any machines should not be removed
    assert machinepool.replicas() + 1 == machinepool.ready_replicas() == machinepool.actual_replicas(), \
        "Ready and existing replicas should be equal spec replicas value. Expected: {0}. Actual ready replicas: {1}. " \
        "Actual existing replicas: {2}".format(machinepool.replicas() + 1,
                                               machinepool.ready_replicas(),
                                               machinepool.actual_replicas())
    show_step(8)
    first_mp_machine.delete()
    child_cluster.wait_machine_deletion(first_mp_machine.name)
    machinepool_controlled_machines = child_cluster.get_machinepool_controlled_machines(machinepool.name)
    assert len(machinepool_controlled_machines) == machinepool.actual_replicas(), \
        "Machinepool-controlled machine count does not equal actual replicas of machinepool. Expected machines: {0}. " \
        "Actual machinepool replicas count: {1}".format(len(machinepool_controlled_machines),
                                                        machinepool.actual_replicas())

    child_cluster.check.check_machines_status(timeout=1800)
    child_cluster.check.check_cluster_nodes()
    child_cluster.check.check_k8s_nodes()
    show_step(9)
    mp_name = machinepool.name
    machinepool.delete()
    child_cluster.check.check_deleted_machinepool(mp_name)
    machinepool_controlled_machines = child_cluster.get_machinepool_controlled_machines(mp_name)
    assert len(machinepool_controlled_machines) == 0, \
        "Machines controlled by machinepool still inside cluster after machinepool removal. Machines: {0}".format(
            machinepool_controlled_machines
        )
    child_cluster.check.check_machines_status(timeout=1800)
    child_cluster.check.check_cluster_nodes()
    child_cluster.check.check_k8s_nodes()
