import croniter
import datetime
import yaml

from si_tests import logger, settings
from si_tests.utils import templates
LOG = logger.logger


def test_ceph_performance(kaas_manager):
    """
    This test creates kaascephoperationrequest for running ceph performance test (fio)
    """

    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE

    ns = kaas_manager.get_namespace(namespace_name)
    LOG.info("Namespace name - %s", namespace_name)

    cluster = ns.get_cluster(cluster_name)
    mgmt_cluster = kaas_manager.get_mgmt_cluster()
    LOG.info("Cluster name - %s", cluster_name)
    if cluster.workaround.skip_kaascephcluster_usage():
        ceph_perf_test_yaml_path = settings.CEPH_PERFTEST_CHILD_REQUEST_YAML_PATH
    else:
        ceph_perf_test_yaml_path = settings.CEPH_PERF_TEST_REQUEST_YAML_PATH
    template = templates.render_template(ceph_perf_test_yaml_path)
    ceph_perf_test_conf = yaml.load(template, Loader=yaml.SafeLoader)
    ceph_rbd_image_size = settings.CEPH_PERF_TEST_RBD_IMAGE_SIZE
    is_periodic = settings.CEPH_PERF_TEST_PERIODIC
    image = settings.CEPH_PERF_TEST_IMAGE
    entrypoint_cmd = settings.CEPH_PERF_TEST_ENTRYPOINT_COMMAND
    clean_pool_after_test = False
    if cluster.workaround.skip_kaascephcluster_usage():
        ceph_crd = cluster.get_miracephcluster()
        existed_pools = ceph_crd.data.get('spec').get('pools', [])
    else:
        ceph_crd = cluster.get_cephcluster()
        existed_pools = ceph_crd.data.get('spec').get('cephClusterSpec').get('pools', [])
    ceph_cluster_name = ceph_crd.data['metadata']['name']
    ceph_tools_pod = cluster.get_ceph_tool_pod()

    data = settings.CEPH_PERF_TEST_OPTIONS
    # Clean parameters from '-' and '--' if given via job variable
    data = {k.lstrip('-'): v for k, v in data.items()}
    pool_names = [pool.get('name') + '-' + pool.get('deviceClass') for pool in existed_pools]

    given_pool = data.get('pool', '')
    rbd_test_image_name = data.get('rbdname', 'test-rbd-image')
    engine = data.get('ioengine', 'rbd')
    logs_output_type = data.get('output-format', '')
    LOG.info("Prepare ceph env for testing")
    if engine in ['rbd', 'rados']:
        if not given_pool and not pool_names:
            default_pool_name = 'temp_pool_for_testing'
            msg = (f"The name of the pool for creating test image was not specified and no created pools exist. "
                   f"Creating pool: {default_pool_name}")
            LOG.warning(msg)
            cmd = ['/bin/sh', '-c', f"ceph osd pool create {default_pool_name}"]
            ceph_tools_pod.exec(cmd)
            data['pool'] = default_pool_name
            if not is_periodic:
                clean_pool_after_test = True
        elif not given_pool and pool_names:
            LOG.warning(f"The name of the pool for creating test image was not specified. Using existed "
                        f"pool: {pool_names[0]}")
            data['pool'] = pool_names[0]

        elif given_pool not in pool_names:
            LOG.warning(f"Pool {given_pool} is not existed and will be created. Current pools are: {pool_names}. "
                        f"Test will run using new created pool. For more specific results use one of existed pools")
            cmd = ['/bin/sh', '-c', f"ceph osd pool create {given_pool}"]
            ceph_tools_pod.exec(cmd)
            LOG.info(f"Pool {given_pool} created")
            if not is_periodic:
                clean_pool_after_test = True
        else:
            LOG.info(f"Pool {given_pool} exists")

        # Check rbd image
        pool = data['pool']
        rbd_list_cmd = ['/bin/sh', '-c', f"rbd list --pool {pool}"]
        rbd_create_cmd = ['/bin/sh', '-c', f"rbd create {pool}/{rbd_test_image_name} --size {ceph_rbd_image_size}"]
        rbdlist = ceph_tools_pod.exec(rbd_list_cmd).splitlines()
        if rbd_test_image_name not in rbdlist:
            LOG.info(f"Creating rbd image with name: {rbd_test_image_name} and size: "
                     f"{ceph_rbd_image_size} in pool {pool}")
            ceph_tools_pod.exec(rbd_create_cmd)
        else:
            LOG.info(f"Rbd image {rbd_test_image_name} already exists. Skipping creation")

    perfTestParameters = []
    for k, v in data.items():
        if type(v) is bool and v:
            perfTestParameters.append(f"--{k}")
        elif type(v) is bool and not v:
            continue
        else:
            perfTestParameters.append(f"--{k}={v}")

    periodicParams = {}
    if is_periodic:
        schedule = settings.CEPH_PERF_TEST_SCHEDULE
        if not croniter.croniter.is_valid(schedule):
            msg = f"Wrong schedule format. Current schedule is: {schedule}"
            LOG.error(msg)
            raise ValueError(msg)
        runs_to_keep = settings.CEPH_PERF_TEST_RUNS_TO_KEEP
        periodicParams = {'schedule': schedule, 'runsToKeep': runs_to_keep}
    if cluster.workaround.skip_kaascephcluster_usage():
        ceph_perf_test_conf['spec']['parameters'] = perfTestParameters
        if is_periodic:
            ceph_perf_test_conf['spec']['periodic'] = periodicParams
        if image:
            ceph_perf_test_conf['spec']['image'] = image
        if entrypoint_cmd:
            ceph_perf_test_conf['spec']['command'] = entrypoint_cmd
        LOG.info(f"Check MiraCeph cluster {ceph_cluster_name} for cluster {cluster_name} is Ready")
        cluster.check.wait_miraceph_phase()
        cluster.check.wait_miracephhealth_state()
    else:
        LOG.info("Prepare KaaSCephoperationRequest")
        ceph_perf_test_conf['spec']['kaasCephCluster']['name'] = ceph_cluster_name
        ceph_perf_test_conf['spec']['perfTest']['parameters'] = perfTestParameters
        if is_periodic:
            ceph_perf_test_conf['spec']['perfTest']['periodic'] = periodicParams
        if image:
            ceph_perf_test_conf['spec']['perfTest']['image'] = image
        if entrypoint_cmd:
            ceph_perf_test_conf['spec']['perfTest']['command'] = entrypoint_cmd

        LOG.info(f"Check Ceph cluster {ceph_cluster_name} for cluster {cluster_name} in namespace "
                 f"{namespace_name} is Ready")
        cluster.check.wait_kaascephcluster_state()
    LOG.info("Check Ceph health is OK")
    try:
        health_info = cluster.check.get_ceph_health_detail()
        assert health_info['status'] == "HEALTH_OK", f'Health is not OK. Will not proceed. ' \
                                                     f'Current ceph health status: {health_info}'
    except AssertionError:
        cluster.check.wait_ceph_health_status()
    LOG.info("Ceph cluster is Ready and health is OK.")
    if cluster.workaround.skip_kaascephcluster_usage():
        ceph_request = cluster.k8sclient.cephperftestrequests.create(namespace='ceph-lcm-mirantis',
                                                                     body=ceph_perf_test_conf)
        LOG.info("Wait for ceph operation request complete actions")
        ceph_request_name = ceph_request.data['metadata']['name']
        if is_periodic:
            cluster.check.wait_cephperftestrequest_state(name=ceph_request_name,
                                                         namespace='ceph-lcm-mirantis',
                                                         expected_request_phase='WaitingNextRun')
            cron = croniter.croniter(schedule)
            next_run = datetime.datetime.utcfromtimestamp(cron.get_next()).strftime('%Y-%m-%d %H:%M:%S')
            LOG.info(f"Test is running with schedule. Next run will be executed at {next_run} UTC. For result details "
                     f"proceed to cluster and see logs from pods with prefix {ceph_request_name}-")
            return
        else:
            cluster.check.wait_cephperftestrequest_state(name=ceph_request_name,
                                                         namespace='ceph-lcm-mirantis',
                                                         expected_request_phase='Finished')
    else:
        ceph_request = mgmt_cluster.k8sclient.kaas_cephoperationrequests.create(namespace=namespace_name,
                                                                                body=ceph_perf_test_conf)
        LOG.info("Wait for ceph operation request complete actions")
        ceph_request_name = ceph_request.data['metadata']['name']
        if is_periodic:
            mgmt_cluster.check.wait_cephoperationrequest_state(name=ceph_request_name,
                                                               namespace=namespace_name,
                                                               expected_request_phase='Waiting',
                                                               expected_operation_phase='WaitingNextRun',
                                                               operation_key='perfTestStatus')
            cron = croniter.croniter(schedule)
            next_run = datetime.datetime.utcfromtimestamp(cron.get_next()).strftime('%Y-%m-%d %H:%M:%S')
            LOG.info(f"Test is running with schedule. Next run will be executed at {next_run} UTC. For result details "
                     f"proceed to cluster and see logs from pods with prefix {ceph_request_name}-")
            return
        else:
            mgmt_cluster.check.wait_cephoperationrequest_state(name=ceph_request_name,
                                                               namespace=namespace_name,
                                                               expected_request_phase='Completed',
                                                               expected_operation_phase='Finished',
                                                               operation_key='perfTestStatus')
    # Get logs from pod
    pod = [pod for pod in cluster.k8sclient.pods.list(namespace='rook-ceph') if pod.name.startswith(ceph_request_name)]
    assert pod, f"No ceph performnce test pod found for request {ceph_request_name}"
    pod = pod[0]
    pod_logs = pod.get_logs()
    if logs_output_type in ['json', 'json+']:
        logs_dict = yaml.safe_load(pod_logs)
        with open(
            f'{settings.ARTIFACTS_DIR}/ceph_perf_data_{cluster.name}.yaml',
                mode='w') as f:
            f.write(yaml.dump(logs_dict))
        LOG.info(f"\n{yaml.dump(logs_dict)}")
    else:
        LOG.info(f"\n{pod_logs}")
    # Remove rbd image from pool
    if not is_periodic:
        remove_rbd_cmd = ['/bin/sh', '-c', f"rbd remove {pool}/{rbd_test_image_name}"]
        ceph_tools_pod.exec(remove_rbd_cmd)
    if clean_pool_after_test:
        cleanup_cmd = ['/bin/sh', '-c', f"ceph osd pool rm {pool} {pool} --yes-i-really-really-mean-it"]
        ceph_tools_pod.exec(cleanup_cmd)
