import base64
import json
import pytest
import time
import yaml

from si_tests import logger
from si_tests import settings
from si_tests.utils import waiters, templates

LOG = logger.logger

cluster_name = settings.TARGET_CLUSTER
namespace_name = settings.TARGET_NAMESPACE
rook_ns = settings.ROOK_CEPH_NS


@pytest.mark.usefixtures('log_method_time')
def test_child_ceph_rbd_client(kaas_manager, show_step):
    """Verify creating and accessing Ceph RBD with custom client.
       Test scenario for PRODX-25310.

    Scenario:
        1. Add custom rbd client to kaascephcluster;
        2. Wait for custom rbd client created and cred secret
           exposed in kaascephcluster.status;
        3. Go to child cluster and obtain custom rbd client creds;
        4. Create rbd client keyring file inside ceph-tools pod;
        5. Verify ceph rbd access with custom rbd client;
        6. Clean up all created files and resources in child cluster
           and kaascephcluster.

    """
    LOG.info('Check Ceph RBD access with custom client')
    mgmt_cluster = kaas_manager.get_mgmt_cluster()
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)

    if cluster.workaround.skip_kaascephcluster_usage():
        ceph_crd = cluster.get_miracephcluster()
        ceph_spec = ceph_crd.data.get('spec', {})
    else:
        ceph_crd = cluster.get_cephcluster()
        ceph_spec = ceph_crd.data.get('spec', {}).get('cephClusterSpec', {})
    ceph_clients = ceph_spec.get('clients', [])
    ceph_pools = ceph_spec.get('pools', [])

    rbd_client_name = 'test-rbd-client-%s' % str(time.time_ns())
    keyring_filename = f'ceph.client.{rbd_client_name}.keyring'

    ceph_tools_pod = None

    try:
        show_step(1)
        # 1. Add custom rbd client to kaascephcluster
        if len(ceph_pools) == 0:
            msg = "Ceph cluster has no pools, skipping Ceph RBD access test"
            LOG.error(msg)
            raise Exception(msg)

        ceph_pool_name = f"{ceph_pools[0].get('name')}-{ceph_pools[0].get('deviceClass')}"
        new_clients = ceph_clients + [
            {
                'name': rbd_client_name,
                'caps': {
                    'mon': 'allow r, allow command "osd blacklist"',
                    'osd': f'profile rbd pool={ceph_pool_name}'
                },
            }
        ]

        if cluster.workaround.skip_kaascephcluster_usage():
            patch_data = {'spec': {'clients': new_clients}}
            cluster.patch_ceph_data(data=patch_data, crd=ceph_crd)
        else:
            patch_data = {'spec': {'cephClusterSpec': {
                'clients': new_clients,
            }}}
            mgmt_cluster.patch_ceph_data(data=patch_data, crd=ceph_crd)

        show_step(2)
        # 2. Wait for custom rbd client created and cred secret
        #    exposed in kaascephcluster.status
        LOG.info(f'Wait for RBD client {rbd_client_name} created and its cred secret')
        waiters.wait(lambda: False if cluster.get_ceph_rbd_client_info(
            client_name=f'client.{rbd_client_name}') is None else True,
                     timeout=600, interval=15)
        rbd_client_creds = cluster.get_ceph_rbd_client_info(
            client_name=f'client.{rbd_client_name}')

        show_step(3)
        # 3. Go to child cluster and obtain custom rbd client creds
        LOG.info(f'Obtain RBD client {rbd_client_name} keyring')
        creds_secret = cluster.k8sclient.secrets.get(
            name=rbd_client_creds.get('secretName'),
            namespace=rbd_client_creds.get('secretNamespace')
        ).read()

        keyring = base64.b64decode(creds_secret.data[rbd_client_name]).decode('utf-8')
        assert keyring != ''

        show_step(4)
        # 4. Create rbd client keyring file inside ceph-tools pod
        ceph_tools_pod = cluster.get_ceph_tool_pod()
        cmd_endpoint = ['/bin/sh', '-c', f'touch /etc/ceph/{keyring_filename}']
        ceph_tools_pod.exec(cmd_endpoint)
        cmd_endpoint = ['/bin/sh', '-c', f'echo "[client.{rbd_client_name}]" >> /etc/ceph/{keyring_filename}']
        ceph_tools_pod.exec(cmd_endpoint)
        cmd_endpoint = ['/bin/sh', '-c', f'echo "    key = {keyring}" >> /etc/ceph/{keyring_filename}']
        ceph_tools_pod.exec(cmd_endpoint)
        cmd_endpoint = ['/bin/sh', '-c', f'cat /etc/ceph/{keyring_filename}']
        output = ceph_tools_pod.exec(cmd_endpoint)
        LOG.info(f'{output}')

        show_step(5)
        # 5. Verify ceph rbd access with custom rbd client
        cmd_endpoint = ['/bin/sh', '-c', f'ceph -n client.{rbd_client_name} -s']
        output = ceph_tools_pod.exec(cmd_endpoint)
        LOG.info(output)
        assert 'HEALTH_OK' in output or 'HEALTH_WARN' in output

        cmd_endpoint = ['/bin/sh', '-c', f'rbd -n client.{rbd_client_name} ls -p {ceph_pool_name}']
        output = ceph_tools_pod.exec(cmd_endpoint)
        assert 'Operation not permitted' not in output

        if len(ceph_pools) > 1:
            not_permitted_pool = ceph_pools[1].get('name')
            cmd_endpoint = ['/bin/sh', '-c', f'rbd -n client.{rbd_client_name} ls -p {not_permitted_pool}']
            output = ceph_tools_pod.exec(cmd_endpoint)
            assert 'rbd: error opening pool' in output
    finally:
        show_step(6)
        # 6. Clean up all created files and resources in child cluster
        #    and kaascephcluster
        if ceph_tools_pod is not None:
            cmd_endpoint = ['/bin/sh', '-c', f'rm /etc/ceph/{keyring_filename}']
            ceph_tools_pod.exec(cmd_endpoint)
        if cluster.workaround.skip_kaascephcluster_usage():
            patch_data = {'spec': {'clients': ceph_clients}}
            cluster.patch_ceph_data(data=patch_data, crd=ceph_crd)
        else:
            patch_data = {'spec': {'cephClusterSpec': {'clients': ceph_clients}}}
            mgmt_cluster.patch_ceph_data(data=patch_data, crd=ceph_crd)

        LOG.info(f'Wait for RBD client {rbd_client_name} deleted')
        waiters.wait(lambda: cluster.get_ceph_rbd_client_info(
            client_name=f'client.{rbd_client_name}') is None,
                     timeout=600, interval=15)
        LOG.info("Ceph RBD access clean up finished")


@pytest.mark.usefixtures('log_method_time')
def test_child_ceph_rgw_external_ip(kaas_manager, show_step):
    """Test child Ceph cluster rgw external IP is accessible.
       Test scenario for PRODX-26092.

    Scenario:
        1. Gather current ceph data.
        2. Check Ceph cluster doesn't have dns name for public endpoint.
        3. Obtain Ceph RGW external IP endpoint.
        4. Check access to Ceph RGW with external IP.
    """
    LOG.info(f"Namespace name: {namespace_name}")
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)
    k8sclient = cluster.k8sclient
    if cluster.workaround.skip_kaascephcluster_usage():
        miraceph = cluster.get_miracephcluster().data
        ceph_spec = miraceph.get('spec', {})
        health_status = cluster.get_miracephhealth().data.get('status') or {}
        ceph_status = health_status.get('fullClusterStatus', {})
    else:
        kaascephcluster = cluster.get_cephcluster().data
        ceph_spec = kaascephcluster.get('spec', {}).get('cephClusterSpec', {})
        kaascephcluster_status = kaascephcluster.get('status') or {}
        ceph_status = kaascephcluster_status.get('fullClusterInfo', {})

    LOG.info("Verify Ceph Object Store loadbalancer IP is accessible")
    show_step(1)
    # verify ceph rgw doesn't require ingress rule either skip
    # check (verified with component level tests)
    ceph_rgw = ceph_spec.get('objectStorage', {}).get('rgw', {})
    ceph_pool_roles = set([pool.get('role', '')
                           for pool in ceph_spec.get('pools', [])])
    ceph_ingress = ceph_spec.get('ingress', {})
    expected_pool_roles = {'images', 'vms', 'volumes', 'backup'}
    msg = "Skip Ceph Object Store loadbalancer IP verification"
    skip = False
    if not ceph_rgw:
        msg += " - no rgw section in spec"
        skip = True
    if ceph_ingress:
        msg += " - rgw ingress defined"
        skip = True
    if len(expected_pool_roles.intersection(ceph_pool_roles)) > 0:
        msg += " - MOSK pools found therefore rgw ingress defined"
        skip = True
    if skip:
        LOG.info(f"{msg}")
        pytest.skip(f"{msg}")

    show_step(2)
    # switch to child KUBECONFIG api and verify there is no
    # ingress
    ceph_ingress = k8sclient.ingresses.list(namespace='rook-ceph')
    assert len(ceph_ingress) == 0
    show_step(3)
    # obtain rgw endpoint as IP address and verify it is the same
    # with rgw service
    rgw_endpoint = ceph_status.get(
        'objectStorageStatus', {}).get('objectStorePublicEndpoint', '')
    rgw_ip_split = rgw_endpoint.split('/')
    rgw_ip_address = (rgw_ip_split[-1]
                      if rgw_ip_split[-1] != '' else rgw_ip_split[-2])
    rgw_external_svc_ip = k8sclient.services.get(
        name='rook-ceph-rgw-%s-external' % ceph_rgw.get('name'),
        namespace='rook-ceph').get_external_ip()
    assert rgw_external_svc_ip == rgw_ip_address

    show_step(4)
    # verify rgw is accessible with checked rgw IP address
    ceph_tools_pod = cluster.get_ceph_tool_pod()
    cmd_endpoint = ['/bin/sh', '-c', 'curl http://%s' % rgw_ip_address]
    rgw_output = ceph_tools_pod.exec(cmd_endpoint)
    assert 'ListAllMyBucketsResult' in rgw_output
    LOG.info("Ceph Object Store loadbalancer IP is verified and accessible")


@pytest.mark.usefixtures('log_method_time')
def test_child_ceph_rgw_public_endpoint(kaas_manager, show_step):
    """Test child Ceph cluster rgw public endpoint is accessible.
       Test scenario for PRODX-26092 for MOSK clusters.

    Scenario:
        1. Gather current ceph data.
        2. Check Ceph cluster has dns name for public endpoint.
        3. Obtain Ceph RGW public endpoint.
        4. Check access to Ceph RGW with public endpoint.
    """
    LOG.info(f"Namespace name: {namespace_name}")
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)
    cluster_release = cluster.clusterrelease_version
    k8sclient = cluster.k8sclient
    if cluster.workaround.skip_kaascephcluster_usage():
        ceph_crd = cluster.get_miracephcluster().data
        ceph_spec = ceph_crd.get('spec', {})
        ceph_status = ceph_crd.get('status', {}).get('fullClusterStatus', {})
    else:
        ceph_crd = cluster.get_cephcluster().data
        ceph_spec = ceph_crd.get('spec', {}).get('cephClusterSpec', {})
        ceph_status = ceph_crd.get('status', {}).get('fullClusterInfo', {})

    # Skip test for non-MOSK clusters
    if 'mos' not in cluster_release:
        pytest.skip("Skip ceph_rgw_public_endpoint for non-MOSK child cluster")

    LOG.info("Verify Ceph Object Store public endpoint is accessible")
    show_step(1)
    # verify ceph rgw and MOSK pools are defined either skip
    # check (verified with component level tests)
    ceph_rgw = ceph_spec.get('objectStorage', {}).get('rgw', {})
    ceph_pool_roles = set([pool.get('role', '')
                           for pool in ceph_spec.get('pools', [])])
    expected_pool_roles = {'images', 'vms', 'volumes', 'backup'}
    msg = "Skip Ceph Object Store loadbalancer IP verification"
    skip = False
    issues = []
    if not ceph_rgw:
        issues.append("no rgw section in spec")
        skip = True
    if len(expected_pool_roles.intersection(ceph_pool_roles)) == 0:
        issues.append("no MOSK pools found")
        skip = True
    if skip:
        LOG.info(f"{msg}: {', '.join(issues)}")
        pytest.skip(f"{msg}")

    show_step(2)
    # switch to child KUBECONFIG api and verify ceph ingress rule
    # is defined
    ceph_ingresses = k8sclient.ingresses.list(namespace='rook-ceph')
    assert len(ceph_ingresses) > 0
    ceph_ingress_rule = ceph_ingresses[0]
    ingress_rule = ceph_ingress_rule.data.get('spec', {}).get('rules', [])
    assert len(ingress_rule) > 0
    ingress_host = ingress_rule[0].get('host', '')
    assert ingress_host != ''

    show_step(3)
    # obtain rgw public endpoint
    rgw_endpoint = ceph_status.get(
        'objectStorageStatus', {}).get('objectStorePublicEndpoint', '')
    endpoint_split = rgw_endpoint.split('/')
    rgw_endpoint = (endpoint_split[-1]
                    if endpoint_split[-1] != '' else endpoint_split[-2])
    assert ingress_host == rgw_endpoint

    show_step(4)
    # verify rgw is accessible with checked rgw public endpoint

    # we're using rgw pod just because it already contains TLS CA cert
    ceph_rgw_pod = cluster.get_ceph_rgw_pod()
    cmd_endpoint = ['/bin/sh', '-c', 'curl https://%s' % rgw_endpoint]
    rgw_output = ceph_rgw_pod.exec(cmd_endpoint)
    assert 'ListAllMyBucketsResult' in rgw_output
    LOG.info("Ceph Object Store public endpoint is verified and accessible")


@pytest.mark.usefixtures('log_method_time')
def test_check_ceph_rgw_users(kaas_manager, show_step):
    """Verify creating and accessing Ceph RGW with custom user.
       Test scenario for PRODX-25310.

    Scenario:
    1. Add custom rgw user to kaascephcluster;
    2. Wait for custom rgw user created and cred secret
       exposed in kaascephcluster.status;
    3. Go to child cluster and obtain rgw custom user creds;
    4. Create a pod with s3 client and wait for its Running state;
    5. Verify ceph rgw access with custom rgw user;
    6. Clean up all created resources in child cluster and kaascephcluster.

    """
    LOG.info('Check Ceph RGW access with custom user')

    mgmt_cluster = kaas_manager.get_mgmt_cluster()
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)

    if cluster.workaround.skip_kaascephcluster_usage():
        ceph_crd = cluster.get_miracephcluster()
        ceph_spec = ceph_crd.data.get('spec', {})
    else:
        ceph_crd = cluster.get_cephcluster()
        ceph_spec = ceph_crd.data.get('spec', {}).get('cephClusterSpec', {})
    ceph_rgw = ceph_spec.get('objectStorage', {}).get('rgw', {})

    if not ceph_rgw:
        pytest.skip(f"RGW is not enabled on cluster {cluster.name}")

    ceph_rgw_name = ceph_rgw.get('name')
    rgw_username = 'test-rgw-user-%s' % str(time.time_ns())

    pod = None
    configmap_name = 'test-s3-config-%s' % str(time.time_ns())

    try:
        show_step(1)
        # 1. Add custom rgw user to kaascephcluster
        if len(ceph_spec) == 0 or len(ceph_rgw) == 0:
            LOG.info("Ceph RGW is not deployed, skipping check RGW users")
            return

        ceph_rgw['objectUsers'] = [
            {
                'name': rgw_username,
                'displayName': rgw_username,
                'capabilities': {
                    'user': 'read',
                    'bucket': '*',
                },
                'quotas': {
                    'maxBuckets': 1,
                }
            }
        ]

        if cluster.workaround.skip_kaascephcluster_usage():
            patch_data = {'spec': {
                'objectStorage': {
                    'rgw': ceph_rgw,
                },
            }}
            cluster.patch_ceph_data(data=patch_data, crd=ceph_crd)
        else:
            patch_data = {'spec': {'cephClusterSpec': {
                'objectStorage': {
                    'rgw': ceph_rgw,
                },
            }}}
            mgmt_cluster.patch_ceph_data(data=patch_data, crd=ceph_crd)

        show_step(2)
        # 2. Wait for custom rgw user created and cred secret
        #    exposed in kaascephcluster.status
        LOG.info('Wait for RGW user %s created and its cred secret' % rgw_username)
        waiters.wait(lambda: False if cluster.get_ceph_rgw_user_info(
            rgw_username=rgw_username) is None else True,
                     timeout=600, interval=15)
        rgw_user_creds = cluster.get_ceph_rgw_user_info(
            rgw_username=rgw_username)

        show_step(3)
        # 3. Go to child cluster and obtain rgw custom user creds
        LOG.info('Obtain RGW user %s s3 creds' % rgw_username)
        creds_secret = cluster.k8sclient.secrets.get(
            name=rgw_user_creds.get('secretName'),
            namespace=rgw_user_creds.get('secretNamespace')
        ).read()

        access_key = base64.b64decode(creds_secret.data['AccessKey']).decode('utf-8')
        secret_key = base64.b64decode(creds_secret.data['SecretKey']).decode('utf-8')
        assert access_key != ''
        assert secret_key != ''

        show_step(4)
        # 4. Create a pod with s3 client and wait for its Running state
        pod_name = 'test-s3-user-access-%s' % str(time.time_ns())

        LOG.info('Create s3 config %s for rgw user verification' % configmap_name)
        cm_template = templates.render_template(
            settings.CEPH_S3_CONFIGMAP_YAML,
            {'CONFIGMAP_NAME': configmap_name,
             'ACCESS_KEY': access_key,
             'SECRET_KEY': secret_key})
        cm_json_body = json.dumps(
            yaml.load(cm_template, Loader=yaml.SafeLoader))
        cluster.k8sclient.configmaps.create(name=configmap_name,
                                            namespace="rook-ceph",
                                            body=json.loads(cm_json_body))

        LOG.info('Create s3 client pod %s for rgw user verification' % pod_name)
        pod_template = templates.render_template(
            settings.CEPH_S3_CLIENT_YAML,
            {'POD_NAME': pod_name,
             'AWSCONFIG_NAME': configmap_name})
        pod_json_body = json.dumps(
            yaml.load(pod_template, Loader=yaml.SafeLoader))
        pod = cluster.k8sclient.pods.create(name=pod_name,
                                            namespace="rook-ceph",
                                            body=json.loads(pod_json_body))
        LOG.info('Wait for s3 client pod %s become Running' % pod_name)
        pod.wait_ready(timeout=600, interval=30)

        LOG.info('Verify rgw user access with s3 client')
        show_step(5)
        # 5. Verify ceph rgw access with custom rgw user
        rgw_endpoint = f'https://rook-ceph-rgw-{ceph_rgw_name}.rook-ceph.svc:8443'
        bucket_name = 'test-bucket'
        cmd_endpoint = [
            '/bin/sh',
            '-c',
            f'aws --endpoint-url {rgw_endpoint} --ca-bundle /etc/rgwcerts/cacert '
            f's3api create-bucket --bucket {bucket_name}']
        output = pod.exec(cmd_endpoint)
        LOG.info(f"create-bucket:\n{output}")
        cmd_endpoint = [
            '/bin/sh',
            '-c',
            f'aws --endpoint-url {rgw_endpoint} --ca-bundle /etc/rgwcerts/cacert '
            's3api list-buckets']
        output = pod.exec(cmd_endpoint)
        assert bucket_name in output
        LOG.info(f"list-buckets:\n{output}")
        cmd_endpoint = [
            '/bin/sh',
            '-c',
            f'aws --endpoint-url {rgw_endpoint} --ca-bundle /etc/rgwcerts/cacert '
            f's3api delete-bucket --bucket {bucket_name}']
        output = pod.exec(cmd_endpoint)
        LOG.info(f"delete-bucket:\n{output}")
    finally:
        show_step(6)
        # 6. Clean up all created resources in child cluster and kaascephcluster
        if pod is not None and pod.exists():
            pod.delete()
        try:
            cm = cluster.k8sclient.configmaps.get(
                namespace='rook-ceph', name=configmap_name).data or {}
            if cm:
                cluster.k8sclient.configmaps.delete(
                    namespace='rook-ceph', name=configmap_name)
        except Exception:
            LOG.info(f'ConfigMap {configmap_name} not found, clean up skipped')

        ceph_rgw['objectUsers'] = []
        if cluster.workaround.skip_kaascephcluster_usage():
            patch_data = {'spec': {
                'objectStorage': {
                    'rgw': ceph_rgw,
                },
            }}
            cluster.patch_ceph_data(data=patch_data, crd=ceph_crd)
        else:
            patch_data = {'spec': {'cephClusterSpec': {
                'objectStorage': {
                    'rgw': ceph_rgw,
                },
            }}}
            mgmt_cluster.patch_ceph_data(data=patch_data, crd=ceph_crd)
        LOG.info('Wait for RGW user %s deleted' % rgw_username)
        waiters.wait(lambda: cluster.get_ceph_rgw_user_info(
            rgw_username=rgw_username) is None,
                     timeout=600, interval=15)
        LOG.info("Ceph RGW access clean up finished")
