import pytest
import yaml


from si_tests import settings
from si_tests import logger
from si_tests.utils import waiters, utils
from si_tests.managers import openstack_manager

LOG = logger.logger

cluster_name = settings.TARGET_CLUSTER
namespace_name = settings.TARGET_NAMESPACE
osdpl_name = settings.OSH_DEPLOYMENT_NAME


@pytest.mark.usefixtures("introspect_distribution_not_changed")
@pytest.mark.usefixtures("introspect_child_target_objects")
@pytest.mark.usefixtures("collect_downtime_statistics")     # Should be used if ALLOW_WORKLOAD == True
@pytest.mark.usefixtures('log_method_time')
def test_add_bm_compute(kaas_manager, show_step):
    """Add bm compute node.

    Scenario:
        1. Gather nodes information from yaml file
        2. Check hypervisor count and state before scale
        3. Get info about nodeforscale node
        4. Create bmh
        5. Create BareMetal machine and add labels
        6. Run cluster check and apply hotfix labels
        7. Wait nova-compute pod on new compute
        8. Wait for nova service become ready
        9. Run nova cell mapping job
        10. Check hypervisor count and state after scale


    """

    # Get namespace
    LOG.info(f"Namespace name: {namespace_name}")
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)

    # Check or set customHostnamesEnabled flag in Cluster object to match settings.CUSTOM_HOSTNAMES
    cluster.set_custom_hostnames_enabled(flag=settings.CUSTOM_HOSTNAMES)

    # Create ssh key
    public_key_name = "{cluster_name}-key".format(
        cluster_name=cluster_name)
    LOG.info("Public key name - {public_key_name}".format(
        public_key_name=public_key_name))
    ns.get_publickey(public_key_name)

    show_step(1)
    mgmt_version = \
        kaas_manager.get_mgmt_cluster().spec['providerSpec']['value']['kaas'][
            'release']
    LOG.info(f"KaaS mgmt version is:{mgmt_version}")
    child_data = utils.render_child_data(kaas_manager.si_config, {'mgmt_version': mgmt_version})

    bm_hosts_data = child_data['nodes']
    child_cluster_data = child_data.get('child_config')
    tf = child_cluster_data.get('tungstenfabric', [])
    # Getting pods with openstack client
    client_pod = cluster.k8sclient.pods.list(
        namespace="openstack",
        name_prefix='keystone-client')
    assert len(client_pod) > 0, ("No pods found with prefix "
                                 "keystone-client in namespace "
                                 "openstack")
    client_pod = client_pod[0]
    child_kubeconfig_name, child_kubeconfig = cluster.get_kubeconfig_from_secret()
    with open('child_conf', 'w') as f:
        f.write(child_kubeconfig)
    os_manager = openstack_manager.OpenStackManager(kubeconfig='child_conf')
    cmp_service_list = ['/bin/sh', '-c', 'PYTHONWARNINGS=ignore::UserWarning '
                                         'openstack compute service list -f yaml']
    if os_manager.is_ironic_enabled() and cluster.workaround.prodx_36209():
        # Remove ironic services in down state manualy.
        # If cluster was updated from version < 23.3 to version >= 23.3 then 2 of 3 ironic compute services will be
        # in down state, because it was reconfigured to use 1 replica instead of 3.
        # https://gerrit.mcp.mirantis.com/c/mcp/openstack-controller/+/179719
        # nova-service cleanup job is skipping nova-compute services for cleanup, so we need
        # to remove them manually
        ironic_services_list = [s for s in yaml.safe_load(client_pod.exec(cmp_service_list))
                                if s.get('Host').startswith('nova-compute-ironic-')]
        assert ironic_services_list, "Ironic service is enabled, but doesn't have enabled services in openstack"
        if len(ironic_services_list) > 1:
            down_services = [s for s in ironic_services_list if s.get('State') == 'down']
            assert len(down_services) != len(ironic_services_list), ("All ironic services in down state. "
                                                                     "Can not continue test")
            for service in down_services:
                service_name = service.get('Host')
                service_id = service.get('ID')
                remove_service_cmd = ['/bin/sh', '-c',
                                      f"PYTHONWARNINGS=ignore::UserWarning "
                                      f"openstack compute service delete {service_id}"]
                LOG.info(f"Removing service {service_name} with ID {service_id}")
                client_pod.exec(remove_service_cmd)
                LOG.info(f"Service {service_name} removed")
        else:
            LOG.info("Ironic service currently has 1 replica. Skipping deletion")

    cmd_hypervisor_list = ['/bin/sh', '-c',
                           'PYTHONWARNINGS=ignore::UserWarning '
                           'openstack hypervisor list -f yaml']
    show_step(2)
    cmp_count = len(yaml.safe_load(client_pod.exec(
        cmd_hypervisor_list)))

    def check(count=None):
        hyper_list = yaml.safe_load(client_pod.exec(cmd_hypervisor_list))
        hyper_states = sum(h.get('State') == 'up' for h in hyper_list)
        LOG.info(f"Count hypervisors: {hyper_states}, expected: {count}")
        if hyper_states == count and len(hyper_list) == count:
            return True
        else:
            return False

    waiters.wait(
        lambda: check(count=cmp_count),
        timeout=300, interval=10,
        timeout_msg="Some hypervisor is not ready")
    show_step(3)
    worker_scale = []
    for node in bm_hosts_data:
        if 'nodeforscale' in node.get('si_roles', []):
            cred_name = node['name'] + '-cred'
            # Dirty hack, for not copy-paste bmh_name|cred_name across whole
            # test.
            bmh_name = utils.render_bmh_name(node['name'],
                                             cluster_name,
                                             node.get('bootUEFI', True),
                                             node['bmh_labels'],
                                             si_roles=node.get('si_roles',
                                                               False))
            _node = node.copy()
            _node.update({'bmh_name': bmh_name,
                          'cred_name': cred_name})
            worker_scale.append(_node)
            secret_data = {
                "username": node['ipmi']['username'],
                "password": node['ipmi']['password']
            }
            #
            if "kaas.mirantis.com/baremetalhost-credentials-name" in node.get('bmh_annotations', {}):
                if _node['ipmi'].get('monitoringUsername', False):
                    secret_data.update({
                        "monitoringPassword": _node['ipmi']['monitoringPassword'],
                        "monitoringUsername": _node['ipmi']['monitoringUsername']
                    })
                if not kaas_manager.api.kaas_baremetalhostscredentials.present(name=cred_name,
                                                                               namespace=namespace_name):
                    region = kaas_manager.get_mgmt_cluster().region_name
                    ns.create_baremetalhostcredential(name=cred_name, data=secret_data, region=region,
                                                      provider="baremetal")
                else:
                    LOG.warning(f'bmhc: {cred_name} already exist, skipping')
            else:
                raise Exception("IPMI credentials supported only over baremetalhostcredentials")

    assert len(worker_scale) >= 1, "BMH with si_roles " \
                                   "'nodeforscale' not found"
    show_step(4)
    new_nodes_name = []
    osdpl_node_overrides = dict()
    for node in worker_scale:
        bmh_name = node['bmh_name']
        new_nodes_name.append(bmh_name)
        node_override = node.get('osdpl_node_override', {})
        if node_override:
            osdpl_node_overrides[node['name']] = node_override
        elif tf and 'hotfix/tfvrouter-dpdk' in node.get('node_labels', {}):
            # TF vrouter controller rely on own data for dpdk nodes
            # instead of node_overrides. But we need to collect dpdk
            # nodes to perform additional steps (install drivers,
            # host aggregates and etc.)
            osdpl_node_overrides[node['name']] = None
        ns.create_baremetalhost(bmh_name=bmh_name,
                                bmh_secret=node['cred_name'],
                                bmh_mac=node['networks'][0]['mac'],
                                bmh_ipmi=node['ipmi'],
                                hardwareProfile=node.get('hardwareProfile',
                                                         False),
                                labels=node['bmh_labels'],
                                annotations=node.get('bmh_annotations', {}),
                                bootUEFI=node.get('bootUEFI', True),
                                bmhi_credentials_name=node['cred_name'])
    LOG.info(f"New BMH nodes:\n{new_nodes_name}")

    ns.wait_baremetalhosts_statuses(nodes=new_nodes_name,
                                    wait_status='ready',
                                    retries=40,
                                    interval=60)
    show_step(5)
    new_machine_names = []
    release_name = cluster.clusterrelease_version
    for node in worker_scale:
        distribution = utils.get_distribution_for_node(kaas_manager, node, release_name)
        labels = node.get('node_labels', {})
        add_labels = {'openstack-compute-node': 'enabled'}
        if tf:
            if 'hotfix/tfvrouter-dpdk' not in labels:
                add_labels['tfvrouter'] = 'enabled'
        else:
            add_labels['openvswitch'] = 'enabled'
        labels.update(add_labels)
        LOG.info(f"Add labels for machine {node['bmh_name']}:\n{labels}")
        custom_bmhp = False
        if node.get('bmh_profile'):
            custom_bmhp = {
                'namespace': namespace_name,
                'name': node['bmh_profile']
            }
        machine = cluster.create_baremetal_machine(
            genname=node['bmh_name'],
            node_pubkey_name=public_key_name,
            matchlabels={'kaas.mirantis.com/'
                         'baremetalhost-id':
                             node['bmh_labels']
                             ['kaas.mirantis.com/baremetalhost-id']},
            baremetalhostprofile=custom_bmhp,
            l2TemplateSelector=node.get('l2TemplateSelector', dict()),
            labels=node['machine_labels'],
            node_labels=labels,
            distribution=distribution,
        )
        new_machine_names.append(machine.name)

    # Wait Baremetal hosts be provisioned
    ns.wait_baremetalhosts_statuses(nodes=new_nodes_name,
                                    wait_status='provisioned',
                                    retries=50,
                                    interval=90)

    show_step(6)
    # Waiting for cluster,pods,hb are Ready after machines were added
    cluster.check.check_machines_status()
    cluster.check.check_cluster_nodes()
    cluster.check.check_k8s_nodes()

    # This WA to set labels for OpenStack nodes. Once
    # KaaS can set labels this WA should be removed.
    cluster.apply_hotfix_labels(bm_nodes=worker_scale)
    cluster.apply_nodes_annotations(bm_nodes=worker_scale)

    cluster.check.check_cluster_readiness()
    cluster.check.check_helmbundles()

    new_machines = [cluster.get_machine(name=machine_name) for machine_name in new_machine_names]
    # Check that Machine hostname is created with respect to Cluster flag 'customHostnamesEnabled'
    cluster.check.check_custom_hostnames_on_machines(machines=new_machines)

    cluster.check.check_actual_expected_ceph_pods(
        scale_machines_names_list=new_machine_names)
    # Refresh expected objects
    cluster._refresh_expected_objects()
    show_step(7)
    machines = cluster.get_machines()
    nodeforscale = [machine for machine in machines if 'nodeforscale' in machine.name][0]
    node_name = nodeforscale.get_k8s_node_name()
    field_selector = f"spec.nodeName={node_name}"
    LOG.info(f"Get nova-compute pod from node : {field_selector}")

    def check_cmp_service_status():
        service_list = yaml.safe_load(client_pod.exec(cmp_service_list))
        service_states_up = [service for service in service_list if service.get('State') == 'up']
        LOG.info(f"Current service status {service_list}")
        if len(service_list) == len(service_states_up):
            return True
        else:
            return False

    if settings.OPENSTACK_DEPLOY_DPDK:
        aggregates_cmd = ['/bin/sh', '-c',
                          'PYTHONWARNINGS=ignore::UserWarning openstack aggregate list -f yaml']
        zone = settings.OPENSTACK_DPDK_AVAILABILITY_ZONE_NAME
        # Automation of these steps
        # https://docs.mirantis.com/mosk/latest/deploy/deploy-openstack/advanced-config/enable-dpdk.html # noqa: E501

        for node_name, node_data in osdpl_node_overrides.items():
            machine = [
                machine for machine in cluster.get_machines()
                if node_name in machine.name][0]
            LOG.info("Get DPDK driver package")
            distrib_codename = machine.get_distribution().split('/')[-1]
            openstack_dpdk_driver_package = utils.get_dpdk_driver_package(distrib_codename)
            LOG.info(f"Install DPDK driver on new compute: {node_name}")
            machine.run_cmd(
                "sudo apt install -y "
                f"{openstack_dpdk_driver_package}",
                timeout=600)
            k8s_node = machine.get_k8s_node()
            osdpl = os_manager.get_openstackdeployment(osdpl_name)
            if node_data:
                osdpl_update = {'spec': {'nodes': {
                    'kubernetes.io/hostname::{}'.format(
                        k8s_node.name): node_data}}}
                osdpl.patch(osdpl_update)
            os_manager.wait_all_osdpl_children_status()
            os_manager.wait_openstackdeployment_health_status()
            os_manager.wait_all_os_pods(timeout=900, interval=90)
            # Adding new compute to aggregate advanced-compute if it exists
            aggregates = yaml.safe_load(client_pod.exec(
                aggregates_cmd))
            if aggregates:
                aggregate = [
                    a for a in aggregates if zone in a[
                        'Availability Zone']]
                if aggregate:
                    # If several aggregates exist with the same AZ
                    # it is not matter where new host will be added
                    # So we will take first aggregate
                    aggregate = aggregate[0]
                    host_name = k8s_node.name
                    aggr_name = aggregate.get('Name')
                    aggr_id = aggregate.get('ID')
                    add_host_cmd = [
                        '/bin/sh', '-c', 'PYTHONWARNINGS=ignore::UserWarning'
                                         ' openstack aggregate add host '
                                         '{} {}'.format(aggr_id, host_name)]
                    LOG.info("Host {} will be added to "
                             "aggregate {}".format(host_name, aggr_name))
                    res = client_pod.exec(add_host_cmd)
                    LOG.info("Adding host result: \n{}".format(res))
                else:
                    LOG.warning(
                        "There are no aggregates with {} AZ. "
                        "Existing aggregates: {}".format(
                            zone, [a['Name'] for a in aggregates]))
            else:
                LOG.warning("There are no aggregates. Compute will "
                            "be in default (nova) availability zone")

    # TODO: gvlasov_cleanup_runtime
    # Add temporary check for actual runtime on added node for 2.27 -> 2.28 -> 2.29
    if settings.DESIRED_RUNTIME:
        cluster.check.compare_machines_runtime_with_desired([nodeforscale], machine_is_new=True)

    show_step(8)
    waiters.wait(
        lambda: check_cmp_service_status(),
        timeout=600, interval=10,
        timeout_msg="Some cmp service is not ready")
    cluster.check.check_k8s_pods()
    cluster.check.check_cluster_readiness()
    show_step(10)
    new_compute_count = cmp_count + len(worker_scale)
    waiters.wait(
        lambda: check(count=new_compute_count),
        timeout=3660, interval=60,
        timeout_msg="new hypervisor not ready")
    cluster.check.check_diagnostic_cluster_status()

    cluster.check.check_bmh_inventory_presense()
