import base64
import time
import json
import yaml
from kubernetes.client.rest import ApiException

from si_tests import settings
from si_tests.utils import utils
from si_tests.deployments.utils import extra_context, file_utils, wait_utils, commons, kubectl_utils, helpers
from si_tests.deployments.utils.namespace import NAMESPACE


def get_osdpl_node_override(os_manager, kubectl):
    secret = os_manager.api.secrets.get("osdpl-nodes-override", NAMESPACE.openstack)
    data = {}
    try:
        for key, value in secret.read().data.items():
            data[key] = json.loads(commons.decode(value))
    except ApiException as e:
        if e.reason != "Not Found":
            raise e
        commons.LOG.info("Node overrides are empty.")
    return data


def get_value_from(secretName, fieldName):
    return {
        "value_from": {
            "secret_key_ref": {
                "name": secretName,
                "key": fieldName,
            }
        }
    }


def create_osdpl_hidden_secret(os_manager, name, data):
    data = {
        key: base64.b64encode(value.encode()).decode()
        for key, value in data.items()
    }
    secret_data = {
        "apiVersion": "v1",
        "kind": "Secret",
        "metadata": {
            "name": name,
            "namespace": NAMESPACE.openstack,
            "labels": {
                "openstack.lcm.mirantis.com/osdpl_secret": "true",
            }
        },
        "type": "Opaque",
        "data": data,
    }
    commons.LOG.info("Apply openstack secret for hidden data")
    os_manager.api.secrets.create(namespace=NAMESPACE.openstack, body=secret_data)


def get_vault_data(os_manager, kubectl, wait):
    commons.LOG.info("Use vault as barbican backend")

    commons.LOG.info("Wait for vault to be deployed")

    def _wait():
        assert len(os_manager.api.pods.list_starts_with("vault"))
        return True

    wait.wait_pass(_wait, expected=AssertionError)

    vault_secret = os_manager.api.secrets.get("vault-keys", NAMESPACE.vault)

    def _wait():
        secret_data = vault_secret.read()
        assert hasattr(secret_data, "data")
        assert "vault-root" in secret_data.data

    wait.wait_pass(_wait, expected=Exception)

    vault_token = commons.decode(vault_secret.read().data["vault-root"])

    commons.LOG.info("Enable approle backend")
    kubectl.exec(
        "vault-0",
        "sh -c 'export VAULT_TOKEN={}; " "vault auth enable approle || true'".format(vault_token),
        NAMESPACE.vault,
    )

    commons.LOG.info("Create policy")
    kubectl.exec(
        "vault-0",
        "sh -c 'export VAULT_TOKEN=%s;"
        ' echo "path \\"secret/*\\" {capabilities = [\\"read\\", '
        '\\"list\\", \\"create\\", \\"delete\\", \\"update\\"]}"'
        " | vault policy write barbican-policy -'" % vault_token,
        NAMESPACE.vault,
    )

    commons.LOG.info("Create role")
    kubectl.exec(
        "vault-0",
        "sh -c 'export VAULT_TOKEN={};"
        " vault write auth/approle/role/barbican-role "
        'token_policies="barbican-policy" '
        "bind_secret_id=true secret_id_num_uses=0 "
        "period=0'".format(vault_token),
        NAMESPACE.vault,
    )

    commons.LOG.info("Read role-id")
    role_yaml = kubectl.exec(
        "vault-0",
        "sh -c 'export VAULT_TOKEN={};"
        " vault read auth/approle/role/barbican-role/role-id "
        "-format=yaml'".format(vault_token),
        NAMESPACE.vault,
    ).result_yaml

    commons.LOG.info("Generate secret-id")
    secret_yaml = kubectl.exec(
        "vault-0",
        "sh -c 'export VAULT_TOKEN={}; vault write "
        "-f auth/approle/role/barbican-role/secret-id "
        "-format=yaml'".format(vault_token),
        NAMESPACE.vault,
    ).result_yaml

    vault_domain = ""
    if NAMESPACE.vault != NAMESPACE.openstack:
        vault_domain = ".{}.svc".format(NAMESPACE.vault)

    vault_conf = {
        "spec": {
            "features": {
                "barbican": {
                    "backends": {
                        "vault": {
                            "enabled": True,
                            "approle_role_id": role_yaml["data"]["role_id"],
                            "approle_secret_id": secret_yaml["data"]["secret_id"],
                            "vault_url": "http://vault{}:8200".format(vault_domain),
                            "use_ssl": False,
                        }
                    }
                }
            }
        }
    }
    return vault_conf


@utils.log_method_time()
def deploy_openstack_cluster(os_manager, timeout):
    hidden_data = {}
    commons.LOG.info("Read ssl certs")
    ca_cert, api_cert, api_key = file_utils.get_ssl()

    openstack_config = file_utils.get_osdpl_content()
    openstack_spec = openstack_config["spec"]

    openstack_features = openstack_spec["features"]
    if "services" not in openstack_features:
        openstack_features["services"] = []
    extra = extra_context.ExtraContext()
    context = extra.openstack_cluster

    dynamic_network_opts = extra.dynamic_network_opts
    utils.merge_dicts(openstack_config, dynamic_network_opts)

    if context:
        opts = extra.openstack_opts
        if opts:
            orig_networks = openstack_features["neutron"].get("external_networks", [])

            for network in orig_networks:
                bridge = network.get("bridge")
                if bridge == "br-ex":
                    network["interface"] = opts.get("external_network_interface")
                if bridge == "ironic-pxe":
                    network["interface"] = opts.get("ironic_baremetal_network_interface")
            if opts.get("ironic_baremetal_network_gateway"):
                brmtl = openstack_features["ironic"]["networks"]["baremetal"]
                brmtl["subnets"]["baremetal_subnet"]["gateway"] = opts.get("ironic_baremetal_network_gateway")

        utils.merge_dicts(openstack_config, context)

    commons.LOG.info("Set ssl features in openstack config")
    ssl_data = {
        "ca_cert": ca_cert,
        "api_cert": api_cert,
        "api_key": api_key
    }
    if os_manager.is_value_from_used:
        hidden_data.update(ssl_data)
        ssl_data = {
            "ca_cert": get_value_from("osh-dev-hidden", "ca_cert"),
            "api_cert": get_value_from("osh-dev-hidden", "api_cert"),
            "api_key": get_value_from("osh-dev-hidden", "api_key")
        }
    ssl_features = openstack_features["ssl"]
    ssl_features["public_endpoints"].update(ssl_data)

    openstack_version = settings.OPENSTACK_DEPLOY_OPENSTACK_VERSION

    commons.LOG.info("Set openstack version to %s", openstack_version)
    openstack_spec["openstack_version"] = openstack_version

    local_domain = helpers.get_local_domain(os_manager)

    openstack_spec["internal_domain_name"] = local_domain
    dns_config = {"spec": {"features": {"neutron": {"dns_servers": [helpers.get_external_dns_ip(os_manager)]}}}}
    utils.merge_dicts(openstack_config, dns_config)

    fake_install = settings.OPENSTACK_DEPLOY_FAKE_DEPLOYMENT
    openstack_spec["draft"] = fake_install

    if settings.OPENSTACK_DEPLOY_NEUTRON_PORTPROBER:
        portprober_deployment = {"spec": {"features": {"neutron": {"extensions": {"portprober": {"enabled": True}}}}}}
        utils.merge_dicts(openstack_config, portprober_deployment)

    kubectl = kubectl_utils.Kubectl(namespace=NAMESPACE.openstack)

    if settings.OPENSTACK_DEPLOY_TELEMETRY:
        telemetry_services = ["alarming", "metering", "metric"]
        services_to_add = list(
            filter(lambda service: service not in openstack_features["services"], telemetry_services)
        )
        openstack_features["services"].extend(services_to_add)
        openstack_features["telemetry"] = {"mode": "autoscaling"}

    if settings.OPENSTACK_DEPLOY_MANILA:
        manila_services = ["shared-file-system"]
        services_to_add = list(filter(lambda service: service not in openstack_features["services"], manila_services))
        openstack_features["services"].extend(services_to_add)

    if settings.OPENSTACK_DEPLOY_MASAKARI:
        masakari_services = ["instance-ha"]
        services_to_add = list(filter(lambda service: service not in openstack_features["services"], masakari_services))
        openstack_features["services"].extend(services_to_add)

    if settings.CEPH_DEPLOY_RGW:
        rgw_services = ["object-storage"]
        services_to_add = list(filter(lambda service: service not in openstack_features["services"], rgw_services))
        openstack_features["services"].extend(services_to_add)

    wait = wait_utils.Waiter(os_manager, timeout)

    if settings.OPENSTACK_DEPLOY_VAULT:
        vault_data = get_vault_data(os_manager, kubectl, wait)
        if os_manager.is_value_from_used:
            vault_hidden_data = vault_data["spec"]["features"]["barbican"]["backends"]["vault"]
            hidden_data.update({
                "approle_role_id": vault_hidden_data["approle_role_id"],
                "approle_secret_id": vault_hidden_data["approle_secret_id"]
            })
            vault_hidden_data["approle_role_id"] = get_value_from("osh-dev-hidden", "approle_role_id")
            vault_hidden_data["approle_secret_id"] = get_value_from("osh-dev-hidden", "approle_secret_id")
        openstack_config = utils.merge_dicts(openstack_config, vault_data)

    if settings.OPENSTACK_DEPLOY_IAM:
        keystone_feature = {
            "keycloak": {
                "enabled": True,
                "oidc": {"OIDCSSLValidateServer": False, "OIDCOAuthSSLValidateServer": False},
                "url": "https://keycloak.it.just.works",
            }
        }
        openstack_features["keystone"] = keystone_feature

    elif settings.OPENSTACK_USE_MGMT_IAM:
        # Get IAM URL from the osdpl container environment variables
        deployment_name = os_manager.oc_name
        os_controller = os_manager.api.deployments.get(deployment_name, "osh-system")
        os_containers = os_controller.read().spec.template.spec.containers
        osdpl_containers = [c for c in os_containers if c.name == "osdpl"]
        if not osdpl_containers:
            raise Exception(f"osdpl container not found in the deployment osh-system/{deployment_name}")
        OSDPL_IAM_DATA = [e for e in osdpl_containers[0].env if e.name == "OSDPL_IAM_DATA"]
        if not OSDPL_IAM_DATA:
            raise Exception(
                f"OSDPL_IAM_DATA not found in the env variables for the deployment osh-system/{deployment_name}"
            )
        iam_data = yaml.load(OSDPL_IAM_DATA[0].value, Loader=yaml.SafeLoader)
        iam_url = iam_data["url"]

        keystone_feature = {
            "keycloak": {
                "enabled": True,
                "oidc": {"OIDCSSLValidateServer": False, "OIDCOAuthSSLValidateServer": False},
                "url": iam_url,
            }
        }
        openstack_features["keystone"] = keystone_feature

    nodes_override = get_osdpl_node_override(os_manager, kubectl)
    if nodes_override:
        openstack_spec["nodes"] = {}
        for node_name, node_override in nodes_override.items():
            openstack_spec["nodes"][f"kubernetes.io/hostname::{node_name}"] = node_override

    if os_manager.is_value_from_used and hidden_data:
        create_osdpl_hidden_secret(os_manager, "osh-dev-hidden", hidden_data)

    new_openstack_yaml_path = file_utils.get_new_name(file_utils.get_osdpl())

    commons.LOG.info("Save openstack config to '%s'", new_openstack_yaml_path)
    file_utils.save_to_yaml(openstack_config, new_openstack_yaml_path)

    if fake_install:
        commons.LOG.info("Do not trigger openstack deployment... " "To deploy run update osdpl spec.draft to false")
        return

    commons.LOG.info("Apply openstack config")
    kubectl.apply(new_openstack_yaml_path)

    commons.LOG.info("Wait for openstack deployment")

    os_controller_version = os_manager.os_controller_version()
    commons.LOG.info(f"OpenStack controller version: {os_controller_version}")

    os_manager.wait_os_deployment_status(timeout=wait.timeout, status="APPLYING")
    os_manager.wait_os_deployment_status(timeout=wait.timeout, status="APPLIED")
    os_manager.wait_osdpl_services(status="APPLIED")

    commons.LOG.info("Wait osdpl health status=Ready")
    start_time = time.time()
    os_manager.wait_openstackdeployment_health_status(timeout=wait.timeout)
    wait.update_timeout(time.time() - start_time)

    commons.LOG.info("Wait os jobs to success and pods to become Ready")
    start_time = time.time()
    os_manager.wait_os_resources(timeout=wait.timeout, interval=20)
    wait.update_timeout(time.time() - start_time)

    job_suffix = utils.gen_random_string(3)
    # Force running cell setup unless PRODX-172 is implemented
    commons.LOG.info("Run cell setup")
    kubectl.create("job", "--from=cronjob/nova-cell-setup nova-cell-setup-pd01-{}".format(job_suffix))
    commons.LOG.info("%s", kubectl.result_str)
    commons.LOG.info("Openstack successfully deployed")

    if settings.OPENSTACK_DEPLOY_DPDK and settings.OPENSTACK_DPDK_AVAILABILITY_ZONE_NAME != "nova":
        advanced_nodes = [node for node, data in nodes_override.items() if data.get(
            'features', {}).get('neutron', {}).get('dpdk', {}).get('enabled', False)]
        commons.LOG.info("Creating dedicated availability zone with dpdk " f"computes: {advanced_nodes}")
        keystone_client_name = (
            os_manager.api.pods.list_starts_with("keystone-client", NAMESPACE.openstack)[0].read().metadata.name
        )
        aggregate_create_cmd = (
            "bash -c 'openstack aggregate list | grep -q "
            "advanced-compute-aggregate || openstack aggregate create "
            f"--zone {settings.OPENSTACK_DPDK_AVAILABILITY_ZONE_NAME} "
            "advanced-compute-aggregate'"
        )
        kubectl.exec(keystone_client_name, aggregate_create_cmd, namespace=NAMESPACE.openstack)

        if openstack_features["neutron"].get("backend") == 'tungstenfabric':
            advanced_nodes.extend(kubectl.get_node_name_by_label(
                settings.TF_NODES_WITH_DPDK_LABEL))
        for node_name in set(advanced_nodes):
            commons.LOG.info(f"Adding {node_name} to aggregate")
            add_host_cmd = (
                "bash -c 'openstack aggregate show advanced-compute-aggregate"
                f" -f value -c hosts |grep -q {node_name} || openstack "
                " aggregate add host advanced-compute-aggregate "
                f"{node_name}'"
            )
            kubectl.exec(keystone_client_name, add_host_cmd, namespace=NAMESPACE.openstack)
