import os
import re
import yaml
import tempfile

from si_tests import settings

from si_tests.utils import utils
from si_tests.deployments.utils import (file_utils, wait_utils,
                                        commons, kubectl_utils, helpers)
from si_tests.deployments.utils.namespace import NAMESPACE


@utils.log_method_time()
def deploy_dns(os_manager, timeout):
    kubectl = kubectl_utils.Kubectl()
    commons.LOG.info("Deploy coredns")

    # Substitute links to charts/images in case of caching proxy.
    helmbundle = file_utils.get_yaml_content(file_utils.get_release("ci/30-coredns.yaml"))
    if settings.KAAS_OFFLINE_DEPLOYMENT:
        for release in helmbundle['spec']['releases']:
            chartUrl = release['chartURL']
            chartUrl = chartUrl.replace('https://binary.mirantis.com', 'http://127.0.0.1:8801/bin')
            release['chartURL'] = chartUrl
            repo = release['values']['image']['repository']
            repo = repo.replace('mirantis.azurecr.io', '127.0.0.1:44301')
            release['values']['image']['repository'] = repo

    with tempfile.NamedTemporaryFile(mode="w") as f:
        yaml.dump(helmbundle, f)
        kubectl.apply(f.name)

    commons.LOG.info("Wait for coredns service")
    core_dns_service = os_manager.api.services.get("coredns-coredns", NAMESPACE.coredns)
    wait = wait_utils.Waiter(os_manager, timeout)
    wait.wait_pass(wait.read_checker, k8s_object=core_dns_service, expected=Exception)

    ext_dns_ip = helpers.get_external_dns_ip(os_manager)
    commons.LOG.info("Check is coredns or kubedns")
    is_core_dns = helpers.is_core_dns(os_manager)

    if is_core_dns:
        commons.LOG.info("is_core_dns=True")
        wait.replicasets(name='coredns', namespace=NAMESPACE.coredns)
        commons.LOG.info("Coredns successfully deployed")
        return

    commons.LOG.info("Configure kubedns")
    kube_dns_content = """
    apiVersion: v1
    kind: ConfigMap
    metadata:
      name: kube-dns
      namespace: kube-system
    data:
      stubDomains: |
        {"mirantis.net": ["172.18.208.44"],
         "it.just.works": ["%s"]
        }
    """ % ext_dns_ip
    artifacts_dir = settings.ARTIFACTS_DIR

    kube_dns_yaml = os.path.abspath(os.path.join(artifacts_dir, "kube-dns.yml"))
    with open(kube_dns_yaml, "w") as f:
        f.write(kube_dns_content)

    commons.LOG.info("Deploy kubedns")
    kubectl.apply(kube_dns_yaml)

    wait.replicasets(name='coredns', namespace=NAMESPACE.kube_system)

    commons.LOG.info("Kubedns successfully deployed")


@utils.log_method_time()
def configure_dns(os_manager, timeout):
    configure_dns_kubesys(os_manager=os_manager, timeout=timeout)

    if settings.OPENSTACK_DEPLOY_FAKE_DEPLOYMENT:
        commons.LOG.info("Wont configure dns, since fake openstack deployment")
        return

    commons.LOG.info("Get ingress ip")
    ingress_service = os_manager.api.services.get("ingress", "openstack")

    wait = wait_utils.Waiter(os_manager, timeout)

    wait.wait_pass(wait.read_checker, k8s_object=ingress_service, expected=Exception)

    ingress_service = ingress_service.read()
    ingress_ip = ingress_service.status.load_balancer.ingress[0].ip
    ingress_ip = ingress_ip or ingress_service.spec.cluster_ip

    commons.LOG.info("Update coredns yaml")
    coredns = os_manager.get_os_coredns_helmbundle()
    zone_files = coredns.spec["releases"][0]["values"]["zoneFiles"]
    dns_content = zone_files[0]["contents"].replace("1.2.3.4", ingress_ip)
    kubectl = kubectl_utils.Kubectl(namespace=NAMESPACE.kube_system)

    if settings.OPENSTACK_DEPLOY_IAM:
        iam_service = os_manager.api.services.get(
            "openstack-iam-keyclo-http", "iam").read()

        iam_ext_ip = iam_service.status.load_balancer.ingress[0].ip
        iam_domain = "\n          keycloak.it.just.works.   " \
                     " IN      A       {}".format(iam_ext_ip)
        if iam_domain not in dns_content:
            dns_content = dns_content + iam_domain

    zone_files[0]["contents"] = dns_content

    commons.LOG.info("Update coredns")
    os_manager.api.kaas_helmbundles.update(coredns.metadata.name, coredns.metadata.namespace, body=coredns.to_dict())

    commons.LOG.info("Wait for coredns service")
    core_dns_service = os_manager.api.services.get("coredns-coredns", NAMESPACE.coredns)

    wait.wait_pass(wait.read_checker, k8s_object=core_dns_service, expected=Exception)
    wait.replicasets(name='coredns', namespace=NAMESPACE.coredns)
    wait.wait_pass(lambda: wait.wait_for_dns_warmup(kubectl,
                                                    keystone_pod_name=os_manager.api.pods.list_starts_with(
                                                        "keystone-client", NAMESPACE.openstack
                                                    )[0].read().metadata.name,
                                                    resolved_expected_ip=ingress_ip,
                                                    ns=NAMESPACE.openstack),
                   expected=Exception)
    commons.LOG.info("coredns successfully configured")

    if settings.OPENSTACK_DEPLOY_IAM or settings.OPENSTACK_USE_MGMT_IAM:
        commons.LOG.info('Creating openstack group and role for m:os@admin')
        keystone_client_name = os_manager.api.pods.list_starts_with(
            "keystone-client", NAMESPACE.openstack
        )[0].read().metadata.name
        kubectl.exec(
            keystone_client_name,
            "bash -c 'openstack group create m:os@admin"
            " && openstack role add --project admin --group "
            "m:os@admin member'",
            namespace=NAMESPACE.openstack)
        commons.LOG.info('Creating openstack group and role for m:os@admin finished')


def configure_dns_kubesys(os_manager, timeout):
    commons.LOG.info("Check is coredns or kubedns")
    is_core_dns = helpers.is_core_dns(os_manager)
    kubectl = kubectl_utils.Kubectl(namespace=NAMESPACE.kube_system,
                                    kubeconfig=os_manager.kubeconfig)

    if is_core_dns:
        commons.LOG.info("is_core_dns=True")
        coredns_config = os_manager.api.configmaps.get(
            "coredns", NAMESPACE.kube_system)
        dns_content = coredns_config.read().data["Corefile"]
        external_dns_ip = helpers.get_external_dns_ip(os_manager)
        os_dns_section = """
it.just.works:53 {
    errors
    cache 30
    forward . %s
}""" % external_dns_ip
        if os_dns_section not in dns_content:
            dns_content_re = re.sub(r"it.just.works:53 \{([^\$]+)\}",
                                    os_dns_section, dns_content, 0)
            if dns_content_re == dns_content:
                dns_content = dns_content + os_dns_section
            else:
                dns_content = dns_content_re
            coredns_config.patch(
                {"data": {"Corefile": dns_content}})

        coredns_pods = os_manager.api.pods.list_starts_with(
            "coredns", NAMESPACE.kube_system)
        for pod in coredns_pods:
            kubectl.delete("pod", pod.name)

        wait = wait_utils.Waiter(os_manager, timeout, NAMESPACE.kube_system)
        wait.pods("coredns", replicas=1)
        wait.replicasets(name="coredns", namespace=NAMESPACE.coredns)

    commons.LOG.info("coredns successfully configured")
