import datetime
import shutil
import subprocess

import requests
import pytest
import tarfile

from si_tests import logger
from si_tests import settings
import si_tests.utils.templates as template_utils
from pathlib import Path


LOG = logger.logger

EXPECTED_FILES = [
    "version.yaml",
    "cluster-resources/priorityclasses.json",
    "cluster-resources/pvcs/orc-system.json",
    "cluster-resources/pvcs/kube-public.json",
    "cluster-resources/pvcs/k0s-autopilot.json",
    "cluster-resources/pvcs/kcm-system.json",
    "cluster-resources/pvcs/kube-system.json",
    "cluster-resources/pvcs/projectsveltos.json",
    "cluster-resources/pvcs/default.json",
    "cluster-resources/pvcs/mgmt.json",
    "cluster-resources/pvcs/kube-node-lease.json",
    "cluster-resources/cronjobs/orc-system.json",
    "cluster-resources/cronjobs/kube-public.json",
    "cluster-resources/cronjobs/k0s-autopilot.json",
    "cluster-resources/cronjobs/kcm-system.json",
    "cluster-resources/cronjobs/kube-system.json",
    "cluster-resources/cronjobs/projectsveltos.json",
    "cluster-resources/cronjobs/default.json",
    "cluster-resources/cronjobs/mgmt.json",
    "cluster-resources/cronjobs/kube-node-lease.json",
    "cluster-resources/pvs.json",
    "cluster-resources/network-policy/orc-system.json",
    "cluster-resources/network-policy/kube-public.json",
    "cluster-resources/network-policy/k0s-autopilot.json",
    "cluster-resources/network-policy/kcm-system.json",
    "cluster-resources/network-policy/kube-system.json",
    "cluster-resources/network-policy/projectsveltos.json",
    "cluster-resources/network-policy/default.json",
    "cluster-resources/network-policy/mgmt.json",
    "cluster-resources/network-policy/kube-node-lease.json",
    "cluster-resources/configmaps/orc-system.json",
    "cluster-resources/configmaps/kube-public.json",
    "cluster-resources/configmaps/k0s-autopilot.json",
    "cluster-resources/configmaps/kcm-system.json",
    "cluster-resources/configmaps/kube-system.json",
    "cluster-resources/configmaps/projectsveltos.json",
    "cluster-resources/configmaps/default.json",
    "cluster-resources/configmaps/mgmt.json",
    "cluster-resources/configmaps/kube-node-lease.json",
    "cluster-resources/endpoints/orc-system.json",
    "cluster-resources/endpoints/kube-public.json",
    "cluster-resources/endpoints/k0s-autopilot.json",
    "cluster-resources/endpoints/kcm-system.json",
    "cluster-resources/endpoints/kube-system.json",
    "cluster-resources/endpoints/projectsveltos.json",
    "cluster-resources/endpoints/default.json",
    "cluster-resources/endpoints/mgmt.json",
    "cluster-resources/endpoints/kube-node-lease.json",
    "cluster-resources/limitranges/orc-system.json",
    "cluster-resources/limitranges/kube-public.json",
    "cluster-resources/limitranges/k0s-autopilot.json",
    "cluster-resources/limitranges/kcm-system.json",
    "cluster-resources/limitranges/kube-system.json",
    "cluster-resources/limitranges/projectsveltos.json",
    "cluster-resources/limitranges/default.json",
    "cluster-resources/limitranges/mgmt.json",
    "cluster-resources/limitranges/kube-node-lease.json",
    "cluster-resources/statefulsets/orc-system.json",
    "cluster-resources/statefulsets/kube-public.json",
    "cluster-resources/statefulsets/k0s-autopilot.json",
    "cluster-resources/statefulsets/kcm-system.json",
    "cluster-resources/statefulsets/kube-system.json",
    "cluster-resources/statefulsets/projectsveltos.json",
    "cluster-resources/statefulsets/default.json",
    "cluster-resources/statefulsets/mgmt.json",
    "cluster-resources/statefulsets/kube-node-lease.json",
    "cluster-resources/statefulsets-errors.json/orc-system.json",
    "cluster-resources/statefulsets-errors.json/kube-public.json",
    "cluster-resources/statefulsets-errors.json/k0s-autopilot.json",
    "cluster-resources/statefulsets-errors.json/kcm-system.json",
    "cluster-resources/statefulsets-errors.json/kube-system.json",
    "cluster-resources/statefulsets-errors.json/projectsveltos.json",
    "cluster-resources/statefulsets-errors.json/default.json",
    "cluster-resources/statefulsets-errors.json/mgmt.json",
    "cluster-resources/statefulsets-errors.json/kube-node-lease.json",
    "cluster-resources/ingress/orc-system.json",
    "cluster-resources/ingress/kube-public.json",
    "cluster-resources/ingress/k0s-autopilot.json",
    "cluster-resources/ingress/kcm-system.json",
    "cluster-resources/ingress/kube-system.json",
    "cluster-resources/ingress/projectsveltos.json",
    "cluster-resources/ingress/default.json",
    "cluster-resources/ingress/mgmt.json",
    "cluster-resources/ingress/kube-node-lease.json",
    "cluster-resources/rolebindings/orc-system.json",
    "cluster-resources/rolebindings/kube-public.json",
    "cluster-resources/rolebindings/k0s-autopilot.json",
    "cluster-resources/rolebindings/kcm-system.json",
    "cluster-resources/rolebindings/kube-system.json",
    "cluster-resources/rolebindings/projectsveltos.json",
    "cluster-resources/rolebindings/default.json",
    "cluster-resources/rolebindings/mgmt.json",
    "cluster-resources/rolebindings/kube-node-lease.json",
    "cluster-resources/roles/orc-system.json",
    "cluster-resources/roles/kube-public.json",
    "cluster-resources/roles/k0s-autopilot.json",
    "cluster-resources/roles/kcm-system.json",
    "cluster-resources/roles/kube-system.json",
    "cluster-resources/roles/projectsveltos.json",
    "cluster-resources/roles/default.json",
    "cluster-resources/roles/mgmt.json",
    "cluster-resources/roles/kube-node-lease.json",
    "cluster-resources/resources.json",
    "cluster-resources/serviceaccounts/orc-system.json",
    "cluster-resources/serviceaccounts/kube-public.json",
    "cluster-resources/serviceaccounts/k0s-autopilot.json",
    "cluster-resources/serviceaccounts/kcm-system.json",
    "cluster-resources/serviceaccounts/kube-system.json",
    "cluster-resources/serviceaccounts/projectsveltos.json",
    "cluster-resources/serviceaccounts/default.json",
    "cluster-resources/serviceaccounts/mgmt.json",
    "cluster-resources/serviceaccounts/kube-node-lease.json",
    "cluster-resources/clusterrolebindings.json",
    "cluster-resources/volumeattachments.json",
    "cluster-resources/namespaces.json",
    "cluster-resources/custom-resources/debuggingconfigurations.lib.projectsveltos.io.yaml",
    "cluster-resources/custom-resources/debuggingconfigurations.lib.projectsveltos.io.json",
    "cluster-resources/custom-resources/managements.k0rdent.mirantis.com.json",
    "cluster-resources/custom-resources/accessmanagements.k0rdent.mirantis.com.yaml",
    "cluster-resources/custom-resources/managements.k0rdent.mirantis.com.yaml",
    "cluster-resources/custom-resources/accessmanagements.k0rdent.mirantis.com.json",
    "cluster-resources/custom-resources/classifiers.lib.projectsveltos.io.yaml",
    "cluster-resources/custom-resources/releases.k0rdent.mirantis.com.yaml",
    "cluster-resources/custom-resources/releases.k0rdent.mirantis.com.json",
    "cluster-resources/custom-resources/classifiers.lib.projectsveltos.io.json",
    "cluster-resources/custom-resources/providerinterfaces.k0rdent.mirantis.com.yaml",
    "cluster-resources/custom-resources/providertemplates.k0rdent.mirantis.com.json",
    "cluster-resources/custom-resources/controlnodes.autopilot.k0sproject.io.yaml",
    "cluster-resources/custom-resources/providerinterfaces.k0rdent.mirantis.com.json",
    "cluster-resources/custom-resources/controlnodes.autopilot.k0sproject.io.json",
    "cluster-resources/custom-resources/providertemplates.k0rdent.mirantis.com.yaml",
    "cluster-resources/leases/orc-system.json",
    "cluster-resources/leases/kube-public.json",
    "cluster-resources/leases/k0s-autopilot.json",
    "cluster-resources/leases/kcm-system.json",
    "cluster-resources/leases/kube-system.json",
    "cluster-resources/leases/projectsveltos.json",
    "cluster-resources/leases/default.json",
    "cluster-resources/leases/mgmt.json",
    "cluster-resources/leases/kube-node-lease.json",
    "cluster-resources/pods/orc-system.json",
    "cluster-resources/pods/kube-public.json",
    "cluster-resources/pods/k0s-autopilot.json",
    "cluster-resources/pods/kcm-system.json",
    "cluster-resources/pods/kube-system.json",
    "cluster-resources/pods/projectsveltos.json",
    "cluster-resources/pods/default.json",
    "cluster-resources/pods/mgmt.json",
    "cluster-resources/pods/kube-node-lease.json",
    "cluster-resources/daemonsets/orc-system.json",
    "cluster-resources/daemonsets/kube-public.json",
    "cluster-resources/daemonsets/k0s-autopilot.json",
    "cluster-resources/daemonsets/kcm-system.json",
    "cluster-resources/daemonsets/kube-system.json",
    "cluster-resources/daemonsets/projectsveltos.json",
    "cluster-resources/daemonsets/default.json",
    "cluster-resources/daemonsets/mgmt.json",
    "cluster-resources/daemonsets/kube-node-lease.json",
    "cluster-resources/groups.json",
    "cluster-resources/auth-cani-list/orc-system.json",
    "cluster-resources/auth-cani-list/kube-public.json",
    "cluster-resources/auth-cani-list/k0s-autopilot.json",
    "cluster-resources/auth-cani-list/kcm-system.json",
    "cluster-resources/auth-cani-list/kube-system.json",
    "cluster-resources/auth-cani-list/projectsveltos.json",
    "cluster-resources/auth-cani-list/default.json",
    "cluster-resources/auth-cani-list/mgmt.json",
    "cluster-resources/auth-cani-list/kube-node-lease.json",
    "cluster-resources/endpointslices/orc-system.json",
    "cluster-resources/endpointslices/kube-public.json",
    "cluster-resources/endpointslices/k0s-autopilot.json",
    "cluster-resources/endpointslices/kcm-system.json",
    "cluster-resources/endpointslices/kube-system.json",
    "cluster-resources/endpointslices/projectsveltos.json",
    "cluster-resources/endpointslices/default.json",
    "cluster-resources/endpointslices/mgmt.json",
    "cluster-resources/endpointslices/kube-node-lease.json",
    "cluster-resources/pod-disruption-budgets-errors.json",
    "cluster-resources/clusterroles.json",
    "cluster-resources/jobs/orc-system.json",
    "cluster-resources/jobs/kube-public.json",
    "cluster-resources/jobs/k0s-autopilot.json",
    "cluster-resources/jobs/kcm-system.json",
    "cluster-resources/jobs/kube-system.json",
    "cluster-resources/jobs/projectsveltos.json",
    "cluster-resources/jobs/default.json",
    "cluster-resources/jobs/mgmt.json",
    "cluster-resources/jobs/kube-node-lease.json",
    "cluster-resources/events/orc-system.json",
    "cluster-resources/events/kube-public.json",
    "cluster-resources/events/k0s-autopilot.json",
    "cluster-resources/events/kcm-system.json",
    "cluster-resources/events/kube-system.json",
    "cluster-resources/events/projectsveltos.json",
    "cluster-resources/events/default.json",
    "cluster-resources/events/mgmt.json",
    "cluster-resources/events/kube-node-lease.json",
    "cluster-resources/custom-resource-definitions.json",
    "cluster-resources/storage-classes.json",
    "cluster-resources/resource-quota/orc-system.json",
    "cluster-resources/resource-quota/kube-public.json",
    "cluster-resources/resource-quota/k0s-autopilot.json",
    "cluster-resources/resource-quota/kcm-system.json",
    "cluster-resources/resource-quota/kube-system.json",
    "cluster-resources/resource-quota/projectsveltos.json",
    "cluster-resources/resource-quota/default.json",
    "cluster-resources/resource-quota/mgmt.json",
    "cluster-resources/resource-quota/kube-node-lease.json",
    "cluster-resources/services/orc-system.json",
    "cluster-resources/services/kube-public.json",
    "cluster-resources/services/k0s-autopilot.json",
    "cluster-resources/services/kcm-system.json",
    "cluster-resources/services/kube-system.json",
    "cluster-resources/services/projectsveltos.json",
    "cluster-resources/services/default.json",
    "cluster-resources/services/mgmt.json",
    "cluster-resources/services/kube-node-lease.json",
    "cluster-resources/nodes.json",
    "cluster-resources/deployments/orc-system.json",
    "cluster-resources/deployments/kube-public.json",
    "cluster-resources/deployments/k0s-autopilot.json",
    "cluster-resources/deployments/kcm-system.json",
    "cluster-resources/deployments/kube-system.json",
    "cluster-resources/deployments/projectsveltos.json",
    "cluster-resources/deployments/default.json",
    "cluster-resources/deployments/mgmt.json",
    "cluster-resources/deployments/kube-node-lease.json",
    "cluster-info/cluster_version.json",
    "execution-data/summary.txt",
    "analysis.json",
]


@pytest.fixture(scope="module")
def support_dump_binary():
    """
    Fixture to download the ReplicatedHQ support dump for a given version.
    """
    # If binary already exists, return its path
    support_bundle_bin_path = f"{settings.SI_BINARIES_DIR}/support-bundle"
    if Path(support_bundle_bin_path).exists():
        return support_bundle_bin_path
    version = settings.KSI_SUPPORT_BUNDLE_VERSION
    platform = (
        "darwin"
        if subprocess.run(["uname"], capture_output=True, text=True).stdout.strip()
        == "Darwin"
        else "linux"
    )

    url = (
        f"https://github.com/replicatedhq/troubleshoot/releases/download/"
        f"v{version}/support-bundle_{platform}_amd64.tar.gz"
    )
    response = requests.get(url, stream=True)
    if response.status_code == 200:
        with open("support-dump.tar.gz", "wb") as file:
            for chunk in response.iter_content(chunk_size=8192):
                file.write(chunk)
        # Extract the tar file and move the binary to bin directory
        with tarfile.open("support-dump.tar.gz", "r:gz") as tar:
            tar.extractall()
        shutil.move("support-bundle", support_bundle_bin_path)
        # Make the support-bundle binary executable
        subprocess.run(["chmod", "+x", support_bundle_bin_path], check=True)
        return support_bundle_bin_path
    else:
        pytest.fail(
            f"Failed to download support dump for version {version} at {url}, status code: {response.status_code}"
        )


@pytest.fixture(scope="module")
def render_config(tmp_path_factory):
    """
    Fixture to render the configuration for the support dump binary.
    """
    options = {
        "namespace": settings.KCM_NAMESPACE,
    }
    content = template_utils.render_template(
        settings.KSI_SUPPORT_BUNDLE_TEMPLATE_PATH,
        options,
    )
    output_path = tmp_path_factory.mktemp("config") / "rendered_config.yaml"
    with open(output_path, "w") as file:
        file.write(content)
    yield output_path
    # Cleanup the rendered config file after the test
    if output_path and Path(output_path).exists():
        Path(output_path).unlink()  # Remove the rendered config file after the test


@pytest.fixture(scope="module")
def gather_support_bundle(support_dump_binary, render_config):
    """
    Fixture to gather a support bundle using the support dump binary.
    """
    binary_path = support_dump_binary
    date_suffix = datetime.datetime.now().strftime("%Y%m%d_%H%M%S")
    filepath = Path(f"support-dump_{date_suffix}.tar.gz")
    if not settings.KSI_KUBECONFIG_PATH:
        pytest.fail("KSI_KUBECONFIG_PATH is not set in the settings.")
    result = subprocess.run(
        [
            binary_path,
            "--kubeconfig",
            settings.KSI_KUBECONFIG_PATH,
            "--output",
            filepath,
            "--debug",
            render_config,
        ],
        capture_output=True,
        text=True,
    )
    LOG.info(f"Gather support dump output: {result.stdout.strip()}")

    # Copy support dump to artifacts directory
    artifacts_dir = Path(settings.ARTIFACTS_DIR)
    shutil.copy(filepath, artifacts_dir / filepath)

    if result.returncode != 0:
        pytest.fail(
            f"Support dump binary exited with non-zero status: {result.returncode} {result.stderr.strip()}"
        )
    yield filepath

    # Cleanup the support bundle file after the test

    if filepath.exists():
        filepath.unlink()


@pytest.fixture(scope="module")
def extract_support_bundle(gather_support_bundle, tmp_path_factory):
    """
    Fixture to extract the support bundle into a temporary directory.
    """
    # Path to the support bundle
    support_bundle_path = gather_support_bundle
    # Create a temporary directory for the extracted files
    extract_dir = tmp_path_factory.mktemp("supportbundle")

    # Extract the tar.gz file
    with tarfile.open(support_bundle_path, "r:gz") as tar:
        tar.extractall(path=extract_dir)
        # Get the root directory name of the bundle
        root_dir_name = tar.getmembers()[0].name.split("/")[0]
        LOG.info(f"Root directory of the bundle: {root_dir_name}")
    return extract_dir / root_dir_name


@pytest.fixture(params=EXPECTED_FILES, scope="module")
def file_from_list(request, extract_support_bundle):
    """
    Fixture to provide each file path from the EXPECTED_FILES list as a parameter.
    """
    return extract_support_bundle / request.param


def test_render_config(render_config):
    """
    Test that the KSI_SUPPORT_BUNDLE_TEMPLATE_PATH exists and is set in the settings.
    """
    assert Path(render_config).exists(), (
        f"KSI_SUPPORT_BUNDLE_TEMPLATE_PATH does not exist: {render_config}"
    )


def test_support_dump_binary_version(support_dump_binary):
    """
    Test that calls the support dump binary with version and ensures it exits successfully.
    """
    binary_path = support_dump_binary
    result = subprocess.run([binary_path, "version"], capture_output=True, text=True)
    # Check that version is in output
    assert settings.KSI_SUPPORT_BUNDLE_VERSION in result.stdout, (
        f"{settings.KSI_SUPPORT_BUNDLE_VERSION} not found in output: {result.stdout.strip()}"
    )
    print(f"Support dump binary version output: {result.stdout.strip()}")
    assert result.returncode == 0, (
        f"Binary exited with non-zero status: {result.returncode}"
    )


def test_gather_support_dump(gather_support_bundle):
    """
    Test that gathers a support dump verifies file exists.
    """
    # Check if the support dump file exists
    assert gather_support_bundle.exists(), "Support dump file was not created."
    print(f"Gathered support dump file: {gather_support_bundle.name}")


@pytest.mark.skip(reason="Skipping by default")
def test_file_exists_enterprise(file_from_list):
    """
    Test that specific files exist in the extracted support bundle.
    """
    # Check if the file exists
    assert file_from_list.exists(), f"File {file_from_list.name} does not exist."
