import pytest
import yaml
import os

from si_tests import logger
from si_tests import settings
from si_tests.utils import templates

LOG = logger.logger


def str_presenter(dumper, data):
    if len(data.splitlines()) > 1:
        return dumper.represent_scalar('tag:yaml.org,2002:str', data, style='|')
    return dumper.represent_scalar('tag:yaml.org,2002:str', data)


yaml.add_representer(str, str_presenter)


class KSIConfigManager(object):
    """
    Simple wrapper logic, on top of KSI_CONFIG yaml file.
    Required to prevent uncontrolled write\re-write access from any place in tests.
    Aka, you must use save_data_to_ksi_config, instead direct write to yaml.
    """

    def __init__(self, ksi_config_path=settings.KSI_CONFIG_PATH):
        self.ksi_config_path = ksi_config_path
        if self.ksi_config_path and os.path.isfile(self.ksi_config_path):
            with open(self.ksi_config_path, 'r') as f:
                self.__data = yaml.load(f.read(), Loader=yaml.SafeLoader)
        else:
            pytest.fail(f"ksi_config at  {ksi_config_path} not exists!")

    def save_data_to_ksi_config(self, to_key, data=None) -> None:
        if not data:
            LOG.warning('save_data_to_ksi_config: no data proposed!')
            return
        LOG.info(f'Updating KSI_CONFIG:{to_key} ')
        with templates.YamlEditor(self.ksi_config_path) as editor:
            current_content = editor.content
            current_content[to_key] = data
            editor.content = current_content

    @property
    def data(self) -> dict:
        """Raw data from KSI_CONFIG
            :rtype: dict
        """
        return self.__data

    @data.setter
    def data(self, new_data):
        """Update data in the ksi_config.yaml with extra_dict data"""
        self.__data = new_data
        if self.ksi_config_path and os.path.isfile(self.ksi_config_path):
            with open(self.ksi_config_path, 'w') as f:
                f.write(str(yaml.dump(self.__data)))

    def store_seed_ip(self, seed_ip):
        """
        NIT(alexz-kh): perhaps, we need to add deep_merge function someday
        :param seed_ip:
        :return:
        """
        _remote = self.data.get('run_on_remote', {}) or dict()
        _seed_ip = _remote.get('KSI_SEED_STANDALONE_EXTERNAL_IP', '')
        if _seed_ip:
            pytest.fail(f"KSI_SEED_STANDALONE_EXTERNAL_IP:{_seed_ip}. Overwrite is prohibited!")
        _new_remote = _remote.copy()
        _new_remote['KSI_SEED_STANDALONE_EXTERNAL_IP'] = seed_ip
        LOG.info(f'Writing seed_ip:{seed_ip} to KSI_CONFIG:run_on_remote:KSI_SEED_STANDALONE_EXTERNAL_IP')
        self.save_data_to_ksi_config(to_key='run_on_remote', data=_new_remote)

    # def get_seed_ip(self) -> str:
    #     assert 'KSI_SEED_IP' in self.data, "KSI_CONFIG not found in KSI_CONFIG"
    #     return self.__data['KSI_SEED_IP']

    def get_os_heat_stack_info(self, strict=True) -> dict:
        """Get stored heat stack info from ksi_config.yaml"""
        if strict:
            assert 'openstack_env' in self.data, "No openstack_env found in KSI_CONFIG"
        return self.data.get('openstack_env_data', dict())
