#    Copyright 2019 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

import base64
import os

import cachetools.func as cachetools_func
import json

import pytest
from kubernetes.client.rest import ApiException
from tabulate import tabulate
import yaml

import si_tests.settings as settings
from si_tests import logger
from si_tests.clients.k8s import K8sCluster
from si_tests.clients.k8s.k0_clusterdeployment import K0rdentClusterDeployment
from si_tests.managers.clusterdeploymentcheck_manager import ClusterDeploymentCheckManager
from si_tests.managers.kof_manager import KOFManager
from si_tests.managers.helm3_manager import Helm3Manager
from si_tests.managers.machine_manager import (MachineProviders, AwsProviderMachine, AzureProviderMachine,
                                               VSphereProviderMachine, OpenStackProviderMachine, GCPProviderMachine,
                                               EKSProviderMachine, Metal3Machines)
from si_tests.fixtures.decorators import collect_cluster_readiness
import si_tests.utils.templates as template_utils
from si_tests.utils import utils, waiters, ssl_tools
from si_tests.utils import packaging_version as version

from typing import List

LOG = logger.logger

system_namespaces = [
    'default', 'kube-system', 'kube-public', 'kcm-system', 'kof'
    'kube-node-lease', 'local-path-storage'
]


class Manager(object):
    kubeconfig = None

    def __init__(self, kubeconfig=settings.KUBECONFIG_PATH, auth_data=None):
        self._api = None
        self.kubeconfig = kubeconfig
        self.auth_data = auth_data
        self.__expected_pods = None
        self.__mgmt_object = None
        self.__release = None
        self._helmmgr = None
        self._kof = None
        self._source = None

    @property
    def api(self) -> K8sCluster:
        """
            :rtype: cluster.K8sCluster
        """
        if self._api is None:
            if self.kubeconfig:
                if os.path.isfile(self.kubeconfig):
                    self._api = K8sCluster(kubeconfig=self.kubeconfig)
                else:
                    raise FileNotFoundError(f"kubeconfig file={self.kubeconfig} not found!")
            elif self.auth_data:
                config = yaml.load(self.auth_data, Loader=yaml.SafeLoader)
                self._api = K8sCluster(config_data=config)
            else:
                raise RuntimeError('Neither kubeconfig nor auth_data provided!')
            try:
                self._api.pods.list()
                LOG.info('Provided cluster available')
            except Exception as e:
                LOG.error('Provided kubeconfig navigates to unavailable cluster')
                raise e
        return self._api

    @property
    def mgmt(self):
        if not self.__mgmt_object:
            self.__mgmt_object = self.get_mgmt_object()
        return self.__mgmt_object

    @property
    def release(self):
        """Release of management object"""
        if not self.__release:
            self.__release = self.get_release(self.mgmt.release)
        return self.__release

    @property
    def expected_pods_template_path(self):
        # oss determination
        mgmt_ver = 'kcm-' + self.release.version.replace('.', '-')
        if self.release.is_dev:
            mgmt_ver = 'kcm-main'
        elif not self.release.is_dev and '-rc' in self.release.version:
            mgmt_ver = 'kcm-' + self.release.version.split('-')[0].replace('.', '-')

        # enterprise override determination
        if self.release.is_enterprise:
            dev_match = ['-rc', '-alpha', '-beta']
            if any(suff in self.release.version for suff in dev_match):
                mgmt_ver = 'kcm-enterprise-' + self.release.version.split('-')[0].replace('.', '-')
            else:
                mgmt_ver = 'kcm-enterprise-' + self.release.version.replace('.', '-')
        # no dev versions builds observed for enterprise yet, but this is a proper place for that case

        path = settings.EXPECTED_PODS_TEMPLATES_DIR + 'kcm/' + mgmt_ver + '.yaml'
        return path

    @property
    def expected_pods(self):
        if not self.__expected_pods:
            self.refresh_expected_objects()
        return self.__expected_pods

    @property
    def helmmgr(self):
        if not self._helmmgr:
            self._helmmgr = self.get_helmmgr_for_ns()
        return self._helmmgr

    @property
    def kof(self) -> KOFManager:
        if not self._kof:
            self._kof = KOFManager(self)
        return self._kof

    @property
    def source(self):
        if not self._source:
            # This helmrepo can definitely determine the source of installation (enterprise, custom enterprise or oss)
            templates_hrepo = self.api.helmrepositories.get(name=settings.KSI_KCM_TEMPLATES_REPO_NAME,
                                                            namespace=settings.KCM_NAMESPACE)
            url = templates_hrepo.data['spec'].get('url', None)
            if 'ghcr.io' in url:
                source = 'opensource'
            elif 'registry.mirantis.com/k0rdent-enterprise' in url:
                source = 'enterprise'
            else:
                source = 'custom-enterprise'
            self._source = source
        return self._source

    def get_helmmgr_for_ns(self, namespace=None):
        """Helm manager without default namespace

        :param namespace:
        :return:
        """
        return Helm3Manager(kubeconfig=self.kubeconfig, auth_data=self.auth_data, namespace=namespace)

    def refresh_expected_objects(self):
        """Force update expected objects"""
        self.__expected_pods = self.get_expected_objects()

    def get_expected_objects(self, exclude_pods=None):
        """Returns list of expected pods (and their quantity for the cluster)"""
        # NOTE(va4st): This is compatible with clusterdeployment-based function which get expected objects.
        # Represented by the shortened copy of clusterdeployment-based function.
        # It will be used in 2 ways - render objects for basic mgmt (like k0s with kcm installed) and partially
        # for mgmt which were deployed by basic mgmt (usual standalone cluster by some template with kcm installed)
        cluster_type = 'management'

        json_body = self._render_expected_pods_template()
        all_pods = {}
        all_nodes = self.api.nodes.list_all()
        num_nodes = len(all_nodes)

        if exclude_pods:
            LOG.warning("These pods will be excluded from "
                        "expected pod list {}".format(exclude_pods))
        else:
            exclude_pods = []
        for ns in json_body[cluster_type]['namespaces']:
            pods = json_body[cluster_type]['namespaces'][ns]
            pods = {k: v for k, v in pods.items() if k not in exclude_pods}
            if ns not in all_pods:
                all_pods[ns] = pods
            else:
                all_pods[ns].update(pods)

        options = None

        if self.kof.kof_op_deployed() or self.kof.kof_is_deployed():
            # NOTE(va4st): it's not possible now to deploy kof without kof-operators chart,
            # so use this as main switch to determine kof enabled or not. The install can be started with istio,
            # so threat both conditions as deployment factor

            kof_ms = True if self.kof.kof_ms_deployed() else False
            kof_operators = True if self.kof.kof_op_deployed() else False
            kof_istio = True if self.kof.kof_is_deployed() else False
            kof_collectors = True if self.kof.kof_co_deployed() else False

            options = {
                'kof_mothership_installed': kof_ms,
                'kof_operators_installed': kof_operators,
                'kof_istio_installed': kof_istio,
                'kof_collectors_installed': kof_collectors,
                'num_nodes': num_nodes
            }
            path = self.kof.template_path
            json_body = self._render_expected_pods_template(path, options)

            # second render round. this needed because in theory we can have mgmt 1.1.0 and kof 1.0.0.
            # so render separately these components and merge into one depending on rendered versions
            for ns in json_body[cluster_type]['namespaces']:
                pods = json_body[cluster_type]['namespaces'][ns]
                pods = {k: v for k, v in pods.items() if k not in exclude_pods}
                if ns not in all_pods:
                    all_pods[ns] = pods
                else:
                    all_pods[ns].update(pods)

        LOG.info(json.dumps(all_pods, indent=4))
        return all_pods

    def _render_expected_pods_template(self, path=None, options=None):
        """Render the template with expected pods for the mgmt template

        By default, we do not have any daemonsets or any objects depending on node count in management cluster,
        so there no needs try to get nodes count and collect such data in this particular case.
        """
        if not path:
            path = self.expected_pods_template_path
        templates = template_utils.render_template(path, options)

        json_body = yaml.load(templates, Loader=yaml.SafeLoader)
        return json_body

    def create_namespace(self, namespace) -> "Namespace":
        ns = self.api.namespaces.create(
            namespace=namespace,
            body={
                "apiVersion": "v1",
                "kind": "Namespace",
                "metadata": {
                    "name": namespace
                }
            }
        )
        return Namespace(self, ns)

    def create_namespace_raw(self, data) -> "Namespace":
        ns = self.api.namespaces.create(
            namespace=data['metadata']['name'],
            body=data,
        )
        LOG.info("create_namespace_raw")
        return Namespace(self, ns)

    @cachetools_func.ttl_cache(ttl=10)
    def get_or_create_namespace(self, namespace) -> "Namespace":
        try:
            LOG.info("Getting namespace %s", namespace)
            ns = self.get_namespace(namespace)
            ns_uid = ns.uid
            LOG.info("Got a namespace %s with uid - %s", namespace, ns_uid)
            return ns
        except ApiException as ex:
            if ex.status != 404:
                raise ex
            LOG.info("Namespace %s doesn't exists. Going to create", namespace)
            try:
                return self.create_namespace(namespace)
            except ApiException as ex:
                if ex.status == 409:
                    if ex.reason == "Conflict":
                        LOG.info("Namespace %s was created in parallel thread", namespace)
                        ns = self.get_namespace(namespace)
                        ns_uid = ns.uid
                        LOG.info("Got a namespace %s with uid - %s", namespace, ns_uid)
                        return ns
                raise ex

    @cachetools_func.ttl_cache(ttl=10)
    def get_namespace(self, namespace: str) -> "Namespace":
        ns = self.api.namespaces.get(name=namespace, namespace=namespace)
        return Namespace(self, ns)

    @cachetools_func.ttl_cache(ttl=10)
    def get_namespaces(
            self,
            hide_system_namespaces: bool = True) -> List["Namespace"]:
        namespaces = self.api.namespaces.list_all()
        if hide_system_namespaces:
            namespaces = [
                ns for ns in namespaces
                if ns.name not in system_namespaces
            ]
        return [Namespace(self, ns) for ns in namespaces]

    def get_mgmt_object(self):
        """Return raw kcm mgmt object for mgmt cluster

        :return:
        """
        return self.api.k0rdent_managements.get(name=settings.KCM_CHART_NAME)

    def get_events(self, namespace, event_prefix=None, sort=True):
        """Return dict, where values contain lists of dicts with events data for the namespace"""
        events = self.api.events.list(namespace=namespace)
        # return self.parse_events(events, event_prefix, sort)
        return utils.parse_events(events, event_prefix, sort)

    def get_events_by_uid(self, namespace, uid, event_prefix=None, sort=True):
        """Return list of events for the specified object"""
        events = self.get_events(namespace, event_prefix, sort)
        filtered_events = []
        for group, values in events.items():
            if values[0]['object_uid'] == uid:
                filtered_events.extend(values)
        return filtered_events

    def get_events_by_uid_str(self, namespace, uid, event_prefix=None,
                              sort=True):
        events = self.get_events_by_uid(namespace, uid, event_prefix, sort)
        messages = []
        for event in events:
            data = event['data']
            messages.append(
                f"{data.last_timestamp or data.event_time}  "
                f"{data.type}  {data.reason}  {data.message}")
        return '\n'.join(messages)

    def get_system_events(self):
        result = {}
        for ns in system_namespaces:
            result.update(self.get_events(namespace=ns, sort=True))
        return result

    @cachetools_func.ttl_cache(ttl=10)
    def get_machines(self, namespace=settings.KCM_NAMESPACE):
        return self.api.capi_machines.list(namespace=namespace)

    @cachetools_func.ttl_cache(ttl=10)
    def get_machine(self, name, namespace=settings.KCM_NAMESPACE):
        return self.api.capi_machines.get(name=name, namespace=namespace)

    def get_capi_cluster(self, name, namespace):
        return self.api.capi_clusters.get(name=name, namespace=namespace)

    def get_capi_clusters(self, namespace):
        return self.api.capi_clusters.list(namespace=namespace)

    def list_all_credentials(self, name_prefix=None):
        return self.api.k0rdent_credentials.list_all(name_prefix=name_prefix)

    def list_all_clusterdeployments(self):
        clustersdeployments = self.api.k0rdent_cluster_deployments.list_all()
        return [ClusterDeployment(self, cluster_deployment) for cluster_deployment in clustersdeployments]

    def list_all_helmreleases(self, name_prefix=None):
        return self.api.helmreleases.list_all(name_prefix)

    def create_secret(self, name, namespace, data, datafield="stringData", s_type='Opaque', labels=None):
        """Create Secret object on Management cluster in the specified Namespace"""
        body = {
            "apiVersion": "v1",
            "kind": "Secret",
            "metadata": {
                "name": name,
                "namespace": namespace,
                "labels": labels or {},
            },
            "type": s_type,
            datafield: data
        }
        return self.api.secrets.create(namespace=namespace, body=body)

    def get_release(self, name):
        """Get release object"""
        return self.api.k0rdent_releases.get(name)


class Namespace(object):
    __manager = None
    __namespace = None

    def __init__(self, manager: Manager, k8s_namespace):
        """
        manager: <Manager> instance
        k8s_namespace: <K8sNamespace> instance
        """
        self.__manager = manager
        self.__namespace = k8s_namespace
        self._helmmanager = None

    @property
    def name(self):
        """Namespace name"""
        return self.__namespace.name

    @property
    def uid(self):
        """Namespace uid"""
        return self.data['metadata']['uid']

    @property
    def data(self):
        """Returns dict of k8s object

        Data contains keys like api_version, kind,

        metadata, spec, status or items
        """
        return self.__namespace.read().to_dict()

    @property
    def helmmgr(self):
        if not self._helmmanager:
            self._helmmanager = self.__manager.get_helmmgr_for_ns(self.name)
        return self._helmmanager

    def delete(self, a_sync=True):
        """Deletes the current namespace"""
        self.__namespace.delete()
        if not a_sync:
            self.wait_for_deletion()

    def exists(self):
        """Verifies the current namespace deletion"""
        all_ns = self.__manager.get_namespaces()
        for n in all_ns:
            if n.name == self.name:
                LOG.info(f"Namespace with name {self.name} found")
                return True
        return False

    def wait_for_deletion(self, timeout=900, interval=15):
        """Wait for the current namespace deletion"""
        waiters.wait(lambda: not self.exists(),
                     timeout=timeout, interval=interval,
                     timeout_msg='Timeout waiting for namespace deletion')
        LOG.info("Namespace has been deleted.")

    def add_labels(self, labels):
        """Add labels to the namespace

        :param labels:
        :return:
        """
        return self.__namespace.add_labels(labels)

    def remove_labels(self, labels):
        """Remove labels of the namespace

        :param labels:
        :return:
        """
        return self.__namespace.remove_labels(labels)

    def create_registry_auth_secret(self, name):
        data = {
            'username': settings.KSI_KCM_CUSTOM_REGISTRY_USERNAME,
            'password': settings.KSI_KCM_CUSTOM_REGISTRY_PASSWORD
        }
        labels = {
            "k0rdent.mirantis.com/component": "kcm"
        }
        LOG.hide(settings.KSI_KCM_CUSTOM_REGISTRY_USERNAME)
        LOG.hide(settings.KSI_KCM_CUSTOM_REGISTRY_PASSWORD)
        return self.__manager.create_secret(name, settings.KCM_NAMESPACE, data, labels=labels)

    def create_registry_cert_secret(self, name):
        if settings.KSI_CUSTOM_REGISTRY_CERT_TEXT:
            cert_data = settings.KSI_CUSTOM_REGISTRY_CERT_TEXT
        elif settings.KSI_CUSTOM_REGISTRY_CERT_PATH:
            with open(settings.KSI_CUSTOM_REGISTRY_CERT_PATH, "rb") as f:
                cert_data = f.read().decode('utf-8')

        data = {
            'ca.crt': cert_data
        }
        labels = {
            "k0rdent.mirantis.com/component": "kcm"
        }
        return self.__manager.create_secret(name, settings.KCM_NAMESPACE, data, labels=labels)

    def create_aws_auth_secret(self, name):
        data = {
            'AccessKeyID': settings.AWS_ACCESS_KEY_ID,
            'SecretAccessKey': settings.AWS_SECRET_ACCESS_KEY
        }
        labels = {
            "k0rdent.mirantis.com/component": "kcm"
        }
        LOG.hide(settings.AWS_ACCESS_KEY_ID)
        LOG.hide(settings.AWS_SECRET_ACCESS_KEY)
        return self.__manager.create_secret(name, settings.KCM_NAMESPACE, data, labels=labels)

    def create_aws_cluster_static_identity(self, name, secret):
        body = {
            "apiVersion": "infrastructure.cluster.x-k8s.io/v1beta2",
            "kind": "AWSClusterStaticIdentity",
            "metadata": {
                "name": name,
                "labels": {
                    "k0rdent.mirantis.com/component": "kcm"
                }
            },
            "spec": {
                "secretRef": secret.name,
                "allowedNamespaces": {
                    "selector": {
                        "matchLabels": {}
                    }
                }
            }
        }
        return self.__manager.api.awsclusterstaticidentities.create(name, body=body)

    def create_azure_auth_secret(self, name):
        data = {
            'clientSecret': settings.AZURE_CLIENT_SECRET
        }
        labels = {
            "k0rdent.mirantis.com/component": "kcm"
        }
        LOG.hide(settings.AZURE_CLIENT_SECRET)
        return self.__manager.create_secret(name, settings.KCM_NAMESPACE, data, labels=labels)

    def create_azure_cluster_identity(self, name, secret):
        body = {
            "apiVersion": "infrastructure.cluster.x-k8s.io/v1beta1",
            "kind": "AzureClusterIdentity",
            "metadata": {
                "name": name,
                "namespace": self.name,
                "labels": {
                    "clusterctl.cluster.x-k8s.io/move-hierarchy": "true",
                    "k0rdent.mirantis.com/component": "kcm"
                }
            },
            "spec": {
                "clientSecret": {
                    'name': secret.name,
                    'namespace': secret.namespace
                },
                'type': 'ServicePrincipal',
                'clientID': settings.AZURE_CLIENT_ID,
                'tenantID': settings.AZURE_TENANT_ID,
                'allowedNamespaces': {
                }
            }
        }
        return self.__manager.api.azureclusteridentities.create(name, namespace=self.name, body=body)

    def create_azure_aks_auth_secret(self, name):
        data = {
            'AZURE_CLIENT_ID': settings.AZURE_CLIENT_ID,
            'AZURE_CLIENT_SECRET': settings.AZURE_CLIENT_SECRET,
            'AZURE_SUBSCRIPTION_ID': settings.AZURE_SUBSCRIPTION_ID,
            'AZURE_TENANT_ID': settings.AZURE_TENANT_ID
        }
        labels = {
            "k0rdent.mirantis.com/component": "kcm"
        }
        LOG.hide(settings.AZURE_CLIENT_ID)
        LOG.hide(settings.AZURE_CLIENT_SECRET)
        return self.__manager.create_secret(name, settings.KCM_NAMESPACE, data, labels=labels)

    def create_vsphere_auth_secret(self, name):
        data = {
            'username': settings.VSPHERE_USER,
            'password': settings.VSPHERE_PASSWORD
        }
        labels = {
            "k0rdent.mirantis.com/component": "kcm"
        }
        LOG.hide(settings.VSPHERE_USER)
        LOG.hide(settings.VSPHERE_PASSWORD)
        return self.__manager.create_secret(name, settings.KCM_NAMESPACE, data, labels=labels)

    def create_vsphere_cluster_identity(self, name, secret):
        body = {
            "apiVersion": "infrastructure.cluster.x-k8s.io/v1beta1",
            "kind": "VSphereClusterIdentity",
            "metadata": {
                "name": name,
                "labels": {
                    "k0rdent.mirantis.com/component": "kcm"
                }
            },
            "spec": {
                "secretName": secret.name,
                "allowedNamespaces": {
                    "selector": {
                        "matchLabels": {}
                    }
                }
            }
        }
        return self.__manager.api.vsphereclusteridentities.create(name, body=body)

    def create_os_auth_secret(self, name):
        data = {
            'clouds.yaml': f"clouds:\n"
                           f"  openstack:\n"
                           f"    auth:\n"
                           f"      auth_url: {settings.OS_AUTH_URL}\n"
                           f"      application_credential_id: {settings.OS_APPLICATION_CREDENTIAL_ID}\n"
                           f"      application_credential_secret: {settings.OS_APPLICATION_CREDENTIAL_SECRET}\n"
                           f"    region_name: {settings.OS_REGION_NAME}\n"
                           f"    interface: {settings.OS_INTERFACE}\n"
                           f"    identity_api_version: {settings.OS_IDENTITY_API_VERSION}\n"
                           f"    auth_type: {settings.OS_AUTH_TYPE}"
        }
        labels = {
            "k0rdent.mirantis.com/component": "kcm"
        }
        LOG.hide(settings.OS_APPLICATION_CREDENTIAL_ID)
        LOG.hide(settings.OS_APPLICATION_CREDENTIAL_SECRET)
        return self.__manager.create_secret(name, settings.KCM_NAMESPACE, data, labels=labels)

    def create_gcp_auth_secret(self, name):
        auth_data = "{\n" \
                    f"  \"type\": \"{settings.GCP_AUTH_TYPE}\",\n" \
                    f"  \"project_id\": \"{settings.GCP_PROJECT_ID}\",\n" \
                    f"  \"private_key_id\": \"{settings.GCP_PRIVATE_KEY_ID}\",\n" \
                    f"  \"private_key\": \"{settings.GCP_PRIVATE_KEY}\",\n" \
                    f"  \"client_email\": \"{settings.GCP_CLIENT_EMAIL}\",\n" \
                    f"  \"client_id\": \"{settings.GCP_CLIENT_ID}\",\n" \
                    f"  \"auth_uri\": \"{settings.GCP_AUTH_URI}\",\n" \
                    f"  \"token_uri\": \"{settings.GCP_TOKEN_URI}\",\n" \
                    f"  \"auth_provider_x509_cert_url\": \"{settings.GCP_AUTH_PROVIDER_X509_CERT_URI}\",\n" \
                    f"  \"client_x509_cert_url\": \"{settings.GCP_CLIENT_X509_CERT_URI}\",\n" \
                    f"  \"universe_domain\": \"{settings.GCP_UNIVERCE_DOMAIN}\"\n" \
                    "}\n"
        str_auth = str(auth_data).encode('ascii')
        b64_auth = base64.b64encode(str_auth).decode('ascii')
        data = {
            'credentials': b64_auth
        }
        labels = {
            "k0rdent.mirantis.com/component": "kcm"
        }
        LOG.hide(b64_auth)
        return self.__manager.create_secret(name, settings.KCM_NAMESPACE, data, labels=labels, datafield='data')

    def create_credential(self, name, cred_object):
        body = {
            "apiVersion": "k0rdent.mirantis.com/v1beta1",
            "kind": "Credential",
            "metadata": {
                "name": name,
                "namespace": self.name
            },
            "spec": {
                "description": "k0rdent system integration tests auth",
                "identityRef": {
                    "apiVersion": cred_object.api_version,
                    "kind": cred_object.kind,
                    "name": cred_object.name,
                    "namespace": self.name
                }
            }
        }
        return self.__manager.api.k0rdent_credentials.create(name, namespace=self.name, body=body)

    def list_credentials(self, namespace, name_prefix):
        return self.__manager.api.k0rdent_credentials.list(namespace=namespace, name_prefix=name_prefix)

    def create_helmrepository(self, name, repo_url, repo_type, repo_interval, repo_username=None, repo_password=None):
        if self.__manager.api.helmrepositories.present(name=name, namespace=self.name):
            LOG.info(f"HelmRepository '{self.name}/{name}' already exists, use existing object")
            return self.__manager.api.helmrepositories.get(name=name, namespace=self.name)

        LOG.info(f"Creating HelmRepository '{self.name}/{name}'")
        body = {
            "apiVersion": "source.toolkit.fluxcd.io/v1",
            "kind": "HelmRepository",
            "metadata": {
                "labels": {
                    "k0rdent.mirantis.com/managed": "true",
                },
                "name": name,
                "namespace": self.name,
            },
            "spec": {
                "interval": repo_interval,
                "provider": "generic",
                "type": repo_type,
                "url": repo_url,
            }
        }
        if repo_username and repo_password:
            secret_name = f"helmrepository-{name}"
            secret_data = {
                'username': repo_username,
                'password': repo_password
            }
            body['spec']['secretRef'] = {'name': secret_name}
            LOG.info(f"Creating Secret '{self.name}/{secret_name}' for the HelmRepository")
            LOG.hide(repo_username)
            LOG.hide(repo_password)
            self.__manager.create_secret(secret_name, self.name, secret_data)

        return self.__manager.api.helmrepositories.create(name, namespace=self.name, body=body)

    def get_helmrelease(self, name):
        return self.__manager.api.helmreleases.get(name=name, namespace=self.name)

    def create_cm(self, name, labels=None, annotations=None, data=None):
        body = {
            "apiVersion": "v1",
            "kind": "ConfigMap",
            "metadata": {
                "name": name,
                "namespace": self.name,
                "labels": labels or {},
                "annotations": annotations or {}
            },
        }
        if data:
            body['data'] = data
        return self.__manager.api.configmaps.create(name, namespace=self.name, body=body)

    def create_sveltos_resource_template_cm(self, cluster_identity):
        name = f"{cluster_identity.name}-resource-template"
        labels = {
            "k0rdent.mirantis.com/component": "kcm"
        }
        annotations = {
            "projectsveltos.io/template": "true"
        }
        data = None

        # TODO(va4st): need proper provider detection
        if cluster_identity.kind == 'AWSClusterStaticIdentity':
            LOG.info('AWS Identity found. Sveltos resource template not required to be '
                     'filled off but needs to be present')
        elif cluster_identity.kind == 'AzureClusterIdentity':
            data = {
                "configmap.yaml": "{{- $cluster := .InfrastructureProvider -}}\n"
                                  "{{- $identity := (getResource \"InfrastructureProviderIdentity\") -}}\n"
                                  "{{- $secret := (getResource \"InfrastructureProviderIdentitySecret\") -}}\n"
                                  "{{- $subnetName := \"\" -}}\n{{- $securityGroupName := \"\" -}}\n"
                                  "{{- $routeTableName := \"\" -}}\n"
                                  "{{- range $cluster.spec.networkSpec.subnets -}}\n"
                                  "  {{- if eq .role \"node\" -}}\n"
                                  "    {{- $subnetName = .name -}}\n"
                                  "    {{- $securityGroupName = .securityGroup.name -}}\n"
                                  "    {{- $routeTableName = .routeTable.name -}}\n"
                                  "    {{- break -}}\n  {{- end -}}\n{{- end -}}\n{{- $cloudConfig := dict\n"
                                  "  \"aadClientId\" $identity.spec.clientID\n"
                                  "  \"aadClientSecret\" (index $secret.data \"clientSecret\" | b64dec)\n"
                                  "  \"cloud\" $cluster.spec.azureEnvironment\n  \"loadBalancerName\" \"\"\n"
                                  "  \"loadBalancerSku\" \"Standard\"\n  \"location\" $cluster.spec.location\n"
                                  "  \"maximumLoadBalancerRuleCount\" 250\n"
                                  "  \"resourceGroup\" $cluster.spec.resourceGroup\n"
                                  "  \"routeTableName\" $routeTableName\n  \"securityGroupName\" $securityGroupName\n"
                                  "  \"securityGroupResourceGroup\" $cluster.spec.networkSpec.vnet.resourceGroup\n"
                                  "  \"subnetName\" $subnetName\n  \"subscriptionId\" $cluster.spec.subscriptionID\n"
                                  "  \"tenantId\" $identity.spec.tenantID\n  \"useInstanceMetadata\" true\n"
                                  "  \"useManagedIdentityExtension\" false\n  \"vmType\" \"vmss\"\n"
                                  "  \"vnetName\" $cluster.spec.networkSpec.vnet.name\n"
                                  "  \"vnetResourceGroup\" $cluster.spec.networkSpec.vnet.resourceGroup\n-}}\n"
                                  "---\napiVersion: v1\nkind: Secret\nmetadata:\n  name: azure-cloud-provider\n"
                                  "  namespace: kube-system\ntype: Opaque\ndata:\n"
                                  "  cloud-config: {{ $cloudConfig | toJson | b64enc }}\n"
            }
        elif cluster_identity.kind == 'VSphereClusterIdentity':
            data = {
                "configmap.yaml": "{{- $cluster := .InfrastructureProvider -}}\n"
                                  "{{- $identity := (getResource \"InfrastructureProviderIdentity\") -}}\n"
                                  "{{- $secret := (getResource \"InfrastructureProviderIdentitySecret\") -}}\n"
                                  "---\n"
                                  "apiVersion: v1\n"
                                  "kind: Secret\n"
                                  "metadata:\n"
                                  "  name: vsphere-cloud-secret\n"
                                  "  namespace: kube-system\n"
                                  "type: Opaque\n"
                                  "data:\n"
                                  "  {{ printf \"%s.username\" $cluster.spec.server }}:"
                                  " {{ index $secret.data \"username\" }}\n"
                                  "  {{ printf \"%s.password\" $cluster.spec.server }}:"
                                  " {{ index $secret.data \"password\" }}\n"
                                  "---\n"
                                  "apiVersion: v1\n"
                                  "kind: Secret\n"
                                  "metadata:\n"
                                  "  name: vcenter-config-secret\n"
                                  "  namespace: kube-system\n"
                                  "type: Opaque\n"
                                  "stringData:\n"
                                  "  csi-vsphere.conf: |\n"
                                  "    [Global]\n"
                                  "    cluster-id = \"{{ $cluster.metadata.name }}\"\n"
                                  "\n"
                                  "    [VirtualCenter \"{{ $cluster.spec.server }}\"]\n"
                                  "    insecure-flag = \"true\"\n"
                                  "    user = \"{{ index $secret.data \"username\" | b64dec }}\"\n"
                                  "    password = \"{{ index $secret.data \"password\" | b64dec }}\"\n"
                                  "    port = \"443\"\n"
                                  f"    datacenters = {settings.VSPHERE_DATACENTER_NAME}\n"
                                  "---\n"
                                  "apiVersion: v1\n"
                                  "kind: ConfigMap\n"
                                  "metadata:\n"
                                  "  name: cloud-config\n"
                                  "  namespace: kube-system\n"
                                  "data:\n"
                                  "  vsphere.conf: |\n"
                                  "    global:\n"
                                  "      insecureFlag: true\n"
                                  "      port: 443\n"
                                  "      secretName: vsphere-cloud-secret\n"
                                  "      secretNamespace: kube-system\n"
                                  "    labels:\n"
                                  "      region: k8s-region\n"
                                  "      zone: k8s-zone\n"
                                  "    vcenter:\n"
                                  "      {{ $cluster.spec.server }}:\n"
                                  "        datacenters:\n"
                                  f"          - {settings.VSPHERE_DATACENTER_NAME}\n"
                                  "        server: {{ $cluster.spec.server }}\n"
            }
        else:
            raise NotImplementedError(f"{cluster_identity.kind} is not supported in KSI now. Please update the code")
        self.create_cm(name=name, labels=labels, annotations=annotations, data=data)

    def create_sveltos_resource_template_cm_using_secret(self, secret, provider):
        name = f"{secret.name}-resource-template"
        labels = {
            "k0rdent.mirantis.com/component": "kcm"
        }
        annotations = {
            "projectsveltos.io/template": "true"
        }
        if provider == 'openstack':
            data = {
                "configmap.yaml": "{{- $cluster := .InfrastructureProvider -}}\n"
                                  "{{- $identity := (getResource \"InfrastructureProviderIdentity\") -}}\n"
                                  "\n"
                                  "{{- $clouds := fromYaml (index $identity \"data\" \"clouds.yaml\" | b64dec) -}}\n"
                                  "{{- if not $clouds }}\n"
                                  "  {{ fail \"failed to decode clouds.yaml\" }}\n"
                                  "{{ end -}}\n"
                                  "\n"
                                  "{{- $openstack := index $clouds \"clouds\" \"openstack\" -}}\n"
                                  "\n"
                                  "{{- if not (hasKey $openstack \"auth\") }}\n"
                                  "  {{ fail \"auth key not found in openstack config\" }}\n"
                                  "{{- end }}\n"
                                  "{{- $auth := index $openstack \"auth\" -}}\n"
                                  "\n"
                                  "{{- $auth_url := index $auth \"auth_url\" -}}\n"
                                  "{{- $app_cred_id := index $auth \"application_credential_id\" -}}\n"
                                  "{{- $app_cred_name := index $auth \"application_credential_name\" -}}\n"
                                  "{{- $app_cred_secret := index $auth \"application_credential_secret\" -}}\n"
                                  "\n"
                                  "{{- $network_id := $cluster.status.externalNetwork.id -}}\n"
                                  "{{- $network_name := $cluster.status.externalNetwork.name -}}\n"
                                  "---\n"
                                  "apiVersion: v1\n"
                                  "kind: Secret\n"
                                  "metadata:\n"
                                  "  name: openstack-cloud-config\n"
                                  "  namespace: kube-system\n"
                                  "type: Opaque\nstringData:\n"
                                  "  cloud.conf: |\n"
                                  "    [Global]\n"
                                  "    auth-url=\"{{ $auth_url }}\"\n"
                                  "\n"
                                  "    {{- if $app_cred_id }}\n"
                                  "    application-credential-id=\"{{ $app_cred_id }}\"\n"
                                  "    {{- end }}\n"
                                  "\n"
                                  "    {{- if $app_cred_name }}\n"
                                  "    application-credential-name=\"{{ $app_cred_name }}\"\n"
                                  "    {{- end }}\n"
                                  "\n"
                                  "    {{- if $app_cred_secret }}\n"
                                  "    application-credential-secret=\"{{ $app_cred_secret }}\"\n"
                                  "    {{- end }}\n"
                                  "\n"
                                  "    {{- if and (not $app_cred_id) (not $app_cred_secret) }}\n"
                                  "    username=\"{{ index $openstack \"username\" }}\"\n"
                                  "    password=\"{{ index $openstack \"password\" }}\"\n"
                                  "    {{- end }}\n"
                                  "    region=\"{{ index $openstack \"region_name\" }}\"\n"
                                  "\n"
                                  "    [LoadBalancer]\n"
                                  "    {{- if $network_id }}\n"
                                  "    floating-network-id=\"{{ $network_id }}\"\n"
                                  "    {{- end }}\n"
                                  "\n"
                                  "    [Networking]\n"
                                  "    {{- if $network_name }}\n"
                                  "    public-network-name=\"{{ $network_name }}\"\n"
                                  "    {{- end }}\n"
            }
        elif provider == 'gcp':
            data = {
                "configmap.yaml": "{{- $secret := (getResource \"InfrastructureProviderIdentity\") -}}\n"
                                  "---\n"
                                  "apiVersion: v1\n"
                                  "kind: Secret\n"
                                  "metadata:\n"
                                  "  name: gcp-cloud-sa\n"
                                  "  namespace: kube-system\n"
                                  "type: Opaque\n"
                                  "data:\n"
                                  "  cloud-sa.json: {{ index $secret \"data\" \"credentials\" }}\n"
            }
        elif provider == 'aks':
            LOG.info('AKS Provider found. Sveltos resource template not required to be '
                     'filled off but needs to be present')
            data = {}
        else:
            raise NotImplementedError(f"{provider} is not supported in KSI now. Please update the code")
        self.create_cm(name=name, labels=labels, annotations=annotations, data=data)

    def get_cluster_config(self, provider, **kwargs):
        if provider == 'aws':
            cluster_config = self.build_aws_ec2_cluster_config(**kwargs)
        elif provider == 'eks':
            cluster_config = self.build_aws_eks_cluster_config(**kwargs)
        elif provider == 'azure':
            cluster_config = self.build_azure_cluster_config(**kwargs)
        elif provider == 'aks':
            cluster_config = self.build_azure_aks_cluster_config(**kwargs)
        elif provider == 'vsphere':
            cluster_config = self.build_vsphere_cluster_config(**kwargs)
        elif provider == 'vsphere-hosted':
            cluster_config = self.build_vsphere_hosted_cluster_config(**kwargs)
        elif provider == 'openstack':
            cluster_config = self.build_os_cluster_config(**kwargs)
        elif provider == 'openstack-hosted':
            cluster_config = self.build_os_hosted_cluster_config(**kwargs)
        elif provider == 'gcp':
            cluster_config = self.build_gcp_cluster_config(**kwargs)
        else:
            raise NotImplementedError(f"Provider {provider} is not supported")
        return cluster_config

    @staticmethod
    def get_regional_config(storageclass, region):
        body = {
            'clusterAnnotations': {
                'k0rdent.mirantis.com/kof-storage-class': storageclass
            },
            'region': region
        }
        return body

    @staticmethod
    def build_aws_ec2_cluster_config(region=settings.AWS_DEFAULT_REGION,
                                     cp_instance_type=settings.AWS_INSTANCE_TYPE_CTL,
                                     w_instatce_type=settings.AWS_INSTANCE_TYPE_WRK,
                                     controlplanenumber=settings.KCM_CLUSTER_CONTROLPLANE_NUMBER,
                                     workersnumber=settings.KCM_CLUSTER_WORKERS_NUMBER,
                                     controlplane_instance_ami=settings.AWS_INSTANCE_AMI_CTL,
                                     worker_instance_ami=settings.AWS_INSTANCE_AMI_WRK,
                                     controlplane_instance_root_size=settings.AWS_INSTANCE_ROOT_SIZE_CTL,
                                     worker_instance_root_size=settings.AWS_INSTANCE_ROOT_SIZE_WRK):

        config = {
            "region": region,
            "controlPlane": {
                "instanceType": cp_instance_type,
            },
            "worker": {
                "instanceType": w_instatce_type,
            },
            "controlPlaneNumber": controlplanenumber,
            "workersNumber": workersnumber
        }
        if controlplane_instance_ami:
            config['controlPlane']['amiID'] = controlplane_instance_ami
        if worker_instance_ami:
            config['worker']['amiID'] = worker_instance_ami
        if controlplane_instance_root_size:
            config['controlPlane']['rootVolumeSize'] = controlplane_instance_root_size
        if worker_instance_root_size:
            config['worker']['rootVolumeSize'] = worker_instance_root_size
        return config

    @staticmethod
    def build_aws_eks_cluster_config(region=settings.AWS_DEFAULT_REGION,
                                     w_instatce_type=settings.AWS_INSTANCE_TYPE_WRK,
                                     workersnumber=settings.KCM_CLUSTER_WORKERS_NUMBER,
                                     worker_instance_ami=settings.AWS_INSTANCE_AMI_WRK,
                                     worker_instance_root_size=settings.AWS_INSTANCE_ROOT_SIZE_WRK,
                                     ssh_key_name=None,
                                     cluster_labels=None):

        if cluster_labels is None:
            cluster_labels = {}
        config = {
            "region": region,
            "worker": {
                "instanceType": w_instatce_type,
            },
            "workersNumber": workersnumber
        }
        if worker_instance_ami:
            config['worker']['amiID'] = worker_instance_ami
        if worker_instance_root_size:
            config['worker']['rootVolumeSize'] = worker_instance_root_size
        if ssh_key_name:
            config['sshKeyName'] = ssh_key_name
        if cluster_labels:
            config['clusterLabels'] = cluster_labels
        return config

    @staticmethod
    def build_azure_cluster_config(locaion=settings.AZURE_CLUSTER_LOCATION,
                                   subscription_id=settings.AZURE_SUBSCRIPTION_ID,
                                   cp_vm_size=settings.AZURE_VM_SIZE_CTL,
                                   wr_vm_size=settings.AZURE_VM_SIZE_WRK,
                                   controlplanenumber=settings.KCM_CLUSTER_CONTROLPLANE_NUMBER,
                                   workersnumber=settings.KCM_CLUSTER_WORKERS_NUMBER):
        config = {
            'location': locaion,
            'subscriptionID': subscription_id,
            'controlPlane': {
                'vmSize': cp_vm_size
            },
            'controlPlaneNumber': controlplanenumber,
            'worker': {
                'vmSize': wr_vm_size
            },
            'workersNumber': workersnumber
        }
        return config

    @staticmethod
    def build_azure_aks_cluster_config(locaion=settings.AZURE_CLUSTER_LOCATION,
                                       cp_vm_size=settings.AZURE_VM_SIZE_CTL,
                                       wr_vm_size=settings.AZURE_VM_SIZE_WRK,
                                       controlplanenumber=settings.KCM_CLUSTER_CONTROLPLANE_NUMBER,
                                       workersnumber=settings.KCM_CLUSTER_WORKERS_NUMBER):
        config = {
            'location': locaion,
            'machinePools': {
                'system': {
                    'vmSize': cp_vm_size,
                    'count': controlplanenumber
                },
                'user': {
                    'vmSize': wr_vm_size,
                    'count': workersnumber
                },
            }
        }
        return config

    @staticmethod
    def build_vsphere_cluster_config(server=settings.VSPHERE_SERVER_ADDR,
                                     dc=settings.VSPHERE_DATACENTER_NAME,
                                     ds=settings.VSPHERE_DATASTORE_PATH,
                                     resourcepool=settings.VSPHERE_RESOURCE_POOL_PATH,
                                     folderpath=settings.VSPHERE_FOLDER_PATH,
                                     cp_endpoint=settings.VSPHERE_CONTROL_PLANE_ENDPOINT_IP,
                                     ssh_user=settings.VSPHERE_SSH_USER,
                                     ssh_pubkey=settings.VSPHERE_SSH_PUBKEY,
                                     rootvolumesize=settings.VSPHERE_ROOT_VOLUME_SIZE,
                                     controlplanenumber=settings.KCM_CLUSTER_CONTROLPLANE_NUMBER,
                                     workersnumber=settings.KCM_CLUSTER_WORKERS_NUMBER,
                                     cpu=settings.VSPHERE_MACHINE_CPU,
                                     ram=settings.VSPHERE_MACHINE_RAM,
                                     vm_template=settings.VSPHERE_VM_TEMPLATE_PATH,
                                     network=settings.VSPHERE_NETWORK_PATH):
        thumbprint = ssl_tools.get_cert_fingetprint(settings.VSPHERE_SERVER_ADDR, 443)
        config = {
            'clusterLabels': {},
            'controlPlaneNumber': controlplanenumber,
            'workersNumber': workersnumber,
            'vsphere': {
                'server': server,
                'thumbprint': thumbprint,
                'datacenter': dc,
                'datastore': ds,
                'resourcePool': resourcepool,
                'folder': folderpath
            },
            'controlPlaneEndpointIP': cp_endpoint,
            'controlPlane': {
                'ssh': {
                    'user': ssh_user,
                    'publicKey': ssh_pubkey
                },
                'rootVolumeSize': rootvolumesize,
                'cpus': cpu,
                'memory': ram,
                'vmTemplate': vm_template,
                'network': network
            },
            'worker': {
                'ssh': {
                    'user': ssh_user,
                    'publicKey': ssh_pubkey
                },
                'rootVolumeSize': rootvolumesize,
                'cpus': cpu,
                'memory': ram,
                'vmTemplate': vm_template,
                'network': network
            }
        }
        return config

    @staticmethod
    def build_vsphere_hosted_cluster_config(server=settings.VSPHERE_SERVER_ADDR,
                                            dc=settings.VSPHERE_DATACENTER_NAME,
                                            ds=settings.VSPHERE_DATASTORE_PATH,
                                            resourcepool=settings.VSPHERE_RESOURCE_POOL_PATH,
                                            folderpath=settings.VSPHERE_FOLDER_PATH,
                                            cp_endpoint=settings.VSPHERE_CONTROL_PLANE_ENDPOINT_IP,
                                            ssh_user=settings.VSPHERE_SSH_USER,
                                            ssh_pubkey=settings.VSPHERE_SSH_PUBKEY,
                                            rootvolumesize=settings.VSPHERE_ROOT_VOLUME_SIZE,
                                            workersnumber=settings.KCM_CLUSTER_WORKERS_NUMBER,
                                            cpu=settings.VSPHERE_MACHINE_CPU,
                                            ram=settings.VSPHERE_MACHINE_RAM,
                                            vm_template=settings.VSPHERE_VM_TEMPLATE_PATH,
                                            network=settings.VSPHERE_NETWORK_PATH):
        thumbprint = ssl_tools.get_cert_fingetprint(settings.VSPHERE_SERVER_ADDR, 443)
        config = {
            'clusterLabels': {},
            'workersNumber': workersnumber,
            'vsphere': {
                'server': server,
                'thumbprint': thumbprint,
                'datacenter': dc,
                'datastore': ds,
                'resourcePool': resourcepool,
                'folder': folderpath
            },
            'controlPlaneEndpointIP': cp_endpoint,
            'ssh': {
                'user': ssh_user,
                'publicKey': ssh_pubkey
            },
            'rootVolumeSize': rootvolumesize,
            'cpus': cpu,
            'memory': ram,
            'vmTemplate': vm_template,
            'network': network,
            'k0smotron': {
                'service': {
                    'annotations': {
                        'kube-vip.io/loadbalancerIPs': cp_endpoint
                    }
                }
            }
        }
        return config

    @staticmethod
    def build_os_cluster_config(os_auth_secret,
                                controlplanenumber=settings.KCM_CLUSTER_CONTROLPLANE_NUMBER,
                                workersnumber=settings.KCM_CLUSTER_WORKERS_NUMBER,
                                ctl_flavor=settings.OS_CTL_MACHINE_FLAVOR,
                                wrk_flavor=settings.OS_WRK_MACHINE_FLAVOR,
                                image=settings.OS_INSTANCE_IMAGE_NAME,
                                security_group=settings.KSI_OS_SECURITY_GROUP,
                                ssh_keypair_name=settings.KSI_OS_KEYPAIR_NAME,
                                cluster_labels=None,
                                cluster_annotations=None):
        config = {
            "controlPlaneNumber": controlplanenumber,
            "workersNumber": workersnumber,
            "controlPlane": {
                "flavor": ctl_flavor,
                "image": {
                    "filter": {
                        "name": image
                    }
                }
            },
            "worker": {
                "flavor": wrk_flavor,
                "image": {
                    "filter": {
                        "name": image
                    }
                }
            },
            "externalNetwork": {
                "filter": {
                    "name": settings.OS_INTERFACE
                }
            },
            "authURL": settings.OS_AUTH_URL,
            "identityRef": {
                "name": os_auth_secret.name,
                "cloudName": "openstack",
                "region": settings.OS_REGION_NAME
            }
        }
        if cluster_labels:
            config['clusterLabels'] = cluster_labels
        if cluster_annotations:
            config['clusterAnnotations'] = cluster_annotations
        if security_group:
            sg = [
                {
                    'filter': {
                        'name': security_group
                    }
                }
            ]
            config['controlPlane']['securityGroups'] = sg
            config['worker']['securityGroups'] = sg
            config['managedSecurityGroups'] = None

        if ssh_keypair_name:
            config['controlPlane']['sshKeyName'] = ssh_keypair_name
            config['worker']['sshKeyName'] = ssh_keypair_name
        return config

    @staticmethod
    def build_os_hosted_cluster_config(os_auth_secret,
                                       network_name,
                                       router_name,
                                       subnet_name,
                                       workersnumber=settings.KCM_CLUSTER_WORKERS_NUMBER,
                                       wrk_flavor=settings.OS_WRK_MACHINE_FLAVOR,
                                       image=settings.OS_INSTANCE_IMAGE_NAME,
                                       security_group=settings.KSI_OS_SECURITY_GROUP,
                                       ssh_keypair_name=settings.KSI_OS_KEYPAIR_NAME,
                                       cluster_labels=None,
                                       cluster_annotations=None):
        config = {
            "workersNumber": workersnumber,
            "flavor": wrk_flavor,
            "image": {
                "filter": {
                    "name": image
                }
            },
            "externalNetwork": {
                "filter": {
                    "name": settings.OS_INTERFACE
                }
            },
            "identityRef": {
                "name": os_auth_secret.name,
                "cloudName": "openstack",
                "region": settings.OS_REGION_NAME
            },
            "network": {
                "filter": {
                    "name": network_name
                }
            },
            "router": {
                "filter": {
                    "name": router_name
                }
            },
            "subnets": [
                {
                  "filter": {
                    "name": subnet_name
                  }
                }
            ],
            "ports": [
                {
                    "network": {
                        "filter": {
                            "name": network_name
                        }
                    }
                }
            ]
        }
        if cluster_labels:
            config['clusterLabels'] = cluster_labels
        if cluster_annotations:
            config['clusterAnnotations'] = cluster_annotations
        if security_group:
            sg = [
                {
                    'filter': {
                        'name': security_group
                    }
                }
            ]
            config['securityGroups'] = sg
            config['managedSecurityGroups'] = None

        if ssh_keypair_name:
            config['sshKeyName'] = ssh_keypair_name
        return config

    @staticmethod
    def build_gcp_cluster_config(project=settings.GCP_PROJECT_ID,
                                 region=settings.GCP_REGION,
                                 network=settings.GCP_NETWORK_NAME,
                                 cp_flavor=settings.GCP_CP_MACHINE_TYPE,
                                 worker_flavor=settings.GCP_WORKER_MACHINE_TYPE,
                                 cp_image=settings.GCP_CP_IMAGE,
                                 wrk_image=settings.GCP_WRK_IMAGE,
                                 cp_public_ip=True,
                                 wrk_public_ip=True):
        config = {
            'project': project,
            'region': region,
            'network': {
                'name': network
            },
            'controlPlane': {
                'instanceType': cp_flavor,
                'image': cp_image,
                'publicIP': cp_public_ip
            },
            'worker': {
                'instanceType': worker_flavor,
                'image': wrk_image,
                'publicIP': wrk_public_ip
            }
        }
        return config

    @staticmethod
    def build_service_templates_list(tmpl_list):
        svc_list = []
        for template in tmpl_list:
            tmpl_object = template.read()
            svc = {
                'name': tmpl_object.spec['helm']['chartSpec']['chart'],
                'namespace': tmpl_object.spec['helm']['chartSpec']['chart'],
                'template': tmpl_object.metadata.name
            }
            svc_list.append(svc)
        return svc_list

    def create_cluster_deployment(self,
                                  name,
                                  template,
                                  credential,
                                  config,
                                  services=None,
                                  labels=None,
                                  annotations=None):
        if settings.CUSTOM_CLUSTERDEPLOYMENT_PATH:
            body = self.get_cluster_deployment_from_file(
                settings.CUSTOM_CLUSTERDEPLOYMENT_PATH)
            name = body['metadata']['name']
        else:
            body = {
                "apiVersion": "k0rdent.mirantis.com/v1beta1",
                "kind": "ClusterDeployment",
                "metadata": {
                    "name": name,
                    "namespace": self.name
                },
                "spec": {},
            }

        if annotations:
            if body['metadata'].get('annotations', None):
                exists = body['metadata']['annotations']
                utils.merge_dicts(exists, annotations)
                body['metadata']['annotations'] = exists
            else:
                body['metadata']['annotations'] = annotations

        body_generated_spec = {
            "template": template,
            "credential": credential.name,
            "config": config
        }

        body["spec"] = utils.merge_dicts(body["spec"], body_generated_spec)

        if services:
            body["spec"].setdefault("serviceSpec", {}).setdefault('services', []).extend(services)

        if labels:
            body['metadata']['labels'] = labels

        LOG.info("\n" + yaml.dump(body))
        cluster_deployment = self.__manager.api.k0rdent_cluster_deployments.create(
            name=name, namespace=self.name, body=body)
        return ClusterDeployment(self.__manager, cluster_deployment)

    def create_cluster_deployment_raw(self, body=None):
        name = body['metadata']['name']
        cluster_deployment = self.__manager.api.k0rdent_cluster_deployments.create(
            name=name, namespace=self.name, body=body)
        return ClusterDeployment(self.__manager, cluster_deployment)

    def get_cluster_deployment_from_file(self, file_path):
        """Read ClusterDeployment from <file_path> and render it as a Jinja2 template

        For Jinja2 template, available the function {{ create_configmap_from_file(<cm_file_path>) }}
        which creates a ConfigMap object from the specified file and returns it's name
        into ClusterDeployment template.
        """

        def create_configmap_from_file(cm_file_path):
            # Looking up for a configmap template relatively to clusterdeployment file_path
            base_dir = os.path.dirname(file_path)
            combined_path = os.path.join(base_dir, cm_file_path)
            full_path = os.path.abspath(combined_path)

            LOG.info(f"Creating ConfigMap from file '{cm_file_path}' ...")

            cm_template = template_utils.render_template(file_path=full_path)
            cm_body = yaml.load(cm_template, Loader=yaml.SafeLoader)
            cm_name = cm_body['metadata']['name']
            cm_body['metadata']['namespace'] = self.name
            cm_configmap = self.__manager.api.configmaps.create(cm_name, namespace=self.name, body=cm_body)
            return cm_configmap.name

        options = {
            'create_configmap_from_file': create_configmap_from_file,
        }
        LOG.info(f"Creating ClusterDeployment from file '{file_path}' ...")
        template = template_utils.render_template(
            file_path=file_path,
            options=options
        )
        body = yaml.load(template, Loader=yaml.SafeLoader)
        return body

    def get_cluster_deployment(self, name):
        cluster_deployment = self.__manager.api.k0rdent_cluster_deployments.get(
            name=name, namespace=self.name)
        return ClusterDeployment(self.__manager, cluster_deployment)

    def list_clusterdeployments(self):
        clustersdeployments = self.__manager.api.k0rdent_cluster_deployments.list(namespace=self.name)
        return [ClusterDeployment(self.__manager, cluster_deployment) for cluster_deployment in clustersdeployments]

    def get_service_templates(self, name_prefix=None):
        return self.__manager.api.k0rdent_service_templates.list(namespace=self.name, name_prefix=name_prefix)

    def get_cluster_templates(self, name_prefix=None):
        return self.__manager.api.k0rdent_cluster_templates.list(namespace=self.name, name_prefix=name_prefix)

    def get_latest_template_by_prefix(self, prefix, strategy='name'):
        filtered_tmpl_list = self.get_cluster_templates(prefix)
        if strategy == 'name':
            filtered_tmpl_list.sort(key=lambda t: version.Version(t.name.replace(f"{prefix}-", '').replace('-', '.')))
        elif strategy == 'chart':
            filtered_tmpl_list.sort(key=lambda t: version.Version(t.chart_version))
        else:
            LOG.error(f'Unknown strategy for getting latest template: {strategy}')
            raise RuntimeError

        return filtered_tmpl_list[-1].name

    def get_cluster_template_by_name(self, name):
        return self.__manager.api.k0rdent_cluster_templates.get(name, namespace=self.name)

    def get_service_template_by_name(self, name):
        return self.__manager.api.k0rdent_service_templates.get(name, namespace=self.name)

    def get_baremetalhosts(self):
        return self.__manager.api.metal3_baremetalhosts.list_all()

    def check_template_valid(self, template):
        template_data = template.data
        template_status = template_data.get('status') or {}
        error = template_status.get('validationError')
        if error:
            LOG.warning(f"{template.kind} '{template.namespace}/{template.name}' validation error: {error}")
        return template.is_valid

    def create_cluster_template(self, body):
        name = body['metadata']['name']
        return self.__manager.api.k0rdent_cluster_templates.create(
            name=name, namespace=self.name, body=body)

    def wait_cluster_template(self, cluster_template, timeout=1200) -> None:
        LOG.info(f"Waiting until ClusterTemplate '{self.name}/{cluster_template.name}' becomes valid ...")
        waiters.wait(lambda: self.check_template_valid(cluster_template),
                     timeout=timeout, interval=30)
        LOG.info(f"ClusterTemplate '{self.name}/{cluster_template.name}' is valid")
        return

    def create_cluster_template_from_file(self, file_path, options=None):
        LOG.info(f"Creating ClusterTemplate from file '{file_path}' ...")
        template = template_utils.render_template(
            file_path=file_path,
            options=options
        )
        body = yaml.load(template, Loader=yaml.SafeLoader)
        cluster_template = self.create_cluster_template(body=body)
        self.wait_cluster_template(cluster_template)
        return cluster_template

    def create_service_template_from_file(self, file_path, options=None):
        LOG.info(f"Creating ServiceTemplate from file '{file_path}' ...")
        template = template_utils.render_template(
            file_path=file_path,
            options=options
        )
        body = yaml.load(template, Loader=yaml.SafeLoader)
        name = body['metadata']['name']
        service_template = self.create_servicetemplate(name, body)
        return service_template

    def create_servicetemplate(self, name, body):
        service_template = self.__manager.api.k0rdent_service_templates.create(
            name=name, namespace=self.name, body=body)
        LOG.info(f"Waiting until ServiceTemplate '{self.name}/{service_template.name}' becomes valid ...")
        waiters.wait(lambda: self.check_template_valid(service_template),
                     timeout=1200, interval=30)
        LOG.info(f"ServiceTemplate '{self.name}/{service_template.name}' is valid")
        return service_template

    def create_configmap_from_values(self, name, file_path, options=None):
        LOG.info(f"Creating ConfigMap from file '{file_path}' ...")
        values = template_utils.render_template(
            file_path=file_path,
            options=options
        )
        data = {'values.yaml': values}
        configmap = self.create_cm(name=name, data=data)
        return configmap

    def wait_baremetalhosts_statuses(self, retries=10, interval=60,
                                     wait_status='provisioned',
                                     wait_nodes_msg=None,
                                     bmh_names=None,
                                     check_error=True):
        """bmh_names: list of bmh names
         ['cz7987-child-controller-1', 'cz7842-child-worker-1']
         check_error: raise, if any of bmh in error state.
         wait_status: str with status, or set of strings.
         autotime: guess retries count dynamically
        """
        # convert, to make it bw-compatible
        if isinstance(wait_status, str):
            wait_status = {wait_status}

        bmhs = self.get_baremetalhosts()

        if bmh_names:
            bmhs = [bmh for bmh in bmhs if bmh.name in bmh_names]
            # check that we found all requested nodes
            found_bmh_names = [bmh.name for bmh in bmhs]
            if len(found_bmh_names) != len(bmh_names):
                _msg = ('Not all nodes found! requested:'
                        f'{bmh_names}\n'
                        f'Found only:{found_bmh_names}')
                pytest.fail(_msg)
        # old stub. let's leave it as is, even if currently not required in k0rdent
        if 'ready' in wait_status and len(wait_status) == 1:
            wait_status = {'preparing', 'ready', 'available'}
            LOG.warning(f"Switching  to wait multiply statuses "
                        f"as allowed: {wait_status}")

        def get_statuses(bmhs, wait_nodes_msg):
            error_count_threshold = 3
            bm_statuses = {}
            if not wait_nodes_msg:
                wait_nodes_msg = "BMH {name} has <{status}> provisioning status"
            for bm in bmhs:
                bm_data = bm.read()
                name = bm_data.metadata.name
                if not bm_data.status:
                    LOG.info(f"BMH {name} status isn't ready yet")
                    bm_statuses[name] = f"BMH {name} status isn't ready yet"
                    continue
                pstatus = bm_data.status.get('provisioning').get('state', 'Nil')
                LOG.info(wait_nodes_msg.format(name=name, status=pstatus))
                bm_statuses[name] = pstatus
                if check_error:
                    error_count = int(bm_data.status.get('errorCount', 0))
                    if error_count >= error_count_threshold:
                        error_msg = bm_data.status.get('errorMessage')
                        raise Exception(
                            f"Aborting due to excessive error count "
                            f"({error_count} >= {error_count_threshold}) "
                            f"for BMH {name}, errorMessage: {error_msg}")
            return bm_statuses

        total_time = retries * interval
        LOG.info(f"Wait {total_time} seconds until BMH(s):{[bmh.name for bmh in bmhs]}\n will be in the "
                 f"status(es) <{wait_status}>")
        try:
            waiters.wait(
                lambda: set(
                    get_statuses(bmhs, wait_nodes_msg).values()).issubset(wait_status),
                timeout=total_time, interval=interval)
        except TimeoutError as e:
            # TODO(alex-kh) need to adopt helper self.print_events()
            LOG.error(f"Not all baremetalhosts are in {wait_status} "
                      f"status(es)")
            raise e

    def create_external_dns_auth_secret(self, name):
        """Create external-dns auth config

        Will reuse same credentials as for deployment.

        :return:
        """
        conf = f"""
[default]
aws_access_key_id = {settings.AWS_ACCESS_KEY_ID}
aws_secret_access_key = {settings.AWS_SECRET_ACCESS_KEY}
        """
        str_auth = str(conf).encode('ascii')
        b64_auth = base64.b64encode(str_auth).decode('ascii')
        labels = {
            "k0rdent.mirantis.com/component": "kcm"
        }
        data = {"external-dns-aws-credentials": b64_auth}
        LOG.hide(settings.AWS_ACCESS_KEY_ID)
        LOG.hide(settings.AWS_SECRET_ACCESS_KEY)
        return self.__manager.create_secret(name, self.name, data,
                                            datafield='data', labels=labels)


class ClusterDeployment(object):
    def __init__(self, manager: Manager, clusterdeployment: K0rdentClusterDeployment):
        """
        manager: <Manager> instance
        cluster_deployment: <K0rdentClusterDeployment> instance
        """
        self.__manager = manager
        self.__clusterdeployment = clusterdeployment
        self.__spec = None
        self.__k8sclient = None
        self.__expected_pods = None
        self.__cluster_type = None
        self.__metadata = None
        self.__clusterobject = None
        self.__provider = None
        self.check = ClusterDeploymentCheckManager(self)

    def __repr__(self):
        return f"<{self.__class__.__name__} name='{self.name}' namespace='{self.namespace}'>"

    @property
    def _manager(self):
        return self.__manager

    @property
    def name(self):
        """Cluster name"""
        return self.__clusterdeployment.name

    @property
    def namespace(self):
        """Cluster namespace"""
        return self.__clusterdeployment.namespace

    @property
    def metadata(self):
        """Cached clusterdeployment metadata"""
        if self.__metadata is None:
            self.__metadata = self.data['metadata']
        return self.__metadata

    @property
    def data(self):
        """Returns dict of k8s object

        Data contains keys like api_version, kind,

        metadata, spec, status or items
        """
        return self.__clusterdeployment.read().to_dict()

    @property
    def template_name(self):
        """Cluster template name used for deployment"""

        return self.__clusterdeployment.read().spec['template']

    @property
    def expected_pods_template_path(self):
        return settings.EXPECTED_PODS_TEMPLATES_DIR + 'cluster/' + self.template_name + '.yaml'

    @property
    def clusterobject(self):
        if self.__clusterobject is None:
            self.__clusterobject = self.__manager.get_capi_cluster(name=self.name, namespace=self.namespace)
        return self.__clusterobject

    @property
    def provider(self):
        if self.__provider is None:
            self.__provider = utils.Provider.get_provider_by_infrakind(self.clusterobject.infrakind)
        return self.__provider

    @property
    def spec(self):
        """Cached machine spec"""
        if self.__spec is None:
            self.__spec = self.data['spec']
        return self.__spec

    def patch(self, *args, **kwargs):
        self.__clusterdeployment.patch(*args, **kwargs)

    def replace(self, *args, **kwargs):
        self.__clusterdeployment.replace(*args, **kwargs)

    def delete(self):
        """Deletes the current clusterdeployment"""
        self.__clusterdeployment.delete()

    @property
    def k8sclient(self) -> K8sCluster:
        """Cached corresponding clusterdeployment k8sclient"""
        if self.__k8sclient is None:
            _, kubeconfig = self.get_kubeconfig_from_secret()
            config_data = yaml.load(kubeconfig, Loader=yaml.SafeLoader)
            self.__k8sclient = K8sCluster(config_data=config_data)
        return self.__k8sclient

    @property
    def service_status(self):
        status = self.data.get('status', {}).get('services', [])
        if status:
            for st in status:
                if st.get('clusterName', '') == self.name:
                    return st
        else:
            return {}

    @property
    def expected_pods(self):
        if not self.__expected_pods:
            self._refresh_expected_objects()
        return self.__expected_pods

    def present(self):
        ns = self.__manager.get_namespace(self.namespace)
        try:
            all_clusters = ns.list_clusterdeployments()
        except ApiException as e:
            if e.status == 404:
                return False
            raise e
        for c in all_clusters:
            if c.name == self.name:
                LOG.info(f"Clusterdeployment object '{self.namespace}/{self.name}' found in kcm cluster")
                return True
        LOG.info(f"Clusterdeployment object '{self.namespace}/{self.name}' not found in kcm cluster")
        return False

    def is_default_storageclass_present(self):
        """Check if default storageclass is present in the cluster"""
        scs = self.k8sclient.storageclass.list_all()
        return any([sc.name for sc in scs if sc.is_default])

    def get_storageclass_by_name(self, name):
        return self.k8sclient.storageclass.get(name)

    def _refresh_expected_objects(self):
        """Force update expected objects after cluster scale"""
        self.__expected_pods = self.get_expected_objects()

    def get_kubeconfig_from_secret(self, secret_name=''):
        """Get kubeconfig for the cluster from secret

        :param secret_name: str, (optional) use the specified secret
                            name instead of generated from cluster_name
            Returns tuple with kubeconfig name and content in yaml format
        """
        client = self._manager.api
        secret_name = secret_name or "{}-kubeconfig".format(self.name)
        LOG.debug("Fetching kubeconfig of {} cluster deployment".format(self.name))
        assert client.secrets.present(secret_name, namespace=self.namespace), (
            f"Secret {self.namespace}/{secret_name} not found. May be not populated yet.")
        kubeconfig_secret = client.secrets.get(
            secret_name, namespace=self.namespace)
        kubeconfig = kubeconfig_secret.read()
        assert kubeconfig.data, f"Secret {self.namespace}/{secret_name} is empty"
        kubeconfig = kubeconfig.data.get('value')
        assert kubeconfig, f"Secret {self.namespace}/{secret_name} doesn't contain kubeconfig"
        kubeconfig = base64.b64decode(kubeconfig).decode("utf-8")
        name = secret_name
        content = kubeconfig
        return name, content

    @collect_cluster_readiness
    def get_conditions_status(self, expected_fails=None, verbose=False, mode='cluster'):
        expected_fails = expected_fails or {}
        result = {
            'ready': [],
            'not_ready': [],
            'skipped': [],
        }
        data_status = self.data.get('status') or {}
        conditions = []
        if mode == 'services':
            svcs_cfg = data_status.get('services', [])
            LOG.info(f"Checking 'service' conditions of ClusterDeployment {self.namespace}/{self.name}")
            if svcs_cfg:
                for svc in svcs_cfg:
                    if svc.get('clusterName', '') == self.name:
                        conditions = svc.get('conditions', [])
                        break
            if self.get_deployed_services() and not conditions:
                result['not_ready'].append("NO CONDITIONS IN SERVICES STATUS")
        else:
            conditions = data_status.get('conditions', [])
            LOG.info(f"Checking 'cluster' conditions of ClusterDeployment {self.namespace}/{self.name}")
            if not conditions:
                result['not_ready'].append("NO CONDITIONS IN CLUSTER DEPLOYMENT STATUS")

        conditions_data = []
        for condition in conditions:
            if condition['status'] in ['True', 'False']:
                ready = eval(condition['status'])
            else:
                ready = False
            condition_type = condition.get('type')
            condition_message = condition.get('message')
            if (not ready and condition_type in expected_fails.keys() and
                    expected_fails[condition_type] in condition_message):
                result['skipped'].append(condition_type)
            elif ready:
                result['ready'].append(condition_type)
            else:
                result['not_ready'].append(condition_type)
            conditions_data.append({
                'type': condition_type,
                'status': condition['status'],
                'message': condition_message,
            })
        if verbose:
            # message = (f"{self.data['kind']} {self.namespace}/"
            #            f"{self.data['metadata']['name']}: {result}")
            # LOG.info(message)

            headers = ["Type", "Status", "Message"]
            status_data = [[data["type"],
                            data["status"],
                            data["message"]]
                           for data in conditions_data]
            # Show Machines status and not ready conditions
            status_msg = tabulate(status_data, tablefmt="presto", headers=headers)
            LOG.info(f"\n{status_msg}\n")

        return result

    def are_conditions_ready(self, expected_fails=None, verbose=False, mode='cluster'):
        """
        Check if all Cluster Deployment conditions in ready status
        :param mode: Lookup cluster condintions or services conditions
        :param verbose: Verbose or not
        :param expected_fails: None or dict like the following
            {<condition.type>: <expected message pattern>, ...}
            If expected message pattern is empty, then the whole
            condition is ignored. If it is not empty - then
            this pattern is matched to the message from the cluster
            to check if the message is expected or not.
        :rtype bool: bool
        """
        result = self.get_conditions_status(expected_fails, verbose, mode=mode)
        if result['not_ready']:
            return False
        else:
            return True

    def apply_service_templates(self, svc_list):
        """
        Apply changes to service part of cluster deployment
        :param svc_list:
        :return:
        """
        services = self.data['spec'].get('serviceSpec', {}).get('services', [])
        if services:
            LOG.info(f"Found existing services:\n {services}")
            svc_list += services
        body = {
            'spec': {
                'serviceSpec': {
                    'services': svc_list,
                }
            }
        }

        self.patch(body)

    def get_service_condition(self, service_type):
        """
        Get Helm manager condition of cluster deployment. Can indicate what happening with service part of the cluster
        :return:
        """
        if self.service_status:
            for condition in self.service_status.get('conditions', []):
                if condition['type'] == service_type:
                    return condition
        else:
            return None

    def _get_machine_class(self) -> MachineProviders:
        # TODO(va4st): Support other providers
        provider_machines = {
            'aws': AwsProviderMachine,
            'azure': AzureProviderMachine,
            'eks': EKSProviderMachine,
            'gcp': GCPProviderMachine,
            'metal3': Metal3Machines,
            'openstack': OpenStackProviderMachine,
            'vsphere': VSphereProviderMachine,
        }
        provider = self.provider
        adapter_class = provider_machines[provider.provider_name]
        return adapter_class

    def _get_machines(self, raise_if_empty=True) -> List[MachineProviders]:
        """Return list of Machine objects which are belong to the cluster"""
        # collect raw data of all machines in namespace
        machines = self.__manager.api.capi_machines.list(
            namespace=self.namespace)
        # filter out machines not from current cluster
        cl_machines = [x for x in machines if x.cluster_name == self.name]

        # assuming all machines in cluster has the same provider
        if cl_machines:
            # get MachineClass based on the 1st machine in list
            MachineClass = self._get_machine_class()
        elif raise_if_empty:
            raise Exception(f"No Machines have been found for the cluster deployment {self.namespace}/{self.name}")
        else:
            return []
        cl_machine_names = [x.name for x in cl_machines]

        # collect machines objects
        machines_obj = [
            x for x in self.__manager.get_machines(namespace=self.namespace)
            if x.name in cl_machine_names
        ]

        cluster_machines = []
        for machine in machines_obj:
            cluster_machines.append(
                MachineClass(self.__manager, self, machine))
        return cluster_machines

    def get_machines(self, machine_type=None, machine_phase=None, k8s_labels=None, raise_if_empty=True):
        """Return list of Machine objects and set bastion_ip"""
        cluster_machines = self._get_machines(raise_if_empty=raise_if_empty)
        if machine_type is not None:
            cluster_machines = [m for m in cluster_machines
                                if m.is_machine_type(machine_type)]
        if machine_phase is not None:
            cluster_machines = [m for m in cluster_machines
                                if m.machine_phase == machine_phase]
        if k8s_labels is not None:
            cluster_machines = [m for m in cluster_machines
                                if m.has_k8s_labels(k8s_labels)]
        return cluster_machines

    def _render_expected_pods_template(self):
        """Render the template with expected pods for the cluster template"""

        if self.provider == utils.Provider.aks:
            num_ctl_nodes = self.clusterobject.infracluster.systempool.replicas
            num_worker_nodes = self.clusterobject.infracluster.userpool.replicas
            num_nodes = num_ctl_nodes + num_worker_nodes
        else:
            all_machines = self.get_machines()
            num_nodes = len(all_machines)

            ctl_nodes = [x.data['metadata']['name'] for x in all_machines
                         if x.is_machine_type('control')]
            num_ctl_nodes = len(ctl_nodes)

            num_worker_nodes = len(
                [x for x in all_machines
                 if x.is_machine_type('worker')])

        # NOTE (va4st): It's default autoscale behavior for CoreDNS.
        # The formula is pod_num=nodes/16. For 1 node deployments (like our hosted) - pod_num=1 due anti-affinity
        # For 2 and more nodes (up to 16) - pod_num=2 due HA.
        # For 16+ nodes - pod_num grows linearly and 3 coredns pods will be at 48 nodes, that actually not going to be
        # tested with ksi :)
        coredns_factor = 1 if num_nodes < 2 else 2

        options = {
            'num_ctl_nodes': num_ctl_nodes,
            'num_worker_nodes': num_worker_nodes,
            'num_nodes': num_nodes,
            'coredns_factor': coredns_factor,
        }

        templates = template_utils.render_template(self.expected_pods_template_path, options)

        json_body = yaml.load(templates, Loader=yaml.SafeLoader)
        return json_body

    def get_service_templates(self):
        """Returns all services defined in service spec for current cluster
          (ingress-nginx, dex, etc). Listing by template name since different versions
          can significantly differ by pods
        """
        release_names = []
        deployed_services = self.get_deployed_services()
        if deployed_services:
            release_names = [x['template'] for x in deployed_services]

        LOG.debug(f"Deployed services: {deployed_services}")

        return release_names

    def get_deployed_services(self):
        deployed_services = self.spec['serviceSpec'].get('services', [])
        return deployed_services

    def get_expected_namespaces(self):
        if os.path.isfile(self.expected_pods_template_path):
            return self.expected_pods.keys()
        else:
            LOG.info(f"Expected pods not found for ClusterTemplate {self.expected_pods_template_path}, "
                     f"get all namespaces from the cluster")
            return [ns.name for ns in self.k8sclient.namespaces.list_all()]

    def get_expected_objects(self, exclude_pods=None):
        """Returns list of expected pods (and their quantity for the cluster)
           plus all docker objects
        """
        # TODO(va4st): So far we have here only one type of cluster. Later, when we will have in ksi managed clusters
        # TODO:        with kcm installed (for hosted cp) - it should be determined dynamically
        cluster_type = 'managed'

        json_body = self._render_expected_pods_template()
        all_pods = {}

        if exclude_pods:
            LOG.warning("These pods will be excluded from "
                        "expected pod list {}".format(exclude_pods))
        else:
            exclude_pods = []
        for ns in json_body[cluster_type]['namespaces']:
            pods = json_body[cluster_type]['namespaces'][ns]
            pods = {k: v for k, v in pods.items() if k not in exclude_pods}
            if ns not in all_pods:
                all_pods[ns] = pods
            else:
                all_pods[ns].update(pods)

        for component in self.get_service_templates():
            if component in json_body[cluster_type]['components']:
                for ns in json_body[cluster_type]['components'][component]:
                    pods = json_body[cluster_type]['components'][component][ns]
                    pods = {k: v for k, v in pods.items()
                            if k not in exclude_pods}
                    if ns not in all_pods:
                        all_pods[ns] = pods
                    else:
                        all_pods[ns].update(pods)
            else:
                LOG.error("Component {} was not found in "
                          "expected pod templates".format(component))

        LOG.info(json.dumps(all_pods, indent=4))
        return all_pods
