import tempfile

import exec_helpers
import yaml

import os
import weakref
from exec_helpers import CalledProcessError
from si_tests import logger
from si_tests.utils import utils
from si_tests import settings

LOG = logger.logger


class Helm3Manager(object):
    binary = utils.get_binary_path("helm")

    def __init__(self, kubeconfig, auth_data=None, namespace=None):
        self.namespace = namespace

        self._tmp_kubeconfig_path = None
        self._kubeconfig_finalizer = None

        if kubeconfig:
            if os.path.isfile(kubeconfig):
                self.kubeconfig = kubeconfig
            else:
                raise FileNotFoundError(f"Helm3Manager: kubeconfig file {kubeconfig} not found")
        elif auth_data:
            with tempfile.NamedTemporaryFile(mode="w", suffix=".kubeconfig", delete=False) as tmp:
                tmp.write(str(auth_data))
                tmp.flush()
                os.fsync(tmp.fileno())
                self._tmp_kubeconfig_path = tmp.name
                self.kubeconfig = self._tmp_kubeconfig_path
                LOG.debug(f"Helm3Manager: kubeconfig written to temporary file {self.kubeconfig}")

            self._kubeconfig_finalizer = weakref.finalize(
                self, self._cleanup_kubeconfig, self._tmp_kubeconfig_path
            )
        else:
            raise RuntimeError('Neither kubeconfig nor auth_data provided!')

    def _cleanup_kubeconfig(self, path):
        if path and os.path.exists(path):
            try:
                os.remove(path)
                self.kubeconfig = None
                LOG.debug(f"Helm3Manager: removed temporary kubeconfig {path}")
            except FileNotFoundError:
                pass

    def run_cmd(self, cmd, cwd=None, no_namespace=False):
        cmd[:0] = ["--kubeconfig", self.kubeconfig]
        # TODO(alexz-kh) there is still possibility to pass something like '--namespace 123' and destroy everything :(
        # require for operations, w\o namespace, or when requested for all-namespaces, or set
        ns_related_flags = ['-A', '--all-namespaces', '--namespace', '-n']
        if not no_namespace and not any([flag in cmd for flag in ns_related_flags]):
            cmd[:0] = ["--namespace", self.namespace]
        cmd = " ".join([self.binary, *cmd])
        res = exec_helpers.Subprocess().execute(cmd, cwd=cwd)
        if res.exit_code:
            raise CalledProcessError(result=res, expected=(0,))
        return (res.stdout_str, res.stderr_str)

    def dependency_update(self, path):
        # TODO: refactor for run_cmd?
        executor = utils.get_local_executor()
        result = executor.execute(f"{self.binary} dependency update", cwd=path)
        assert result.exit_code == 0, result

    def list(self, namespace=None, args=None):
        args = args or []
        cmd = [
            "list",
        ]
        if namespace:
            cmd.extend(["-n", namespace])
        cmd.extend(["-o", "json", *args])
        stdout, stderr = self.run_cmd(cmd)
        if isinstance(res := yaml.safe_load(stdout), list):
            return res
        else:
            LOG.error("Helm return unexpected answer " + stdout)
            return []

    def list_all(self):
        return self.list(args=['-A'])

    def get_release_values(self, name, namespace=None, args=None):
        args = args or []
        cmd = [
            "get",
            "values",
            name,
        ]
        if namespace:
            cmd.extend(["-n", namespace])
        cmd.extend(["-o", "json", *args])
        stdout, stderr = self.run_cmd(cmd)
        return yaml.safe_load(stdout)

    def get_metadata(self, name, namespace=None, args=None):
        args = args or []
        cmd = [
            "get",
            "metadata",
            name,
        ]
        if namespace:
            cmd.extend(["-n", namespace])
        cmd.extend(["-o", "json", *args])
        stdout, stderr = self.run_cmd(cmd)
        return yaml.safe_load(stdout)

    def get_releases_values(self, args=None):
        args = args or []
        res = {}
        for release in self.list(args):
            name = release['name']
            res[name] = self.get_release_values(name)
        return res

    def delete_chart(self, chart_name, namespace=None, cascade=None, timeout="5m"):
        cmd = [
            "uninstall",
            chart_name,
        ]
        if cascade:
            cmd.extend(["--cascade", cascade])
        if namespace:
            cmd.extend(["-n", namespace])
        cmd.extend(["--timeout", timeout])
        LOG.debug(f"Helm command: {cmd}")
        stdout, stderr = self.run_cmd(cmd)
        LOG.info(stdout)
        if stderr:
            LOG.error(stderr)
            return False
        return True

    def install_chart(self, chart_name, chart_path, version, namespace=None, timeout="5m",
                      create_ns=True, values_path=None, wait=True, insecure=settings.KSI_HELM_INSECURE,
                      source=None, cert_text=None):
        tmp_f = None
        if not source:
            source = settings.KCM_SOURCE
        if not cert_text:
            cert_text = settings.KSI_CUSTOM_REGISTRY_CERT_TEXT
        cmd = [
            "install",
            chart_name,
            chart_path,
            "--version",
            version,
            "--timeout",
            timeout
        ]
        if namespace:
            cmd.extend(["-n", namespace])
        if create_ns:
            cmd.extend(["--create-namespace"])
        if wait:
            cmd.extend(["--wait"])
        if values_path is not None:
            for value_path in values_path:
                cmd.extend(["--values", value_path])
        if insecure:
            cmd.extend(["--insecure-skip-tls-verify"])

        if source == 'custom-enterprise':
            # None(va4st): Only with custon registries we can use custom certs
            if settings.KSI_CUSTOM_REGISTRY_CERT_PATH:
                ca_file = settings.KSI_CUSTOM_REGISTRY_CERT_PATH
                cmd.extend(["--ca-file", ca_file])
            elif cert_text:
                # Note(va4st): If set delete=True - it will be removed before helm exec even if helm will be inside with
                # block
                with tempfile.NamedTemporaryFile(mode='w', suffix='.crt', delete=False) as temp_file:
                    temp_file.write(cert_text)
                    temp_file.seek(0)
                    tmp_f = temp_file
                    ca_file = temp_file.name
                cmd.extend(["--ca-file", ca_file])

        LOG.info(f"Executing helm command: {cmd}")
        stdout, stderr = self.run_cmd(cmd)
        # Remove if tmp cert exists
        if tmp_f:
            os.remove(tmp_f.name)
        LOG.info(stdout)
        if stderr:
            LOG.error(stderr)
            return False
        return True

    def get_helmrelease_status(self, name, namespace):
        """Get status of installed helmrelease.

        :return: str
        """
        metadata = self.get_metadata(name, namespace)
        return metadata.get('status', None)

    def is_release_present(self, name, verbose=False):
        list_hrs = self.list_all()
        _required = [x for x in list_hrs if x['name'] == name]
        if len(_required) > 1:
            LOG.warning(f"Found more than one <{name}> hr release:"
                        f"\n{_required}")
        present = True if _required else False
        if verbose and present:
            LOG.info(f"{name} HelmRelease exists in cluster:\n{_required}")
        elif verbose and not present:
            LOG.warning(f'No HelmRelease with name {name} found')
        return present
