#    Copyright 2017 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations


from si_tests.clients.k8s.base import K8sNamespacedResource
from si_tests.clients.k8s.base import K8sBaseManager
from si_tests import logger

LOG = logger.logger


class K8sNode(K8sNamespacedResource):
    resource_type = 'node'

    def _read(self, **kwargs):
        return self._manager.api.read_node(self.name, **kwargs)

    def _create(self, body, **kwargs):
        return self._manager.api.create_node(body, **kwargs)

    def _patch(self, body, **kwargs):
        return self._manager.api.patch_node(self.name, body, **kwargs)

    def _replace(self, body, **kwargs):
        return self._manager.api.replace_node(self.name, body, **kwargs)

    def _delete(self, **kwargs):
        self._manager.api.delete_node(
            self.name, **kwargs)

    @property
    def runtime(self):
        """
        Get runtime of the node. None if not exist (idk it's possible or not actually)
        :return:
        """
        runtime_version = self.data.get('status', {}).get('node_info', {}).get('container_runtime_version', None)
        return runtime_version

    def get_name(self):
        return self.data['metadata']['name']

    def exists_taint(self, taint: dict) -> bool:
        node_name = self.data['metadata']['name']
        node_taints = self.data['spec']['taints']
        if node_taints is not None:
            for node_taint in node_taints:
                if node_taint.get("key") == taint.get("key") and node_taint.get("effect") == taint.get("effect"):
                    LOG.info('Node {0} has taint: {1}'.format(node_name, taint))
                    return True
        return False

    def add_taint(self, taint):
        existing_taints = self.data['spec']['taints']
        node_name = self.data['metadata']['name']
        LOG.debug("Node {0} current taints: {1}".format(node_name, existing_taints))
        if existing_taints is not None:
            existing_taints.append(taint)
            body = {"spec": {
                "taints": existing_taints
                }
            }
        else:
            body = {"spec": {
                "taints": [taint]
                }
            }
        LOG.info("Adding taints: {0} to node {1}".format(taint, node_name))
        self._patch(body)
        LOG.debug("Updated taints for node {0}: {1}".format(node_name, self.data['spec']['taints']))

    def exists_labels(self, labels: dict) -> bool:
        node_labels = self.data['metadata']['labels']
        for label in labels.keys():
            if labels.get(label) != node_labels.get(label):
                return False
        return True

    def check_status(self, expected_status):
        node_status = self.read().status
        LOG.info("Status fetched")
        status = \
            [x.status for x in node_status.conditions
             if x.type == 'Ready'][0]
        if status != expected_status:
            err = "Actual status {0}, expected {1}".format(status,
                                                           expected_status)
            LOG.error(err)
            raise Exception(err)

    def list_conditions(self):
        node_status = self.read().status
        return node_status.get("conditions", [])

    def cordon(self):
        body = {"spec": {"unschedulable": True}}
        self._patch(body)

    def uncordon(self):
        body = {"spec": {"unschedulable": False}}
        self._patch(body)


class K8sNodeManager(K8sBaseManager):
    resource_class = K8sNode

    @property
    def api(self):
        return self._cluster.api_core

    def _list(self, namespace, **kwargs):
        return self.api.list_node(**kwargs)

    def _list_all(self, **kwargs):
        return self._list(None, **kwargs)

    def all_ready(self):
        return set(['True']) == set(self.list_statuses().values())

    def node_status(self, nodename='', expected_status='Ready'):
        """
        If expected_status 'NotReady' will returned True
        (when status in k8s node:
           - type: Ready
             status: False)
        """
        for node in self.list_raw().to_dict()['items']:
            if node['metadata']['name'] == nodename:
                status = [x['status'] for x in node['status']['conditions']
                          if x['type'] == 'Ready'][0]
                if expected_status == 'NotReady' and \
                        (status == "False" or status == "Unknown"):
                    node_status = True
                elif expected_status == 'Ready' and status == "True":
                    node_status = True
                else:
                    node_status = False
        return node_status

    def list_statuses(self):
        nodes = {}
        for node in self.list_raw().to_dict()['items']:
            ready = [x['status'] for x in node['status']['conditions']
                     if x['type'] == 'Ready'][0]
            nodes[node['metadata']['name']] = ready
        return nodes

    def describe_fixed_resources(self):
        """Collect the release-independent properties of the resources

        To check that the upgrades don't change any collected here property.
        """
        data = {}
        for obj in self.list_raw().to_dict()['items']:
            name = '{0}/{1}'.format(self.resource_type,
                                    obj['metadata']['name'])
            for label, value in obj['metadata']['labels'].items():
                res_name = '{0}/labels/{1}'.format(name, label)
                data[res_name] = value
            for spec, value in obj['spec'].items():
                res_name = '{0}/spec/{1}'.format(name, spec)
                data[res_name] = value
            for addr in obj['status']['addresses']:
                res_name = '{0}/address/{1}'.format(name, addr['address'])
                data[res_name] = addr['type']
            for capacity_type, capacity in obj['status']['capacity'].items():
                res_name = '{0}/capacity/{1}'.format(name, capacity_type)
                data[res_name] = capacity
        return data
