#    Copyright 2021 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.


import argparse
import os
import sys
import tempfile
import time
from uuid import uuid4

import yaml
from exec_helpers import SSHClient, SSHAuth, Subprocess
from exec_helpers.exec_result import ExecResult
from retry import retry

from ksi_runtest import logger
from ksi_runtest import utils

LOG = logger.logger

EXEC_DEFAULTS = {
    'COMMAND_TIMEOUT': 3600,  # secs
    'KSI_JOB_NAME': '',  # to be set by CI, for dir-naming only
    'KSI_BUILD_NUMBER': '',  # to be set by CI, for dir-naming only
    'KSI_RUN_ON_REMOTE': False,
    'KSI_RUN_ON_REMOTE_PULLING': False,
    'KSI_RUN_ON_RHEL': False,
    'KSI_SEED_DOWNLOAD_DIRS': '',
    'KSI_SEED_SSH_LOGIN': 'ksiuser',
    'KSI_SEED_SSH_PRIV_KEY': '',  # SSH private key content from config file (to write into a tmp file)
    'KSI_SEED_SSH_PRIV_KEY_FILE': '',
    'KSI_SEED_STANDALONE_EXTERNAL_IP': '',
    'KSI_SEED_UPLOAD_DIRS': '',
    'KSI_SEED_VIRTUALENV_NAME': 'ksi-venv',
    'KSI_SEED_WORKSPACE_PATH_PREFIX': '/var/lib/ksi-tests',
    'KSI_SEED_WORKSPACE_PATH_USE_CACHE': False,
    'KSI_SETUP_COMMAND': '',
    'KSI_SETUP_COMMAND_TIMEOUT': 1800,
    'KSI_SETUP_RHEL_COMMAND': '',
    'KSI_TEARDOWN_COMMAND': '',
    'KSI_TEARDOWN_COMMAND_TIMEOUT': 1800,
    'KSI_USER': 'ksiuser',
    'KSI_VERBOSE': True,
    'USER_TIME': time.strftime("%Y%m%d_%H_%M_%S"),
}

CONFIG_CREATE_DEFAULTS = {
    'KSI_RUN_ON_REMOTE': False,
    'KSI_RUN_ON_RHEL': False,
    'KSI_SEED_DOWNLOAD_DIRS': 'artifacts:artifacts',
    'KSI_SEED_SSH_LOGIN': 'ksiuser',
    'KSI_SEED_SSH_PRIV_KEY_FILE': '',
    'KSI_SEED_UPLOAD_DIRS': '.:.',
    'KSI_SEED_VIRTUALENV_NAME': 'ksi-venv',
    'KSI_SEED_WORKSPACE_PATH_PREFIX': '/var/lib/ksi-tests',
    'KSI_TEARDOWN_COMMAND': '',
    'KSI_VERBOSE': True,
    'KSI_SEED_STANDALONE_EXTERNAL_IP': '',
    'KSI_SETUP_RHEL_COMMAND': ('[ -d ${KSI_SEED_VIRTUALENV} ] || ( virtualenv -p python3.8 ${KSI_SEED_VIRTUALENV}); '
                               '${KSI_SEED_VIRTUALENV}/bin/pip install -r si_tests/requirements.txt; '),
    'KSI_SETUP_COMMAND': ('[ -d ${KSI_SEED_VIRTUALENV} ] || ( sudo apt update; '
                          'sudo apt-get install -y python3 python3-virtualenv virtualenv;'
                          ' virtualenv -p python3 ${KSI_SEED_VIRTUALENV}); '
                          '${KSI_SEED_VIRTUALENV}/bin/pip install -r si_tests/requirements.txt; '),

}
# Generic vars
SKIP_ENV_VARS = [
    'DBUS_SESSION_BUS_ADDRESS',
    'DISPLAY',
    'HOME',
    'HOSTNAME',
    'LANG',
    'LC_ADDRESS',
    'LD_LIBRARY_PATH',
    'LIBVIRT_DEFAULT_URI',
    'PATH',
    'PS1',
    'PWD',
    'SI_WORKFLOW_PARAMETERS',  # Multiline export issues when some parameters contain JSON
    'SSH_AGENT_PID',
    'SSH_AUTH_SOCK',
    'TMPDIR',
    'VIRTUAL_ENV',
    'WORKSPACE',
    'WORKSPACE_TMP',
]
# Jekins internal vars
SKIP_ENV_VARS += [
    'DEFAULT_GIT_BASE_URL',
    'DEFAULT_GIT_CREDENTIALS',
    'EXECUTOR_NUMBER',
    'GIT_AUTHOR_EMAIL',
    'GIT_AUTHOR_NAME',
    'GIT_COMMITTER_EMAIL',
    'GIT_COMMITTER_NAME',
    'HUDSON_HOME',
    'HUDSON_SERVER_COOKIE',
    'HUDSON_URL',
    'JENKINS_HOME',
    'JENKINS_NODE_COOKIE',
    'JENKINS_SERVER_COOKIE',
    'NODE_LABELS',
    'NODE_NAME',
    'STAGE_NAME',
]

_boolean_states = {
    "1": True,
    "yes": True,
    "true": True,
    "on": True,
    "0": False,
    "no": False,
    "false": False,
    "off": False,
}

env_var_types_cast = {
    str: lambda name: str(name),
    bool: lambda name: _boolean_states.get(name.lower()) if type(name) is str else bool(name),
    int: lambda name: int(name),
}


def get_exec_defaults(extra_defaults, si_config_path=''):
    """Load defaults

    1. Get defaults from ksi_config_path YAML key 'run_on_remote'
    2. Overwrite defaults with environment variables if set
    """
    if si_config_path:
        with open(si_config_path, 'r') as f:
            ksi_config = yaml.load(f.read(), Loader=yaml.SafeLoader)
            config_defaults = ksi_config.get('run_on_remote', {})
    else:
        config_defaults = {}

    for key, value in extra_defaults.items():
        value_type = type(value) if type(value) in env_var_types_cast else str
        env_value = os.environ.get(key) or config_defaults.get(key, value)
        env_value = env_var_types_cast[value_type](env_value)
        config_defaults[key] = env_value

    return config_defaults


@retry(Exception, delay=1, tries=3, jitter=1, logger=LOG)
def remote_seed(opts):
    """Init an SSHClient object"""

    key = utils.load_keyfile(opts.seed_ssh_priv_key_file)['private_obj']
    seed_ip = opts.seed_standalone_external_ip
    assert seed_ip, 'KSI_SEED_STANDALONE_EXTERNAL_IP is empty!'
    ssh_cfg = {
        "*": {
            "StrictHostKeyChecking": False,
            "UserKnownHostsFile": "/dev/null",
        }
    }
    remote = SSHClient(
        host=seed_ip,
        port=22,
        auth=SSHAuth(
            username=opts.seed_ssh_login,
            key=key,
            password=None,
            passphrase=None),
        # we never ever want to touch local agents
        allow_ssh_agent=False,
        keepalive=600,
        ssh_config=ssh_cfg,
        verbose=True)
    remote.logger.addHandler(logger.console)
    return remote


def rsync_run(opts, verbose=False, reverse=False,
              from_dir=None, todir=None, from_file=None, tofile=None, rsync_delete=None, ):
    """
    Run local rsync call
    :param rsync_delete:
    :param verbose:
    :param from_dir:
    :param todir:
    :param from_file:
    :param tofile:
    :param reverse: False -> sync from local to host
                    True -> sync from host to local
    :return:
    """
    rsync_delete = rsync_delete or ''
    rsync_verbose = ''
    if rsync_delete:
        rsync_delete = '--delete'
    if verbose:
        rsync_verbose = '--verbose'
    hostname = opts.seed_standalone_external_ip
    username = opts.seed_ssh_login
    sshkeyfile = opts.seed_ssh_priv_key_file

    # we need to full path to key,in case need to run from pushd
    sshopts = (f'"ssh -i'
               f' {os.path.abspath(sshkeyfile)}'
               ' -o StrictHostKeyChecking=no'
               ' -o ConnectionAttempts=10 '
               ' -o UserKnownHostsFile=/dev/null "')
    rsync_opts = f' --archive --compress --progress --partial {rsync_delete} {rsync_verbose}'
    # 'ksi-runtest.log' - reserved name for the log file, and we don't need to sync it- it's available at
    # runner stdout mainly
    exclude_opts = ("--exclude='ksi-runtest.log' "
                    "--exclude='.tox/*' "
                    "--exclude='.git/*' "
                    "--exclude='.github/*' "
                    "--exclude='.idea/*' ")
    if not (todir or tofile):
        raise Exception("Target not set")
    r = Subprocess()
    r.logger.addHandler(logger.console)
    if from_file:
        _to = tofile or todir
        if reverse:
            LOG.info(f"Downloading '{hostname}:{from_file}' to '{_to}'")
            cmd = f'rsync -e {sshopts} {exclude_opts} {rsync_opts} {username}@{hostname}:{from_file} {_to}'
            r.check_call(cmd, raise_on_err=True, verbose=verbose, timeout=3600)
        else:
            cmd = f'rsync -e {sshopts} {exclude_opts} {rsync_opts} {from_file} {username}@{hostname}:{_to}'
            LOG.info(f"Uploading '{from_file}' to '{hostname}:{_to}'")
            r.check_call(cmd, raise_on_err=True, verbose=verbose, timeout=3600)
    elif from_dir:
        if reverse:
            from_dir = from_dir + '/'
            cmd = f'rsync -e {sshopts} {exclude_opts} {rsync_opts} {username}@{hostname}:{from_dir} .'
            with utils.pushd(todir):
                LOG.info(f"Downloading '{hostname}:{from_dir}' dir to '{todir}'")
                r.check_call(cmd, raise_on_err=True, verbose=verbose, timeout=3600)
        else:
            cmd = f'rsync -e {sshopts} {exclude_opts} {rsync_opts} . {username}@{hostname}:{todir}'
            with utils.pushd(from_dir):
                LOG.info(f"Uploading '{from_dir}' dir to '{hostname}:{todir}'")
                r.check_call(cmd, raise_on_err=True, verbose=verbose, timeout=3600)
    else:
        raise Exception('Wrong invocation')


def create_workspace(opts):
    """Create a workspace directory with access rights for the current user"""
    remote = remote_seed(opts)
    path = opts.seed_workspace_path_prefix
    workspace = os.path.join(path, opts.seed_workspace_name)
    LOG.info(f"Creating workspace at {workspace}")
    cmd = f"sudo mkdir -p {workspace} && sudo chown $(id -u):$(id -g) {path} {workspace}"
    remote.check_call(command=cmd, verbose=False, timeout=60)
    if opts.seed_workspace_path_use_cache:
        cache_dir = os.path.join(opts.seed_workspace_path_prefix, '.cache')
        if remote.islink(cache_dir):
            # remote.islink not enough to check if link alive, so we must double-check
            # for case, when .cache link is broken for some reason
            ret = remote.execute(verbose=True,
                                 timeout=30,
                                 command=f'if [ -e {cache_dir} ] && [ -L {cache_dir} ]; then exit 0; else exit 1 ; fi')
            if ret.exit_code != 0:
                LOG.warning(f'Target cache {cache_dir} dir is broken,removing it')
                remote.check_call(command=f'rm -v {cache_dir}', verbose=True)
            else:
                LOG.info(f'Cache dir exists on seed, trying to pre-rsync locally\n'
                         f':{cache_dir} => {workspace}')
                _cmd = ("rsync "
                        "--exclude='artifacts/*' "
                        "--exclude='.tox/*' "
                        "--exclude='.git/*' "
                        "--exclude='__pycache__' "
                        "--exclude='*.pyc' "
                        "--exclude='*.pyo' "
                        "--archive "
                        "--progress "
                        "--partial "
                        f"{cache_dir}/ "
                        f"{workspace} ")
                remote.check_call(command=_cmd, verbose=False,
                                  timeout=360)


def dirs(paths):
    """Generate pairs of source:destination pairs from the paths string

    :param paths: semicolon-separated pairs of source:destination paths
                  separated with a colon. For example:
                  '.:.;~/.cache/pip:~/.cache/pip'
    """
    pairs = paths.split(';')
    for pair in pairs:
        if ':' in pair:
            yield pair.split(":")
        else:
            LOG.warning(f"Wrong path format, no colon found: '{pair}'")


def upload_dirs(opts):
    """Upload directories specified in opts.seed_upload_dirs to the seed node"""
    path = opts.seed_workspace_path_prefix
    name = opts.seed_workspace_name
    _rsync_delete = False
    if opts.seed_workspace_path_use_cache:
        _rsync_delete = True
    for source, destination in dirs(opts.seed_upload_dirs):
        if not destination.startswith("~") and not destination.startswith("/"):
            destination = f"{path}/{name}/{destination}"
        rsync_run(opts, reverse=False, from_dir=source, todir=destination,
                  rsync_delete=_rsync_delete)


def download_dirs(opts):
    """Download directories specified in opts.seed_download_dirs from the seed node"""
    path = opts.seed_workspace_path_prefix
    name = opts.seed_workspace_name
    for source, destination in dirs(opts.seed_download_dirs):
        if not source.startswith("~") and not source.startswith("/"):
            source = f"{path}/{name}/{source}"
        rsync_run(opts, reverse=True, from_dir=source, todir=destination)


def get_env_vars(opts):
    """Make a list of environment variables to expose into remote node

    - Skip the environment variables that match the already existing
      variables on the remote.
    - Skip the environment variables that match the SKIP_ENV_VARS list.
    - Add extra environment variables KSI_WORKSPACE and KSI_SEED_VIRTUALENV
    """
    cwd = os.getcwd()
    remote = remote_seed(opts)
    cmd = 'env'
    ret = remote.check_call(command=cmd, verbose=False, timeout=60)
    remote_env_vars = [line.split("=")[0] for line in ret.stdout_str.splitlines()]

    local_env_vars = {key: val
                      for key, val in os.environ.items()
                      if key not in remote_env_vars and key not in SKIP_ENV_VARS}
    local_env_vars['KSI_WORKSPACE'] = f"{opts.seed_workspace_path_prefix}/{opts.seed_workspace_name}"
    local_env_vars['KSI_SEED_VIRTUALENV'] = f"{opts.seed_workspace_path_prefix}/{opts.seed_virtualenv_name}"
    LOG.debug(yaml.dump(local_env_vars))

    for key, value in local_env_vars.items():
        if value.startswith(cwd):
            LOG.warning(f"=== Environment variable {key} provides a local path"
                        f" that cannot be used on the remote node: {value}")
    return local_env_vars


def create_remote_source_file(opts, env_vars, workspace, seed_virtualenv):
    """Create a source file on the remote node inside workspace directory

    In additional, try to activate the python virtualenv, if it was created
    in the custom --setup-command
    """
    remote = remote_seed(opts)
    env_vars_str = '\n'.join([f"export {key}='{val}'" for key, val in env_vars.items()])
    source_file = f'{workspace}/.env-si-test'
    cmd = (f"cat << 'EOF' > {source_file}\n"
           f"{env_vars_str}\n"
           f"[ -f {seed_virtualenv}/bin/activate ] && . {seed_virtualenv}/bin/activate ||"
           f" echo WARNING: python virtualenv not found at {seed_virtualenv}\n"
           f"EOF\n")
    remote.check_call(command=cmd, verbose=False, timeout=60)
    return source_file


def runcmd_remote_quiet(cmd, opts, timeout):
    """Execute the command with disabled verbose

    Use to prepare the environment and cleanup steps"""
    remote = remote_seed(opts)
    if cmd:
        LOG.info(f"Quiet run the command: '{cmd}'")
        remote.check_call(command=cmd, verbose=False, timeout=timeout)


@retry(Exception, delay=20, tries=30, jitter=20, logger=LOG)
def _wait_and_parse_logs(remote, log_file, exit_code_file, counter, start_time, timeout=300, interval=60):
    """
    Parse remote log for changes, wait for exit code in file.

    NIT: we assume extremal network connection, so each remote call have 30sec timeout, if failed - @retry handler
    retry, but from log condition state stored in counter dict.

    Options:
    :param exit_code_file: - file, where supposed to be exitCode from initial remote command.
    :param log_file: - log file to parse, on remote host
    :param timeout:  - raise if 'exit_code_file' won't become after
                          this amount of seconds
    :counter:   Proxy-dict param, to store current log position,during retry, formay:
                {'first_line': 0,
                'last_line': 0}
    """
    exec_timeout = 30
    run_limit_timeout = exec_timeout + 10
    run_limit_msg = f"Looks like ssh stuck, timeout was:{run_limit_timeout}"
    with utils.RunLimit(run_limit_timeout, run_limit_msg):
        remote.reconnect()
    with utils.RunLimit(run_limit_timeout, run_limit_msg):
        exit_file_exist = remote.isfile(exit_code_file)
    while not exit_file_exist:
        LOG.debug(f"Remote file {exit_code_file} not yet exist,print lines, and sleep for {interval}s")
        with utils.RunLimit(run_limit_timeout, run_limit_msg):
            counter['last_line'] = int(remote.check_call(f"cat {log_file} |wc -l",
                                                         timeout=exec_timeout).stdout_str)
        if counter['last_line'] > counter['first_line']:
            LOG.debug(f"Printing {counter['first_line'] + 1}:{counter['last_line']} from {log_file}")
            with utils.RunLimit(run_limit_timeout, run_limit_msg):
                log_data = remote.execute(f"sed -n '{counter['first_line'] + 1},{counter['last_line']}p' {log_file}",
                                          verbose=False,
                                          timeout=exec_timeout)
            LOG.info(log_data.stdout_str)
            counter['first_line'] = counter['last_line']
        if start_time + timeout < time.time():
            err_msg = f"Failed to wait for existence file {exit_code_file}," \
                      f"spent:{time.time() - start_time} seconds."
            LOG.error(err_msg)
            raise Exception(err_msg)
        LOG.debug(f"_wait_and_parse_logs sleeping for {interval}")
        time.sleep(interval)
        with utils.RunLimit(run_limit_timeout, run_limit_msg):
            exit_file_exist = remote.isfile(exit_code_file)

    with utils.RunLimit(run_limit_timeout, run_limit_msg):
        result_code = int(remote.check_call(f"cat {exit_code_file}",
                                            verbose=False,
                                            timeout=exec_timeout).stdout_str)
    LOG.info(f"Remote file {exit_code_file} exist, exit with exitCode:{result_code}")
    with utils.RunLimit(run_limit_timeout, run_limit_msg):
        counter['last_line'] = int(remote.check_call(f"cat {log_file} |wc -l", verbose=False,
                                                     timeout=exec_timeout).stdout_str)
    if counter['last_line'] > counter['first_line']:
        with utils.RunLimit(run_limit_timeout, run_limit_msg):
            log_data = remote.execute(f"sed -n '{counter['first_line'] + 1},{counter['last_line']}p' {log_file}",
                                      verbose=False, timeout=exec_timeout)
        LOG.info(log_data.stdout_str)

    return result_code


def _runcmd_remote_pulling(remote, main_command_pre, opts):
    """Executing the command remotely

    NIT: timeout as param for remote useless now, and will be pushed to waiter,
    so general timeout will be increased a little.
    """

    runcmd_remote_quiet('which screen || sudo apt-get install -qq -y screen', opts, opts.setup_command_timeout)
    target_ws = f"{opts.seed_workspace_path_prefix}/{opts.seed_workspace_name}/"
    screen_id = str(uuid4())[:6]
    screen_log = f"screen_log_{screen_id}.log"
    screen_e_code_file = f"screen_e_code_{screen_id}"
    screen_run_test_file_name = f"run_test_{screen_id}.sh"
    main_command = f"{main_command_pre}; " \
                   f"screen -S {screen_run_test_file_name} " \
                   f"-L -Logfile {screen_log} " \
                   f"-d " \
                   f"-m " \
                   f"./{screen_run_test_file_name}"
    # to properly catch exit code from exact commandscenario command, wrap test into file,
    # that's eliminates parsing issues with scree -dm command parsing.
    run_test_file_content = f"""
#!/bin/bash

set -x
{opts.command}
echo $? > {screen_e_code_file}
"""
    LOG.debug(f"Writing to local file {screen_run_test_file_name}, content:\n{run_test_file_content}")
    with open(screen_run_test_file_name, "w") as file:
        file.write(run_test_file_content)
    LOG.info(f"Uploading local file {screen_run_test_file_name} to {target_ws}")
    remote.upload(screen_run_test_file_name, target_ws)
    remote.check_call(f"chmod 0755 {target_ws}/{screen_run_test_file_name} ", verbose=True)

    ret = remote.execute(command=main_command, verbose=opts.verbose, timeout=opts.command_timeout)
    if not ret.exit_code == 0:
        _msg = f"Failed to run command:{main_command}, on target screen:{screen_id}"
        raise Exception(_msg)
    LOG.info(f"Remote command:{main_command}, ran on target screen:{screen_id}, waiting {screen_e_code_file} to appear")

    counter = {'first_line': 0,
               'last_line': 0}
    start_time = time.time()
    waiter_e_code = _wait_and_parse_logs(remote,
                                         log_file=f"{target_ws}/{screen_log}",
                                         exit_code_file=f"{target_ws}/{screen_e_code_file}",
                                         counter=counter,
                                         start_time=start_time,
                                         timeout=opts.command_timeout,
                                         interval=30)
    LOG.info(f"Command: {opts.command} finished on remote host, with exitCode:{waiter_e_code}")
    ret = ExecResult(cmd=opts.command, exit_code=waiter_e_code)
    return ret


def runcmd_remote(opts):
    """Executing the command remotely

    1. Create a unique workspace on remove
    2. Rsync the current directory to the remote workspace (path '.' must be in SEED_UPLOAD_DIRS)
    3. Rsync the pip cache to the remote userdir cache (path to cache must be in SEED_UPLOAD_DIRS)
    4. Create/update the remote virtualenv (commands must be in SETUP_COMMAND)
    5. Install requirements into the remote virtualenv (commands must be in SETUP_COMMAND)
    6. Activate the remote virtualenv and run the main command
    7. Execute cleanup steps (commands must be in TEARDOWN_COMMAND)
    8. Rsync the 'artifacts' directory from the remote workspace (paths must be in SEED_DOWNLOAD_DIRS)
    """
    workspace = f"{opts.seed_workspace_path_prefix}/{opts.seed_workspace_name}"
    seed_virtualenv = f"{opts.seed_workspace_path_prefix}/{opts.seed_virtualenv_name}"
    remote = remote_seed(opts)

    create_workspace(opts)
    upload_dirs(opts)
    if opts.seed_workspace_path_use_cache:
        cache_dir = os.path.join(opts.seed_workspace_path_prefix, '.cache')
        LOG.info(f'Forcing {cache_dir} link update to => {workspace} ')
        _cmd = f'ln -fs {workspace} {cache_dir}'
        remote.check_call(command=_cmd, verbose=False)

    env_vars = get_env_vars(opts)
    source_file = create_remote_source_file(opts, env_vars, workspace, seed_virtualenv)

    setup_command = f"cd {workspace}; . {source_file}; {opts.setup_command}"
    setup_rhel_command = f"cd {workspace}; . {source_file}; {opts.setup_rhel_command}"
    teardown_command = f"cd {workspace}; . {source_file}; {opts.teardown_command}"
    main_command_pre = f"cd {workspace}; . {source_file}"
    main_command = f"{main_command_pre}; {opts.command}"
    if opts.run_on_rhel:
        runcmd_remote_quiet(setup_rhel_command, opts, opts.setup_command_timeout)
    else:
        runcmd_remote_quiet(setup_command, opts, opts.setup_command_timeout)
    try:
        if opts.run_on_remote_pulling:
            ret = _runcmd_remote_pulling(remote, main_command_pre, opts)
        else:
            ret = remote.execute(command=main_command, verbose=opts.verbose, timeout=opts.command_timeout)
    finally:
        time.sleep(5)  # To print out the logs from the remote stream
        download_dirs(opts)
        runcmd_remote_quiet(teardown_command, opts, opts.teardown_command_timeout)

    return ret


def runcmd_local(opts):
    """Executing the command locally"""

    cmd = opts.command
    verbose = opts.verbose

    subprocess = Subprocess()
    subprocess.logger.addHandler(logger.console)
    ret = subprocess.execute(command=cmd, verbose=verbose, timeout=opts.command_timeout)
    return ret


def load_params(args, defaults=None):
    """
    Parse CLI arguments and environment variables

    Returns: ArgumentParser instance
    """
    defaults = defaults or {}
    if defaults.get('JOB_NAME') and defaults.get('BUILD_NUMBER'):
        default_workspace_path = f"workspace_{defaults.get('JOB_NAME')}_{defaults.get('BUILD_NUMBER')}"
    else:
        default_workspace_path = f"workspace_{defaults.get('KSI_USER')}_{defaults.get('USER_TIME')}"

    common_parser = argparse.ArgumentParser(add_help=False)
    common_parser.add_argument('--run-on-remote', dest='run_on_remote',
                               action='store_const', const=True,
                               help='Run the command on a remote node instead of the current local directory',
                               default=defaults.get('KSI_RUN_ON_REMOTE', False))
    common_parser.add_argument('--run-on-rhel', dest='run_on_rhel',
                               action='store_const', const=True,
                               help='Set true for rhel mgmt/seed',
                               default=defaults.get('RUN_ON_RHEL', False))
    common_parser.add_argument('--run-on-remote-pulling', dest='run_on_remote_pulling',
                               action='store_const', const=True,
                               help='Run the command on a remote node instead of the current local directory, and get'
                                    'logs via pulling',
                               default=defaults.get('KSI_RUN_ON_REMOTE_PULLING', False))
    common_parser.add_argument('--seed-ssh-priv-key-file',
                               help='Path to the SSH private key file to access the seed node',
                               default=defaults.get('KSI_SEED_SSH_PRIV_KEY_FILE', ''),
                               type=str)
    common_parser.add_argument('--seed-standalone-external-ip',
                               help='Seed node IP address',
                               default=defaults.get('KSI_SEED_STANDALONE_EXTERNAL_IP', ''),
                               type=str)
    common_parser.add_argument('--seed-ssh-login',
                               help='Seed node SSH username',
                               default=defaults.get('KSI_SEED_SSH_LOGIN', ''),
                               type=str)
    common_parser.add_argument('--seed-upload-dirs',
                               help='Upload directories to seed node before run tests',
                               default=defaults.get('KSI_SEED_UPLOAD_DIRS', ''),
                               type=str)
    common_parser.add_argument('--seed-download-dirs',
                               help='Download directories from seed node after run tests',
                               default=defaults.get('KSI_SEED_DOWNLOAD_DIRS', ''),
                               type=str)
    common_parser.add_argument('--seed-workspace-path-prefix',
                               help='Path to the directory for workspaces from different tests',
                               default=defaults.get('KSI_SEED_WORKSPACE_PATH_PREFIX'), type=str)
    common_parser.add_argument('--seed-workspace-path-use-cache', dest='seed_workspace_path_use_cache',
                               action='store_const', const=True,
                               help='Use cache directory on seed node, to be used for `seed-workspace-path-prefix`'
                                    ' as cache. First si-tests call workspace will be used as cache.',
                               default=defaults.get('KSI_SEED_WORKSPACE_PATH_USE_CACHE', False))
    common_parser.add_argument('--seed-virtualenv-name',
                               help=('Name of the python virtualenv on the seed node that will be activated '
                                     'to run the command, if exists.\n'
                                     'If required, please init the virtualenv using --setup-command with '
                                     '${KSI_SEED_VIRTUALENV} path'),
                               default=defaults.get('KSI_SEED_VIRTUALENV_NAME'),
                               type=str)
    common_parser.add_argument('--setup-command',
                               help=('Command to execute inside the created workspace before run the main command,'
                                     ' with suppressed verbose'),
                               default=defaults.get('KSI_SETUP_COMMAND', ''),
                               type=str)
    common_parser.add_argument('--setup-rhel-command',
                               help=('Command to execute inside the created workspace before run the main command,'
                                     ' with suppressed verbose'),
                               default=defaults.get('KSI_SETUP_RHEL_COMMAND', ''),
                               type=str)
    common_parser.add_argument('--teardown-command',
                               help=('Command to execute inside the created workspace after run the main command,'
                                     ' with suppressed verbose'),
                               default=defaults.get('KSI_TEARDOWN_COMMAND', ''),
                               type=str)
    common_parser.add_argument('--command-timeout',
                               help=('Main command timeout'),
                               default=defaults.get('KSI_COMMAND_TIMEOUT', 3600),
                               type=int)
    common_parser.add_argument('--setup-command-timeout',
                               help=('Setup command timeout'),
                               default=defaults.get('KSI_SETUP_COMMAND_TIMEOUT', 1800),
                               type=int)
    common_parser.add_argument('--teardown-command-timeout',
                               help=('Teardown command timeout'),
                               default=defaults.get('KSI_TEARDOWN_COMMAND_TIMEOUT', 1800),
                               type=int)

    exec_parser = argparse.ArgumentParser(add_help=False)
    exec_parser.add_argument('--command',
                             help='Main command to execute inside the created workspace,'
                                  ' usually run pytest with a test',
                             default='',
                             type=str)
    exec_parser.add_argument('--config-path',
                             help='(Optional) Config with extra options for test runner',
                             default=defaults.get('KSI_CONFIG_PATH', ''),
                             type=str)
    exec_parser.add_argument('--seed-workspace-name',
                             help=('Name of the current workspace on the seed node\n'
                                   'Default: "workspace_${KSI_JOB_NAME}_${KSI_BUILD_NUMBER}" if set,'
                                   ' else "workspace_${KSI_USER}_${USER_TIME}" , where USER_TIME is YYYYMMDD_HH_MM_SS"\n'
                                   'The environment variable ${WORKSPACE} with the full path to the workspace'
                                   'will be available for --setup-command and --teardown-command'),
                             default=default_workspace_path,
                             type=str)

    verbose_parser = argparse.ArgumentParser(add_help=False)
    verbose_parser.add_argument('--verbose', dest='verbose',
                                action='store_const', const=True,
                                help='Show verbosed output',
                                default=defaults.get('KSI_VERBOSE', False))

    parser = argparse.ArgumentParser(
        description="Execute a shell command locally or on a remote host.\n"
                    "For additional help, use with -h/--help option")
    subparsers = parser.add_subparsers(title="Options",
                                       help='Available options',
                                       dest='option')

    subparsers.add_parser('exec',
                          parents=[exec_parser, common_parser, verbose_parser],
                          help="Execute command")

    subparsers.add_parser('config-create',
                          parents=[common_parser, verbose_parser],
                          help="Render a ksi-config YAML to stdout from defaults and provided arguments")

    if len(args) == 0:
        args = ['-h']
    return parser.parse_args(args)


def option_exec(opts, args):
    # Get the --ksi-config argument and load default values from the config
    # Priorities for the arguments, from high to low:
    # - arguments from CLI
    # - arguments from environment variables
    # - arguments from --ksi-config YAML file
    # - defaults defined in DEFAULTS variable
    defaults = get_exec_defaults(EXEC_DEFAULTS, opts.config_path)
    # Get arguments from CLI, using extra defaults
    opts = load_params(args, defaults)

    if not opts.command:
        LOG.info("--command argument not set")
        return 11

    LOG.info(f"=== RUN_ON_REMOTE: {opts.run_on_remote} ===")
    if opts.run_on_remote:
        if opts.seed_ssh_priv_key_file:
            ret = runcmd_remote(opts)
        elif defaults['KSI_SEED_SSH_PRIV_KEY']:
            # TODO(alexz-kh): perhaps, we dont need this part in ksi
            with tempfile.NamedTemporaryFile(mode="w") as tmp:
                tmp.write(defaults['KSI_SEED_SSH_PRIV_KEY'])
                tmp.flush()
                opts.seed_ssh_priv_key_file = tmp.name
                ret = runcmd_remote(opts)
        else:
            raise Exception("KSI_SEED_SSH_PRIV_KEY_FILE not set")
    else:
        ret = runcmd_local(opts)

    return ret.exit_code


def option_config_create(opts, args):
    defaults = get_exec_defaults(CONFIG_CREATE_DEFAULTS)
    opts = load_params(args, defaults)
    # Log informational messages to stderr, to not break the YAML structure in stdout
    LOG.error("=== Render ksi-config YAML to stdout")

    seed_ssh_priv_key = ''
    if opts.seed_ssh_priv_key_file and os.path.isfile(opts.seed_ssh_priv_key_file):
        with open(opts.seed_ssh_priv_key_file, 'r') as f:
            seed_ssh_priv_key = f.read()
    elif opts.run_on_remote:
        raise Exception("KSI_SEED_SSH_PRIV_KEY_FILE not set or file doesn't exist")

    config = {
        'run_on_remote': {
            'KSI_RUN_ON_REMOTE': opts.run_on_remote,
            'KSI_RUN_ON_RHEL': opts.run_on_rhel,
            'KSI_SEED_DOWNLOAD_DIRS': opts.seed_download_dirs,
            'KSI_SEED_SSH_LOGIN': opts.seed_ssh_login,
            'KSI_SEED_SSH_PRIV_KEY': seed_ssh_priv_key,
            'KSI_SEED_STANDALONE_EXTERNAL_IP': opts.seed_standalone_external_ip,
            'KSI_SEED_UPLOAD_DIRS': opts.seed_upload_dirs,
            'KSI_SEED_VIRTUALENV_NAME': opts.seed_virtualenv_name,
            'KSI_SEED_WORKSPACE_PATH_PREFIX': opts.seed_workspace_path_prefix,
            'KSI_SETUP_COMMAND': opts.setup_command,
            'KSI_SETUP_RHEL_COMMAND': opts.setup_rhel_command,
            'KSI_TEARDOWN_COMMAND': opts.teardown_command,
            'KSI_VERBOSE': opts.verbose,
        }
    }

    def str_presenter(dumper, data):
        if len(data.splitlines()) > 1:
            return dumper.represent_scalar('tag:yaml.org,2002:str', data, style='|')
        return dumper.represent_scalar('tag:yaml.org,2002:str', data)

    yaml.add_representer(str, str_presenter)

    print(yaml.dump(config))


OPTIONS = {
    'exec': option_exec,
    'config-create': option_config_create,
}


def main():
    args = sys.argv[1:]

    opts = load_params(args)
    option = opts.option

    if option not in OPTIONS.keys():
        print(f"Unsupported option {option}, please use one of: "
              f"{OPTIONS.keys()}")
        return 12

    method = OPTIONS[option]
    exit_code = method(opts, args)

    return exit_code


if __name__ == '__main__':
    sys.exit(main())
