import kubernetes
import base64
import json

from si_tests import logger

LOG = logger.logger


class k8s_lock:
    def __init__(self, name, namespace, k8s_client, retries=10, delay=6):
        self.name = name
        self.namespace = namespace
        self.k8s_client = k8s_client
        self.retries = retries
        self.delay = delay

    def _asquire_lock(self):
        for i in range(self.retries):
            try:
                self.k8s_client.secrets.create(
                    name=self.name,
                    namespace=self.namespace,
                    body={
                        "apiVersion": "v1",
                        "kind": "Secret",
                        "metadata": {
                            "name": self.name,
                            "namespace": self.namespace,
                        },
                        "data": {},
                    },
                )
                return
            except Exception as e:
                LOG.info(f"Got error: {e} while asquiring lock {self.namespace}/{self.name}")
                pass
        raise Exception(f"Failed to asquire lock {self.namespace}/{self.name}")

    def _release_lock(self):
        for i in range(self.retries):
            try:
                self.k8s_client.secrets.api.delete_namespaced_secret(self.name, self.namespace)
                return
            except kubernetes.client.rest.ApiException as e:
                if e.status == 404:
                    return
                LOG.debug(f"Got error: {e} while releasing lock {self.namespace}/{self.name}")
        raise Exception(f"Failed to asquire lock {self.namespace}/{self.name}")

    def __enter__(self):
        self._asquire_lock()

    def __exit__(self, type, value, traceback):
        self._release_lock()


class NetworkLockManager:
    """The Lock manager for network metadata, is intended to store objects in management cluster.

    The data is stored in two secrets:
        default/network-lock-manager: stores network_data passed during initialization. The update
            if network_data is not supported
        <child-cluster>/<child-cluster-name>-nlm-lock: stores information about used networks for
            the child cluster.

    :param name: the name of child cluster to store metadata for.
    :param namespace: the namespace of child cluster
    :param kaas_manager: kaas_manager instance
    :param network_data: the network data from terrafor (outputs.yaml)
    """

    def __init__(self, name, namespace, kaas_manager, network_data=None):
        self.kaas_manager = kaas_manager

        self._network_data_name = "network-lock-manager"
        self._network_data_namespace = "default"
        self._lock_name = f"{self._network_data_name}-lock-temp"
        self._network_lock_name = f"{self._network_data_name}-network-locks"

        self._lock_data_label = "network-manager-lock"
        self._lock_data_label_value = "data"
        self._lock_data_namespace = namespace
        self._lock_data_name = name

        if network_data is None:
            self._network_data = self.network_data
        else:
            self.network_data = network_data

    @property
    def network_data(self):
        data = self.kaas_manager.get_secret_data(self._network_data_name, self._network_data_namespace, "network_data")
        self._network_data = json.loads(data)
        return self._network_data

    @network_data.setter
    def network_data(self, network_data):
        body = {
            "apiVersion": "v1",
            "kind": "Secret",
            "metadata": {
                "name": self._network_data_name,
                "namespace": self._network_data_namespace,
            },
            "data": {"network_data": base64.b64encode(json.dumps(network_data).encode("utf-8")).decode()},
        }
        with k8s_lock(self._lock_name, self._network_data_namespace, self.kaas_manager.api):
            try:
                self.kaas_manager.get_secret_data(self._network_data_name, self._network_data_namespace, "network_data")
            except Exception as e:
                if e.status == 404:
                    # Do not allow to update network data if present.
                    self.kaas_manager.api.secrets.create(
                        name=self._network_data_name, namespace=self._network_data_namespace, body=body
                    )
                    # TODO: create this in default namespace
                    nlm_default = NetworkLockManager(f"{self._network_data_name}-lock", "default", self.kaas_manager)
                    for seed_ip, seed_net in network_data["seed_nodes"]["value"].items():
                        query = {"mode": "l3", "metro": seed_net["metro"]}
                        nlm_default._lock_vlan(
                            int(seed_net["vlan_id"]),
                            seed_net["metro"],
                            query,
                            tag=f"seed-{seed_ip}-pxe",
                        )

    @property
    def lock_data(self):
        data = {}
        try:
            data = self.kaas_manager.get_secret_data(self._lock_data_name, self._lock_data_namespace, "lock_data")
            return json.loads(data)
        except kubernetes.client.rest.ApiException as e:
            if e.status == 404:
                self.lock_data = {}
        return {}

    @property
    def lock_data_full(self):
        data = {}
        for ns in self.kaas_manager.api.namespaces.list():
            for secret_obj in self.kaas_manager.api.secrets.list(
                namespace=ns.name, label_selector=f"{self._lock_data_label}={self._lock_data_label_value}"
            ):
                raw_data = secret_obj.read().data
                data.update(json.loads(base64.b64decode(raw_data["lock_data"]).decode()))
        return data

    @lock_data.setter
    def lock_data(self, lock_data):
        body = {
            "apiVersion": "v1",
            "kind": "Secret",
            "metadata": {
                "name": self._lock_data_name,
                "namespace": self._lock_data_namespace,
                "labels": {self._lock_data_label: self._lock_data_label_value},
            },
            "data": {"lock_data": base64.b64encode(json.dumps(lock_data).encode("utf-8")).decode()},
        }
        try:
            secret_obj = self.kaas_manager.api.secrets.get(
                name=self._lock_data_name, namespace=self._lock_data_namespace
            )
            secret_obj.patch(body=body)
        except kubernetes.client.rest.ApiException as e:
            if e.status == 404:
                self.kaas_manager.api.secrets.create(
                    name=self._lock_data_name, namespace=self._lock_data_namespace, body=body
                )

    def set_lock_data_owner(self, cluster):
        secret_obj = self.kaas_manager.api.secrets.get(name=self._lock_data_name, namespace=self._lock_data_namespace)
        owner = {
            "apiVersion": cluster.data["api_version"],
            "kind": cluster.data["kind"],
            "name": cluster.name,
            "uid": cluster.uid,
        }
        body = secret_obj.read().to_dict()
        body["metadata"]["ownerReferences"] = [owner]
        secret_obj.patch(body=body)

    def _lock_vlan(self, number, metro, query, tag):
        """Assumed is called under lock."""
        current_data = self.lock_data
        if self._is_vlan_used(number, metro):
            raise Exception(f"The vlan {number} is already used.")
        current_data[f"{metro}_{number}"] = {"query": query, "tag": tag}
        self.lock_data = current_data

    def _unlock_vlan(self, number, metro):
        """Assumed is called under lock."""
        # TODO: implement unlock vlan functionality

    def _is_vlan_used(self, number, metro):
        """Assumed is called under lock."""
        if f"{metro}_{number}" in self.lock_data_full.keys():
            return True
        return False

    def _vid_from_lock_data(self, tag, metro, lock_data):
        for k, v in lock_data.items():
            if v["tag"] == tag and k.startswith(f"{metro}_"):
                return int(k.replace(f"{metro}_", ""))

    def _net_from_lock_data(self, vid, metro_vlans):
        for net in metro_vlans:
            if net["vlan_id"] == vid:
                return net

    def pick_nets(self, selector):
        """
        :param selector: array with network selector objects
            selector: {
             "tag": {
                "mode": network mode l2|l3
                "metro": the name of metro.
              }
            }
        :return : array with network objects
        :raise Error: when not able to pick up networks.
        """
        res = {}
        res_from_lock = {}
        with k8s_lock(self._lock_name, self._network_data_namespace, self.kaas_manager.api):
            lock_data = self.lock_data
            vlans = self.network_data["vlans"]["value"]
            for tag, network_query in selector.items():
                metro = network_query["metro"]
                metro_vlans = vlans[metro]
                vid_from_lock_data = self._vid_from_lock_data(tag, metro, lock_data)
                if vid_from_lock_data:
                    res_from_lock[tag] = self._net_from_lock_data(vid_from_lock_data, metro_vlans)
                    continue
                for net in metro_vlans:
                    if net in res.values():
                        # skip vlans picked in this selector but not yet locked.
                        continue
                    vid = net["vlan_id"]
                    if not self._is_vlan_used(vid, metro):
                        res[tag] = net
                        break
                else:
                    raise Exception(f"Failed to pick up free network {network_query} from {vlans}")
            # Perform lock only when all networks are picked.
            for tag, net in res.items():
                vid = net["vlan_id"]
                query = selector[tag]
                self._lock_vlan(net["vlan_id"], metro, query, tag)
            res.update(res_from_lock)
            return res

    def pick_net_by_tag(self, tag, metro):
        lock_data = self.lock_data
        metro_vlans = self.network_data["vlans"]["value"][metro]
        vid_from_lock_data = self._vid_from_lock_data(tag, metro, lock_data)
        if vid_from_lock_data:
            return self._net_from_lock_data(vid_from_lock_data, metro_vlans)
