import time

import pytest
import yaml

from si_tests import logger, settings
from si_tests.managers.clustercheck_mos_manager import ClusterCheckMosManager
from si_tests.utils import packaging_version as version
from si_tests.utils import waiters

LOG = logger.logger


@pytest.mark.usefixtures('restore_tfoperator_cr')
class TestTFDisableAnalytics(object):
    """Check disable TF analytics feature"""

    @pytest.fixture(scope='class', autouse=True)
    def setup_cls(self, tf_manager):
        tf_manager.detect_api_version()
        if not tf_manager.is_analytics_enabled():
            pytest.skip("TF Analytics is disabled. Test should be run against "
                        "environment with enabled Analytics")

    def test_disable_analytics(self, show_step, tf_manager):
        """ Disable TF analytics component

            Scenario:
                1. Disable TF Analytics
                2. Check analytics components
                3. Check cassandra analytics component
                4. Check zookeeper analytics component
                5. Check kafka component
                6. Delete pvc for disabled components
                7. Restart vRouters
                8. Wait TFOperator status
                9. Run TFTests
        """
        show_step(1)
        tf_manager.enable_analytics(False)

        show_step(2)
        waiters.wait(
            lambda: len(tf_manager.api.pods.list_starts_with(
                'tf-analytics')) == 0,
            timeout=120, interval=10
        )

        show_step(3)
        waiters.wait(
            lambda: len(tf_manager.api.pods.list_starts_with(
                'tf-cassandra-analytics')) == 0,
            timeout=120, interval=10
        )

        show_step(4)
        waiters.wait(
            lambda: len(
                tf_manager.api.pods.list_starts_with('tf-zookeeper-nal')) == 0,
            timeout=60, interval=10
        )

        show_step(5)
        waiters.wait(
            lambda: len(
                tf_manager.api.pods.list_starts_with('tf-kafka')) == 0,
            timeout=60, interval=10
        )

        show_step(6)
        pvcs = tf_manager.api.pvolumeclaims.list(
            label_selector="app=tf-cassandra-analytics")
        pvcs.extend(tf_manager.api.pvolumeclaims.list(
            label_selector="app=tf-zookeeper-nal"))
        pvcs.extend(tf_manager.api.pvolumeclaims.list(
            label_selector="app=tf-kafka"))
        for pvc in pvcs:
            LOG.info(f"Delete pvc {pvc.name}")
            pvc.delete()

        show_step(7)
        pods = tf_manager.get_vrouter_pods()
        for pod in pods:
            LOG.info(f"Delete pod {pod.name}")
            pod.delete()

        show_step(8)
        tf_manager.wait_tf_controllers_healthy(timeout=360)
        tf_manager.wait_tfoperator_healthy(timeout=120)

        show_step(9)
        assert tf_manager.run_pytest() == 'Succeeded'


@pytest.mark.usefixtures('restore_tfoperator_cr')
class TestTFImagePreCaching(object):
    """Check enable/disable TFImagePreCaching"""

    @pytest.fixture(scope='class', autouse=True)
    def setup_cls(self, tf_manager):
        tf_manager.detect_api_version()

    def test_image_precaching(self, show_step, tf_manager):
        """ Check TFImagePreCaching feature

            Scenario:
                1. Disable imagePreCaching
                2. Check preCaching DaemonSets/Pods were deleted
                3. Enable imagePreCaching
                4. Check preCaching pods were created
        """

        ds_prefix = "tf-image-pre-caching"
        ds_list = ['tfconfig', 'tfconfigdb', 'tfcontrol', 'tfvrouter', 'tfvrouter-dpdk', 'tfwebui']
        analytics_enabled = tf_manager.is_analytics_enabled()
        if analytics_enabled:
            ds_list.extend(['tfanalytics', 'tfanalyticsdb'])

        pods = tf_manager.get_imageprecaching_pods()

        show_step(1)
        tf_manager.enable_image_precaching(state=False)
        tf_manager.wait_tf_imageprecaching(exist=False)

        show_step(2)
        for ds in ds_list:
            ds_name = f"{ds_prefix}-{ds}"
            waiters.wait(
                lambda: tf_manager.api.daemonsets.present(ds_name, tf_manager.tf_namespace) is False,
                timeout=300, interval=5,
                timeout_msg="DaemonSet wasn't deleted during the timeout"
            )
            LOG.info(f"DaemonSet {ds_name} was deleted")
        for pod in pods:
            waiters.wait(
                lambda: tf_manager.api.pods.present(pod.name, tf_manager.tf_namespace) is False,
                timeout=210, interval=5,
                timeout_msg="DaemonSet wasn't deleted during the timeout"
            )
            LOG.info(f"Pod {pod.name} was deleted")

        tf_manager.wait_tfoperator_healthy()

        show_step(3)
        tf_manager.enable_image_precaching(state=True)
        tf_manager.wait_tf_imageprecaching(exist=True)
        tf_imageprecaching = tf_manager.tfimageprecaching(read=True)
        assert tf_imageprecaching.spec['enabled'] is True

        show_step(4)
        if analytics_enabled:
            assert tf_imageprecaching.spec['tfAnalyticsEnabled'] == analytics_enabled, \
                "Analytics flag must be False"
        for ds in ds_list:
            ds_name = f"{ds_prefix}-{ds}"
            waiters.wait(
                lambda: tf_manager.api.daemonsets.present(ds_name, tf_manager.tf_namespace) is True,
                timeout=300, interval=5,
                timeout_msg="DaemonSet wasn't created during the timeout"
            )
            LOG.info(f"DaemonSet {ds_name} was created")
        tf_manager.wait_tfoperator_healthy()


@pytest.mark.usefixtures('restore_tfoperator_cr')
class TestTFTools(object):
    """Check enable/disable TF tools"""

    @pytest.fixture(scope='class', autouse=True)
    def setup_cls(self, tf_manager):
        tf_manager.detect_api_version()

    def test_contrail_tools(self, show_step, tf_manager):
        """ Check TF contrail tools

            Scenario:
                1. Enable contrail-tools
                2. Check contrail-tool pods
                3. Check contrail-tool cli
                4. Disable contrail-tools
                5. Check pods are deleted
        """
        pod_prefix = 'tf-tool-ctools'
        show_step(1)
        if tf_manager.is_ctools_enabled():
            LOG.info("TF contrail-tools is enabled. Disable it.")
            tf_manager.enable_ctools(state=False)
            waiters.wait(
                lambda: len(tf_manager.api.pods.list_starts_with(
                    pod_prefix)) == 0,
                timeout=300, interval=10
            )
        tf_manager.enable_ctools(state=True)

        show_step(2)
        vrouter_nodes = tf_manager.api.nodes.list(
            namespace=tf_manager.tf_namespace,
            label_selector='tfvrouter=enabled'
        )
        LOG.info(f"Amount of vrouter nodes is {len(vrouter_nodes)}")
        waiters.wait(
            lambda: len(tf_manager.api.pods.list_starts_with(
                pod_prefix)) == len(vrouter_nodes),
            timeout=300, interval=10
        )

        show_step(3)
        ctools_pods = tf_manager.api.pods.list(
            namespace=tf_manager.tf_namespace,
            name_prefix=pod_prefix)
        # Calling exec and check response code
        cmd = "vif --list"
        out = ctools_pods[0].exec(['/bin/sh', '-c', cmd])
        LOG.info(f"Command output:\n{out}")
        assert ("Vrouter Interface Table" in out), "Check vRouter cli."

        show_step(4)
        tf_manager.enable_ctools(state=False)

        show_step(5)
        waiters.wait(
            lambda: len(tf_manager.api.pods.list_starts_with(
                pod_prefix)) == 0,
            timeout=300, interval=10
        )

    def test_tf_cli(self, show_step, tf_manager):
        """ Check TF cli

            Scenario:
                1. Enable tf-cli tool
                2. Check tf-cli pods
                3. Check tf-cli
                4. Disable tf-cli tool
                5. Check pods are deleted
        """
        pod_prefix = 'tf-tool-cli'
        show_step(1)
        if tf_manager.is_tf_cli_enabled():
            LOG.info("TF cli is enabled. Disable it.")
            tf_manager.enable_tf_cli(state=False)
            waiters.wait(
                lambda: len(tf_manager.api.pods.list_starts_with(
                    pod_prefix)) == 0,
                timeout=300, interval=10
            )

        tf_manager.enable_tf_cli(state=True)

        show_step(2)
        waiters.wait(
            lambda: len(tf_manager.api.pods.list_starts_with(
                pod_prefix)) == 1,
            timeout=180, interval=10
        )

        show_step(3)
        ctools_pods = tf_manager.api.pods.list(
            namespace=tf_manager.tf_namespace,
            name_prefix=pod_prefix)
        # Calling exec and check response code
        cmd = "tf-ist-cli vr status"
        out = ctools_pods[0].exec(['/bin/sh', '-c', cmd])
        LOG.info(f"Command output:\n{out}")
        assert ("module_id: contrail-vrouter-agent" in out), "Check TF cli."

        show_step(4)
        tf_manager.enable_tf_cli(state=False)

        show_step(5)
        waiters.wait(
            lambda: len(tf_manager.api.pods.list_starts_with(
                pod_prefix)) == 0,
            timeout=300, interval=10
        )


@pytest.mark.usefixtures('restore_tfoperator_cr')
class TestTFNetNsAZ(object):
    """Check netns_availability_zone option for scheduling load balancers and
    SNAT routers on the target hypervisors"""

    # Name of AZ should include the name of default az to check PRODX-39436
    az = 'si_test_nova_net'
    default_az = 'nova'

    @pytest.fixture(scope='class')
    def check_requirements(self, hypervisors, tf_manager):
        tfoperator_version = tf_manager.tfoperator_version()
        if version.parse(tfoperator_version) < version.parse("0.15.0"):
            pytest.skip(f"TFOperator version {tfoperator_version} doesn't "
                        f"support netns_availability_zone option")
        required = 2
        if len(hypervisors) < required:
            pytest.skip(f"Not enough OS hypervisors: "
                        f"{required}/{len(hypervisors)}")

    @pytest.fixture(scope='class')
    def hypervisors(self, os_manager):
        cmd = ['/bin/sh', '-c',
               'PYTHONWARNINGS=ignore::UserWarning '
               'openstack hypervisor list -f json -c "Hypervisor Hostname"']
        stdout = os_manager.keystone_client_pod.exec(cmd)
        hypervisors = [hpv["Hypervisor Hostname"].replace('.cluster.local', '') for hpv in yaml.safe_load(stdout)]
        return hypervisors

    @pytest.fixture(scope='class')
    def hypervisor(self, hypervisors, tf_manager):
        # Return host for dedicated AZ
        for pod in tf_manager.get_vrouter_pods():
            host = pod.data['spec'].get('node_name')
            for hypervisor in hypervisors:
                if host == hypervisor:
                    return hypervisor

    @pytest.fixture(scope='class')
    def create_az(self, hypervisor, os_manager):
        LOG.info(f"Create availability zone: {self.az}")
        cmd = ['/bin/sh', '-c',
               f'PYTHONWARNINGS=ignore::UserWarning '
               f'openstack aggregate create --zone {self.az} {self.az}']
        os_manager.keystone_client_pod.exec(cmd)
        LOG.info(f"Add {hypervisor} to availability zone {self.az}")
        cmd = ['/bin/sh', '-c',
               f'PYTHONWARNINGS=ignore::UserWarning '
               f'openstack aggregate add host {self.az} {hypervisor}']
        os_manager.keystone_client_pod.exec(cmd)
        yield
        LOG.info(f"Remove {hypervisor} from availability zone {self.az}")
        cmd = ['/bin/sh', '-c',
               f'PYTHONWARNINGS=ignore::UserWarning '
               f'openstack aggregate remove host {self.az} {hypervisor}']
        os_manager.keystone_client_pod.exec(cmd)
        LOG.info(f"Delete created availability zone: {self.az}")
        cmd = ['/bin/sh', '-c',
               f'PYTHONWARNINGS=ignore::UserWarning '
               f'openstack aggregate delete {self.az}']
        os_manager.keystone_client_pod.exec(cmd)

    @pytest.fixture(scope='class', autouse=True)
    def setup_cls(self, check_requirements, create_az, tf_manager):
        tf_manager.detect_api_version()

    @staticmethod
    def get_netns_list(pod):
        cmd = ['/bin/sh', '-c', 'ip netns list']
        stdout = pod.exec(cmd, container='agent')
        pod_name = pod.data['metadata']['name']
        host = pod.data['spec'].get('node_name')
        LOG.info(f"Network namespaces (vRouter: {pod_name}, host: {host}):\n"
                 f"{stdout}")
        return stdout.splitlines()

    @pytest.mark.dependency()
    def test_netns_az(self, show_step, request, hypervisor,
                      tf_manager, openstack_client_manager):
        """ Check netns option

            Scenario:
                1. Set netns_availability_zone
                2. Create LB instance
                3. Check that network namespaces located on dedicated compute
        """
        show_step(1)
        tf_manager.set_netns_availability_zone(self.az)

        show_step(2)
        ClusterCheckMosManager.created_stack_tf_lb(request, openstack_client_manager,
                                                   custom_params={"availability_zone": self.az})

        show_step(3)
        match = False
        for pod in tf_manager.get_vrouter_pods():
            host = pod.data['spec'].get('node_name')
            if host == hypervisor:
                match = True
                waiters.wait(lambda p: len(self.get_netns_list(p)) > 0,
                             predicate_args=[pod], timeout=180, interval=10,
                             timeout_msg="Network namespace list is not empty")
            else:
                waiters.wait(lambda p: len(self.get_netns_list(p)) == 0,
                             predicate_args=[pod], timeout=600, interval=10,
                             timeout_msg="Network namespace list is not empty")
        assert match, "Dedicated host wasn't found"

    @pytest.mark.dependency(depends=[f"TestTFNetNsAZ::test_netns_az[ENV_NAME={settings.ENV_NAME}]"])
    def test_netns_az_change(self, show_step, request, hypervisor,
                             tf_manager, openstack_client_manager):
        """ Check migration of created network namespaces

            Scenario:
                1. Create LB instance
                2. Change netns_availability_zone
                3. Check network namespaces moved to dedicated compute
        """
        show_step(1)
        ClusterCheckMosManager.created_stack_tf_lb(request, openstack_client_manager,
                                                   custom_params={"availability_zone": self.az})

        show_step(2)
        tf_manager.set_netns_availability_zone(self.default_az)

        show_step(3)
        pod = [pod for pod in tf_manager.get_vrouter_pods() if pod.data['spec'].get('node_name') == hypervisor][0]
        LOG.info(f"Wait until namespaces were removed from pod {pod.data['metadata']['name']} "
                 f"(Host: {pod.data['spec'].get('node_name')})")
        waiters.wait(lambda p: len(self.get_netns_list(p)) == 0,
                     predicate_args=[pod], timeout=120, interval=15,
                     timeout_msg="Network namespaces should be migrated to another hosts.")
        error = False
        cmd = ['/bin/sh', '-c', 'ip netns list']
        for pod in tf_manager.get_vrouter_pods():
            pod_name = pod.data['metadata']['name']
            host = pod.data['spec'].get('node_name')
            stdout = pod.exec(cmd, container='agent')
            out = stdout.splitlines()
            LOG.info(f"Network namespaces (vRouter: {pod_name}, host: {host}):\n"
                     f"{stdout}")
            if host == hypervisor:
                error = True if len(out) != 0 else False
        assert not error, ("Network namespaces should be migrated to another hosts.")

    def test_netns_az_negative(self, show_step, request, tf_manager,
                               openstack_client_manager):
        """ Check netns with multiple availability zones

            Scenario:
                1. Set netns_availability_zone with non-existent AZ
                2. Create LB instance
                3. Check network namespaces are exist
        """
        show_step(1)
        tf_manager.set_netns_availability_zone("si_test_negative")

        show_step(2)
        ClusterCheckMosManager.created_stack_tf_lb(request, openstack_client_manager,
                                                   custom_params={"availability_zone": self.az})

        show_step(3)
        timer = 90
        LOG.info(f"Wait {timer} seconds before check network namespaces")
        time.sleep(timer)
        match = False
        cmd = ['/bin/sh', '-c', 'ip netns list']
        for pod in tf_manager.get_vrouter_pods():
            pod_name = pod.data['metadata']['name']
            host = pod.data['spec'].get('node_name')
            stdout = pod.exec(cmd, container='agent')
            out = stdout.splitlines()
            LOG.info(f"Network namespaces (vRouter: {pod_name}, host: {host}):\n"
                     f"{stdout}")
            if len(out) > 0:
                match = True
        assert match, "Network namespaces weren't found"


@pytest.mark.usefixtures('restore_tfoperator_cr')
class TestTFDBRepair(object):
    """Check enable/disable TF DBRepair tool"""

    @pytest.fixture(scope='class', autouse=True)
    def check_requirements(self, tf_manager):
        tfoperator_version = tf_manager.tfoperator_version()
        if version.parse(tfoperator_version) < version.parse("0.17.0"):
            pytest.skip(f"TFOperator version {tfoperator_version} doesn't "
                        f"support dbRepair feature")

    def test_run_db_repair(self, show_step, tf_manager):
        """ Check TF DBRepair feature

            Scenario:
                1. Enable dbRepair feature
                2. Check TFDBRepair CR
                3. Check tf-dbrepair-job cronjob is created
                4. Trigger cron job
                5. Check cassandra cluster
                6. Suspend dbRepair cronjob
                7. Disable dbRepair feature and check cronjob is deleted
        """
        show_step(1)
        schedule = "0 12 * * 1,3,5"
        tf_manager.configure_dbrepair(state=True, schedule=schedule)

        show_step(2)
        waiters.wait_pass(tf_manager.tfdbrepair, predicate_kwargs={'read': True},
                          timeout=300, interval=10)
        waiters.wait(
            lambda: tf_manager.tfdbrepair(read=True).spec['schedule'] == schedule,
            timeout=300, interval=5
        )

        show_step(3)
        waiters.wait(tf_manager.is_tfdbrepair_cronjob_present, timeout=300, interval=10)
        cronjob = tf_manager.tfdbrepair_cronjob()
        waiters.wait(
            lambda: cronjob.read().spec.schedule == schedule,
            timeout=90, interval=5
        )

        show_step(4)
        job = tf_manager.trigger_tfdbrepair_cronjob()
        job.wait_finished(timeout=900)
        assert job.succeeded, "TF DBRepair job has finished unsuccessfully."

        show_step(5)
        assert tf_manager.check_cassandra_cluster(), "Cassandra cluster is unhealthy"

        show_step(6)
        suspend = True
        tf_manager.configure_dbrepair(suspend=suspend)
        waiters.wait(
            lambda: tf_manager.tfdbrepair(read=True).spec['suspend'] == suspend,
            timeout=300, interval=10
        )
        waiters.wait(
            lambda: cronjob.read().spec.suspend == suspend,
            timeout=90, interval=5
        )

        show_step(7)
        state = False
        tf_manager.configure_dbrepair(state=state)
        waiters.wait(
            lambda: tf_manager.tfdbrepair(read=True).spec['enabled'] == state,
            timeout=300, interval=10
        )

        waiters.wait(
            lambda: tf_manager.is_tfdbrepair_cronjob_present() is False,
            timeout=20, interval=10,
            timeout_msg="Cronjob wasn't deleted in timeout"
        )
