import exec_helpers
from exec_helpers import proc_enums
import yaml

from si_tests import logger
from si_tests import settings
from si_tests.utils import waiters, utils
from si_tests.managers.openstack_client_manager import OpenStackClientManager

LOG = logger.logger

ms_server_name = 'ms01'
ms_segment_name = "allcomputes"
ms_network_name = "masakari_net"
ms_subnet_name = "masakari_subnet"
ms_subnet_range = "10.1.2.0/24"
ms_volume_name = "masakari_volume"
test_cloud = "test"
extra_role = "global-secret-decoder"


def extend_roles_for_test_user(openstack_client_manager, action):
    raw = openstack_client_manager.client.exec(["cat", "/etc/openstack/clouds.yaml"])
    cloud_users = yaml.safe_load(raw)
    user_info = cloud_users.get("clouds", {}).get(test_cloud, {}).get("auth", [])
    if user_info:
        command = ["--user", user_info["username"], "--user-domain", user_info["user_domain_name"],
                   "--project-domain", user_info["project_domain_name"], "--project", user_info["project_name"],
                   extra_role]
        if action.lower() == "add":
            openstack_client_manager.role.add(command)
        elif action.lower() == "remove":
            openstack_client_manager.role.remove(command)
    else:
        raise Exception("There is no entry for " + test_cloud + " in the clouds.yaml")


def get_openstack_pids(ssh):
    openstack_processes = [
        "nova",
        "neutron",
        "cinder",
    ]
    pids = []
    res = ssh.check_call("ps -xaeo pid,command")
    for line in res.stdout:
        line = line.decode().strip()
        for os_pattern in openstack_processes:
            if os_pattern in line:
                pid = line.split(' ')[0]
                if pid:
                    pids.append(pid)
    return pids


def create_masakari_resources(ocm_admin, ocm_test, segment, vm):
    LOG.info("1.1 Create Masakari segment")
    ocm_admin.segment.create([segment, 'auto', 'compute'])

    # Extend test user rights for working with signed images
    extend_roles_for_test_user(ocm_admin, "add")

    # Add Hosts to Masakari segment
    LOG.info("1.2 Add Hosts to Masakari segment")
    for hs in ocm_admin.compute_service.list(['--service', 'nova-compute']):
        ocm_admin.segment.host_create([hs.get('Host'), 'compute', 'SSH', segment])

    LOG.info("Step 2: Boot VM on compute host")

    # Create a network
    LOG.info("2.1: Create network")
    ocm_test.network.create([ms_network_name])

    # Create a subnet
    LOG.info("2.2: Create subnetwork")
    ocm_test.subnet.create(["--subnet-range", ms_subnet_range, "--network", ms_network_name, ms_subnet_name])

    # Create a VM
    LOG.info("2.3: Boot VM on compute host")
    ocm_test.server.create([f"--image={ocm_admin.cirros_image_name}",
                            "--flavor", "m1.extra_tiny_test", "--network", ms_network_name, "--property",
                            "HA_Enabled=True", vm])

    # Check VM is running
    LOG.info("2.4: Check VM is running")
    waiters.wait(lambda: ocm_test.server.show([ms_server_name])["OS-EXT-STS:power_state"] in ["Running", 1],
                 timeout=60, interval=10)

    LOG.info("2.5: Create volume")
    ocm_test.volume.create(["--size", "1", ms_volume_name])

    LOG.info("2.6: Check volume is available")
    waiters.wait(lambda: ocm_test.volume.show([ms_volume_name])["status"] in ["available", 1],
                 timeout=60, interval=10)

    LOG.info("2.7: Attach volume to VM")
    ocm_test.server.add(["volume", ms_server_name, ms_volume_name])


def cleanup_after_test(ocm_admin, ocm_test, step):
    LOG.info("%s.1: Delete VM", step)
    ocm_test.server.delete([ms_server_name])
    LOG.info("%s.2: Wait VM is not present anymore", step)
    waiters.wait(lambda: ms_server_name not in [x["Name"] for x in ocm_test.server.list([])],
                 timeout=120, interval=10)
    LOG.info("%s.3: Delete volume", step)
    ocm_test.volume.delete([ms_volume_name])
    LOG.info("%s.4: Delete Masakari segment", step)
    ocm_admin.segment.delete([ms_segment_name])
    LOG.info("%s.5: Delete Masakari subnet", step)
    ocm_test.subnet.delete([ms_subnet_name])
    LOG.info("%s.6: Delete Masakari net", step)
    ocm_test.network.delete([ms_network_name])
    extend_roles_for_test_user(ocm_admin, "remove")


def ssh_node(hostname):
    with open(settings.NODES_INFO) as f:
        dictionary = yaml.safe_load(f)
    private_key = dictionary.get(hostname, {}).get('ssh')['private_key']
    assert private_key, "Private key is not defined"
    pkey = utils.get_rsa_key(private_key)
    node_ip = dictionary.get(hostname, {}).get('ip')['address']
    username = dictionary.get(hostname, {}).get('ssh')['username']
    auth = exec_helpers.SSHAuth(username=username, password='', key=pkey)
    ssh = exec_helpers.SSHClient(host=node_ip, port=22, auth=auth)
    ssh.logger.addHandler(logger.console)
    ssh.sudo_mode = True
    return ssh


def current_number_notifications(openstack_client_manager):
    return len(openstack_client_manager.notification.list([]))


def wait_masakari_notification(openstack_client_manager, host_uuid,
                               notification_type, remove_notification_num, status="finished",
                               timeout=300):
    def _wait_notification():
        notifications = openstack_client_manager.notification.list([])
        if len(notifications) <= remove_notification_num:
            return False

        for notification in notifications[:len(notifications) - remove_notification_num]:
            if (notification["source_host_uuid"] == host_uuid and
                    notification["type"] == notification_type and
                    notification["status"] == status):
                return True
        return False

    waiters.wait(_wait_notification, timeout=timeout, interval=30)


def test_host_monitor(openstack_client_manager):
    """Verify Masakari host monitor.
    https://mirantis.testrail.com/index.php?/cases/view/4963423

     Scenario:
         1. Create Masakari resources
         2. Boot VM on compute host
         3. Simulate host crush
         4. Wait k8s node is not ready
         5. Check Masakari notification
         6. Check that compute service is disabled on crashed compute
         7. Check VM host is changed
         8. Restore node
         9. Clean up

     """
    ssh = None
    hyper_indx = None
    try:
        ocm_admin = openstack_client_manager
        ocm_test = OpenStackClientManager(os_cloud=test_cloud)
        # Create Masakari segment
        LOG.info("Step 1: Create masakari resources")
        create_masakari_resources(ocm_admin, ocm_test,
                                  segment=ms_segment_name,
                                  vm=ms_server_name)

        # get values of the created Masakari resources
        server = ocm_test.server.show([ms_server_name])
        hostname = server["OS-EXT-SRV-ATTR:host"]
        segment_hosts = ocm_admin.segment.host_list([ms_segment_name])
        host_uuid = [x["uuid"] for x in segment_hosts
                     if x["name"] == hostname][0]
        for hv in ocm_admin.hypervisor.list([]):
            if hostname in hv["Hypervisor Hostname"]:
                hyper_indx = ocm_admin.hypervisor.list([]).index(hv)
                break

        current_notifications_num = current_number_notifications(ocm_admin)

        # Simulate host crush on the node where the instance is running
        LOG.info("Step 3: Simulate host crush")
        ssh = ssh_node(hostname)

        ssh.check_call("systemctl stop docker; sleep 30")
        pids = get_openstack_pids(ssh)
        for pid in pids:
            ssh.check_call(f"kill {pid}",
                           expected=(proc_enums.ExitCodes.EX_OK, proc_enums.ExitCodes.EX_ERROR))
        ssh.check_call(("nohup bash -c 'systemctl stop sshd; sleep 300; "
                        "systemctl start sshd' &"))

        # Wait k8s node is not ready anymore
        LOG.info("Step 4: Wait openstack compute is down.")
        waiters.wait(lambda: len([x["Host"] for x in ocm_admin.compute_service.list([])
                                  if (x["Host"] == hostname and x["State"] == "down")]) == 1,
                     timeout=300, interval=30)

        # Check Masakari notification
        LOG.info("Step 5: Check Masakari notification")
        wait_masakari_notification(ocm_admin, host_uuid,
                                   notification_type="COMPUTE_HOST",
                                   remove_notification_num=current_notifications_num,
                                   timeout=1200)

        # Check that compute service is disabled on crashed compute
        LOG.info("Step 6: Check that compute service is disabled on crashed "
                 "compute")
        assert hostname in [x["Host"] for x in ocm_admin.compute_service.list([])
                            if (x["Host"] == hostname and x["Status"] == "disabled")], \
            "compute service is NOT disabled on crashed compute"

        # Check VM's host is changed
        LOG.info("Step 7: Check VM's host is changed")
        assert hostname != ocm_test.server.show([ms_server_name])["OS-EXT-SRV-ATTR:host"], \
            "VM's host is NOT changed"
        assert ocm_test.server.show([ms_server_name])["OS-EXT-STS:power_state"] \
            in ["Running", 1], "VM's instance is NOT running"

    finally:
        # Restore Node and cleanup
        LOG.info("Step 8: Restore Node")
        if ssh is not None:
            LOG.info("8.1: Start docker on node")
            ssh.check_call("systemctl start docker")
        if hyper_indx is not None:
            LOG.info("8.2: Wait k8s node is ready again")
            waiters.wait(lambda: ocm_admin.hypervisor.list([])[hyper_indx]
                         ["State"] == "up", timeout=300, interval=30)
        LOG.info("8.3: Restore compute service on the crushed node")
        ocm_admin.compute_service.set(["--enable", hostname, "nova-compute"])
        LOG.info("Step 9: Cleanup")
        cleanup_after_test(ocm_admin, ocm_test, "9")


def test_instance_monitor(openstack_client_manager):
    """Verify Masakari instance monitor.
    https://mirantis.testrail.com/index.php?/cases/view/4963424

     Scenario:
         1. Create Masakari resources
         2. Boot VM on compute host
         3. Simulate instance crush
         4. Check Masakari notification
         5. Check VM instance is running
         6. Clean up

     """
    try:
        ocm_admin = openstack_client_manager
        ocm_test = OpenStackClientManager(os_cloud=test_cloud)

        # Create Masakari segment
        LOG.info("Step 1: Create masakari resources")
        create_masakari_resources(ocm_admin, ocm_test,
                                  segment=ms_segment_name,
                                  vm=ms_server_name)

        # get values of the created Masakari resources
        server = ocm_test.server.show([ms_server_name])
        hostname = server["OS-EXT-SRV-ATTR:host"]
        segment_hosts = ocm_admin.segment.host_list([ms_segment_name])
        host_uuid = [x["uuid"] for x in segment_hosts if x["name"] == hostname][0]
        current_notifications_num = current_number_notifications(ocm_admin)

        # Simulate instance crush
        LOG.info("Step 3: Simulate instance crush")
        ssh = ssh_node(hostname)
        ssh.check_call("kill -9 $(pidof qemu-system-x86_64)")

        # Check Masakari notification
        LOG.info("Step 4: Check Masakari notification")
        wait_masakari_notification(ocm_admin, host_uuid, notification_type="VM",
                                   remove_notification_num=current_notifications_num)

        # Wait VM is running again
        LOG.info("Step 5: Wait VM is running again")
        waiters.wait(lambda: ocm_test.server.show([ms_server_name])["OS-EXT-STS:power_state"]
                     in ["Running", 1], timeout=180, interval=30)

    finally:
        # Restore Node and cleanup
        LOG.info("Step 6: Restore Node and cleanup")
        cleanup_after_test(ocm_admin, ocm_test, "6")
