import pytest
import yaml
from si_tests import settings
from si_tests import logger
from si_tests.utils import waiters, utils

LOG = logger.logger

cluster_name = settings.TARGET_CLUSTER
namespace_name = settings.TARGET_NAMESPACE


def update_scale_bmcs(kaas_manager, ns, bm_hosts_data, scale_word):
    region = kaas_manager.get_mgmt_cluster().region_name

    node_scale = []
    for node in bm_hosts_data:
        if scale_word in node.get('si_roles', []):
            cred_name = node['name'] + '-cred'
            # Dirty hack, for not copy-paste bmh_name|cred_name across whole
            # test.
            bmh_name = utils.render_bmh_name(node['name'],
                                             cluster_name,
                                             node.get('bootUEFI', True),
                                             node['bmh_labels'],
                                             si_roles=node.get('si_roles',
                                                               False))
            _node = node.copy()
            _node.update({'bmh_name': bmh_name,
                          'cred_name': cred_name})
            node_scale.append(_node)
            secret_data = {
                "username": node.get('ipmi').get('username'),
                "password": node.get('ipmi').get('password')
            }

            if "kaas.mirantis.com/baremetalhost-credentials-name" in _node.get('bmh_annotations', {}):
                if _node['ipmi'].get('monitoringUsername', False):
                    secret_data.update({
                        "monitoringPassword": _node['ipmi']['monitoringPassword'],
                        "monitoringUsername": _node['ipmi']['monitoringUsername']
                    })
                if not kaas_manager.api.kaas_baremetalhostscredentials.present(name=cred_name,
                                                                               namespace=ns.name):
                    ns.create_baremetalhostcredential(name=cred_name, data=secret_data, region=region,
                                                      provider=settings.BAREMETAL_PROVIDER_NAME)
                else:
                    LOG.warning(f'bmhc: {cred_name} already exist, skipping')
            else:
                raise Exception("IPMI credentials supported only over baremetalhostcredentials")

            return node_scale


@pytest.mark.usefixtures("introspect_distribution_not_changed")
@pytest.mark.usefixtures("collect_downtime_statistics")     # Should be used if ALLOW_WORKLOAD == True
def test_add_bm_control_node(kaas_manager, ipmi_client):
    """Delete node with control label and add new one."""

    # Get namespace
    LOG.info("Namespace name - %s", namespace_name)
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)

    # Check or set customHostnamesEnabled flag in Cluster object to match settings.CUSTOM_HOSTNAMES
    cluster.set_custom_hostnames_enabled(flag=settings.CUSTOM_HOSTNAMES)

    # Create ssh key
    public_key_name = "{cluster_name}-key".format(
        cluster_name=cluster_name)
    LOG.info("Public key name - {public_key_name}".format(
        public_key_name=public_key_name))
    ns.get_publickey(public_key_name)

    # Gather nodes information from yaml file
    mgmt_version = \
        kaas_manager.get_mgmt_cluster().spec['providerSpec']['value']['kaas'][
            'release']
    child_data = utils.render_child_data(kaas_manager.si_config, {'mgmt_version': mgmt_version})

    bm_hosts_data = child_data.get('nodes')
    ns.create_bm_statics(child_data)
    # Delete machine(s) with control label
    machines_for_delete = [machine for machine in ns.get_machines()
                           if 'nodefordelete' in machine.name]
    assert len(machines_for_delete) >= 1, "Not found Machine(s) with " \
                                          "'nodefordelete' si-role"
    bmh_for_delete = [bmh for bmh in ns.get_baremetalhosts()
                      if 'nodefordelete' in bmh.name]

    assert len(bmh_for_delete) >= 1, "Not found BMH(s) with" \
                                     "'nodefordelete' si-role"

    bmh_names_for_delete = [bmh.name for bmh in bmh_for_delete]
    LOG.info(f"Machines for delete: {[m.name for m in machines_for_delete]}")
    for machine in machines_for_delete:
        LOG.info(f"Delete Machine: {machine.name}")
        machine.delete()
        cluster.wait_machine_deletion(machine.name, retries=60)

    if machines_for_delete:
        LOG.info(f"Waiting bmh(s):\n{bmh_names_for_delete}\nto be in "
                 f"'ready' state")
        ns.wait_baremetalhosts_statuses(nodes=bmh_names_for_delete,
                                        wait_status='ready',
                                        retries=30,
                                        interval=60)
    LOG.info(f"BMH for delete: {bmh_names_for_delete}")
    bmcs_data = []
    for bmh in bmh_for_delete:
        bmh_name = bmh.name
        bmcs_data.append(cluster.get_bmc_data(bmh))
        ns.delete_baremetalhost(name=bmh_name)
    ns.wait_all_bmhs_deletion(bmhs_for_delete=bmh_for_delete, wait_bmh_cred=True, bm_hosts_data=bm_hosts_data)
    if settings.BM_CHECK_BMC_STATUS_AFTER_DELETE:
        for bmc_data in bmcs_data:
            power_on = ipmi_client.get_power_on(ipmi_host=bmc_data['host'],
                                                ipmi_user=bmc_data['username'],
                                                ipmi_password=bmc_data['password'],
                                                ipmi_port=bmc_data['port'])
            LOG.info(f"Chassis power on status for {bmc_data['host']}:{bmc_data['port']} is: {power_on}")
            assert not power_on, \
                "Chassis status is not powered off. "
    LOG.info(f"Next BMHs were deleted successfully:\n{bmh_for_delete}")

    controller_scale = update_scale_bmcs(kaas_manager=kaas_manager,
                                         ns=ns,
                                         bm_hosts_data=bm_hosts_data,
                                         scale_word='child-scale-controller')

    assert len(controller_scale) >= 1, "BMH with si_roles " \
                                       "'child-scale-controller' not found"
    new_nodes_name = []
    for node in controller_scale:
        bmh_name = node['bmh_name']
        new_nodes_name.append(bmh_name)
        ns.create_baremetalhost(bmh_name=bmh_name,
                                bmh_secret=node['cred_name'],
                                bmh_mac=node['networks'][0]['mac'],
                                bmh_ipmi=node['ipmi'],
                                hardwareProfile=node.get('hardwareProfile',
                                                         False),
                                labels=node['bmh_labels'],
                                annotations=node.get('bmh_annotations', {}),
                                bootUEFI=node.get('bootUEFI', True),
                                bmhi_credentials_name=node['cred_name'])
    LOG.info(f"New BMH nodes:\n{new_nodes_name}")
    ns.wait_baremetalhosts_statuses(nodes=new_nodes_name,
                                    wait_status='ready',
                                    retries=20,
                                    interval=60)

    new_machine_names = []
    # Create control plane machines
    release_name = cluster.clusterrelease_version
    for node in controller_scale:
        distribution = utils.get_distribution_for_node(kaas_manager, node, release_name)
        LOG.info('Create BareMetal machine')
        custom_bmhp = False
        if node.get('bmh_profile'):
            custom_bmhp = {
                'namespace': namespace_name,
                'name': node['bmh_profile']
            }
        bm_machine = cluster.create_baremetal_machine(
            genname=node['bmh_name'],
            node_pubkey_name=public_key_name,
            matchlabels={'kaas.mirantis.com/'
                         'baremetalhost-id':
                             node['bmh_labels']
                             ['kaas.mirantis.com/baremetalhost-id']},
            baremetalhostprofile=custom_bmhp,
            l2TemplateSelector=node.get('l2TemplateSelector', dict()),
            labels=node['machine_labels'],
            node_labels=node.get('node_labels', {}),
            distribution=distribution,
        )
        new_machine_names.append(bm_machine.name)

    # Wait Baremetal hosts be provisioned
    ns.wait_baremetalhosts_statuses(nodes=new_nodes_name,
                                    wait_status='provisioned',
                                    retries=30,
                                    interval=90)

    # Waiting for machines are Ready and helmbundles are deployed
    cluster.check.check_machines_status()
    cluster.check.check_cluster_nodes()
    cluster.check.check_k8s_nodes()
    cluster.check.check_cluster_readiness()
    cluster.check.check_helmbundles()
    cluster.check.check_deploy_stage_success()

    new_machines = [cluster.get_machine(name=machine_name) for machine_name in new_machine_names]
    # Check that Machine hostname is created with respect to Cluster flag 'customHostnamesEnabled'
    cluster.check.check_custom_hostnames_on_machines(machines=new_machines)

    # Collecting artifacts
    cluster.store_k8s_artifacts()

    if cluster.lcm_type_is_ucp:
        # Check/wait for correct docker service replicas in cluster
        cluster.check.check_actual_expected_docker_services()
    cluster.check.check_k8s_pods()
    cluster.check.check_actual_expected_pods()


@pytest.mark.usefixtures("introspect_distribution_not_changed")
@pytest.mark.usefixtures("collect_downtime_statistics")     # Should be used if ALLOW_WORKLOAD == True
def test_add_delete_bm_worker_node(kaas_manager, ipmi_client):
    """Delete one worker node."""

    # Get namespace
    LOG.info(f"Namespace name: {namespace_name}")
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)

    # Check or set customHostnamesEnabled flag in Cluster object to match settings.CUSTOM_HOSTNAMES
    cluster.set_custom_hostnames_enabled(flag=settings.CUSTOM_HOSTNAMES)

    # Create ssh key
    public_key_name = "{cluster_name}-key".format(
        cluster_name=cluster_name)
    LOG.info("Public key name - {public_key_name}".format(
        public_key_name=public_key_name))
    ns.get_publickey(public_key_name)

    mgmt_version = \
        kaas_manager.get_mgmt_cluster().spec['providerSpec']['value']['kaas'][
            'release']
    # Gather nodes information from yaml file
    child_data = utils.render_child_data(kaas_manager.si_config, {'mgmt_version': mgmt_version})
    ns.create_bm_statics(child_data)
    bm_hosts_data = child_data['nodes']
    worker_scale = update_scale_bmcs(kaas_manager=kaas_manager,
                                     ns=ns,
                                     bm_hosts_data=bm_hosts_data,
                                     scale_word='nodeforscale')

    assert len(worker_scale) >= 1, "BMH with si_roles " \
                                   "'nodeforscale' not found"

    new_nodes_name = []
    for node in worker_scale:
        bmh_name = node['bmh_name']
        new_nodes_name.append(bmh_name)
        ns.create_baremetalhost(bmh_name=bmh_name,
                                bmh_secret=node['cred_name'],
                                bmh_mac=node['networks'][0]['mac'],
                                bmh_ipmi=node['ipmi'],
                                hardwareProfile=node.get('hardwareProfile',
                                                         False),
                                labels=node['bmh_labels'],
                                annotations=node.get('bmh_annotations', {}),
                                bootUEFI=node.get('bootUEFI', True),
                                bmhi_credentials_name=node['cred_name'])
    LOG.info(f"New BMH nodes:\n{new_nodes_name}")

    ns.wait_baremetalhosts_statuses(nodes=new_nodes_name,
                                    wait_status='ready',
                                    retries=20,
                                    interval=60)

    # Create worker machines
    new_machine_names = []
    release_name = cluster.clusterrelease_version
    for node in worker_scale:
        distribution = utils.get_distribution_for_node(kaas_manager, node, release_name)
        LOG.info('Create BareMetal machine')
        custom_bmhp = False
        if node.get('bmh_profile'):
            custom_bmhp = {
                'namespace': namespace_name,
                'name': node['bmh_profile']
            }
        bm_machine = cluster.create_baremetal_machine(
            genname=node['bmh_name'],
            node_pubkey_name=public_key_name,
            matchlabels={'kaas.mirantis.com/'
                         'baremetalhost-id':
                             node['bmh_labels']
                             ['kaas.mirantis.com/baremetalhost-id']},
            baremetalhostprofile=custom_bmhp,
            l2TemplateSelector=node.get('l2TemplateSelector', dict()),
            labels=node['machine_labels'],
            node_labels=node.get('node_labels', {}),
            distribution=distribution,
        )
        new_machine_names.append(bm_machine.name)

    # Wait Baremetal hosts be provisioned
    ns.wait_baremetalhosts_statuses(nodes=new_nodes_name,
                                    wait_status='provisioned',
                                    retries=30,
                                    interval=90)

    # Waiting for cluster,pods,hb are Ready after machines were added
    cluster.check.check_machines_status()
    cluster.check.check_cluster_nodes()
    cluster.check.check_k8s_nodes()
    cluster.check.check_cluster_readiness()
    cluster.check.check_helmbundles()

    cluster.check.check_k8s_pods()

    new_machines = [cluster.get_machine(name=machine_name) for machine_name in new_machine_names]
    # Check that Machine hostname is created with respect to Cluster flag 'customHostnamesEnabled'
    cluster.check.check_custom_hostnames_on_machines(machines=new_machines)

    # Delete machine(s)
    # TODO(alexz) Reference to nodename- not a best choice here. probably
    #  we should add logic into nodeLabels
    machines_for_delete = [machine for machine in ns.get_machines()
                           if 'nodeforscale' in machine.name]
    for machine in machines_for_delete:
        machine.delete()
        cluster.wait_machine_deletion(machine.name, interval=60)
    ns.wait_baremetalhosts_statuses(nodes=new_nodes_name,
                                    wait_status='ready',
                                    retries=30,
                                    interval=60)

    # Waiting for cluster,pods,hb are Ready after machines were deleted
    cluster.check.check_machines_status()
    cluster.check.check_cluster_nodes()
    cluster.check.check_k8s_nodes()
    cluster.check.check_cluster_readiness()
    cluster.check.check_helmbundles()

    cluster.check.check_k8s_pods()

    # Recreate BMH and check it has been inspected
    bmh_for_delete = [bmh for bmh in ns.get_baremetalhosts()
                      if 'nodeforscale' in bmh.name]
    LOG.info("Recreating BHM(s) and check it has been inspected")
    bmcs_data = []
    for bmh in bmh_for_delete:
        bmh_name = bmh.name
        bmcs_data.append(cluster.get_bmc_data(bmh))
        ns.delete_baremetalhost(name=bmh_name)
    ns.wait_all_bmhs_deletion(bmhs_for_delete=bmh_for_delete, wait_bmh_cred=True, bm_hosts_data=bm_hosts_data)
    if settings.BM_CHECK_BMC_STATUS_AFTER_DELETE:
        for bmc_data in bmcs_data:
            power_on = ipmi_client.get_power_on(ipmi_host=bmc_data['host'],
                                                ipmi_user=bmc_data['username'],
                                                ipmi_password=bmc_data['password'],
                                                ipmi_port=bmc_data['port'])
            LOG.info(f"Chassis power on status for {bmc_data['host']}:{bmc_data['port']} is: {power_on}")
            assert not power_on, \
                "Chassis status is not powered off. "
    LOG.info(f"Next BMH(s) were deleted successfully\n:{bmh_for_delete}")
    LOG.info("Recreating Baremetal host(s)")
    worker_scale = update_scale_bmcs(kaas_manager=kaas_manager,
                                     ns=ns,
                                     bm_hosts_data=bm_hosts_data,
                                     scale_word='nodeforscale')

    region = kaas_manager.get_mgmt_cluster().region_name

    for node in worker_scale:
        cred_name = node['name'] + '-cred'
        bmh_name = node['bmh_name']
        # Wait for secret deletion after node was deleted
        waiters.wait(lambda: not kaas_manager.api.kaas_baremetalhostscredentials.present(name=cred_name,
                                                                                         namespace=ns),
                     timeout_msg=f"Secret {cred_name} was not deleted succesfully")

        secret_data = {
                "username": node['ipmi']['username'],
                "password": node['ipmi']['password']
        }

        if "kaas.mirantis.com/baremetalhost-credentials-name" in node.get('bmh_annotations', {}):
            if node['ipmi'].get('monitoringUsername', False):
                secret_data.update({
                    "monitoringPassword": node['ipmi']['monitoringPassword'],
                    "monitoringUsername": node['ipmi']['monitoringUsername']
                })
        else:
            raise Exception("IPMI credentials supported only over baremetalhostcredentials")

        ns.create_baremetalhostcredential(name=cred_name, data=secret_data, region=region,
                                          provider=settings.BAREMETAL_PROVIDER_NAME)

        new_nodes_name.append(bmh_name)
        ns.create_baremetalhost(bmh_name=bmh_name,
                                bmh_secret=node['cred_name'],
                                bmh_mac=node['networks'][0]['mac'],
                                bmh_ipmi=node['ipmi'],
                                hardwareProfile=node.get('hardwareProfile',
                                                         False),
                                labels=node['bmh_labels'],
                                annotations=node.get('bmh_annotations', {}),
                                bootUEFI=node.get('bootUEFI', True),
                                bmhi_credentials_name=node['cred_name'])

    ns.wait_baremetalhosts_statuses(nodes=new_nodes_name,
                                    wait_status='ready',
                                    retries=30,
                                    interval=60)
    LOG.info("Baremetal host(s) is succesfully recreated")

    # Refresh the expected objects stored in the cluster
    cluster._refresh_expected_objects()
    if cluster.lcm_type_is_ucp:
        ucp_worker_agent_name = cluster.check.get_ucp_worker_agent_name()
        # Check/wait for correct docker service replicas in cluster
        cluster.check.check_actual_expected_docker_services(
            changed_after_upd={'ucp-worker-agent-x': ucp_worker_agent_name})
    # Comparing pods moved to last step due to PRODX-2929
    cluster.check.check_actual_expected_pods()
    cluster.check.check_cluster_readiness()


def test_negative_add_fake_node_then_delete(kaas_manager):
    """Negative test: add and delete node

    * Create baremetalhost with invalid data (by label 'si-role/fake-node')
    * Check that baremetalhost created and wait error state status
    * Create baremetal machine
    * Delete machine and wait for it to be deleted
    * Delete baremetalhost and wait for it to be deleted
    """

    region = kaas_manager.get_mgmt_cluster().region_name
    node_data = """\
nodes:
  - name: fake-node
    bmh_annotations:
      "kaas.mirantis.com/baremetalhost-credentials-name": "fake-node-cred"
    bmh_labels:
"""
    if region:
        node_data += """\
      "kaas.mirantis.com/region": """+region

    node_data += """\
      "cluster.sigs.k8s.io/cluster-name": """+settings.TARGET_CLUSTER+"""
      "kaas.mirantis.com/provider": "baremetal"
      "hostlabel.bm.kaas.mirantis.com/worker": "worker"
      "kaas.mirantis.com/baremetalhost-id": fake-node
      "si-role/fake-node": "true"
    machine_labels:
"""
    if region:
        node_data += """\
      "kaas.mirantis.com/region": """+region

    node_data += """\
      "cluster.sigs.k8s.io/cluster-name": """+settings.TARGET_CLUSTER+"""
      "kaas.mirantis.com/provider": "baremetal"
      "hostlabel.bm.kaas.mirantis.com/worker": "worker"
      "si-role/fake-node": "true"
    bmh_profile: cluster-default
    bootUEFI: false
    ipmi:
      ipmi_ip: 10.20.30.11
      username: ZmFrZXVzZXIK
      password: ZmFrZXVzZXIK
    networks:
      - mac: ff:aa:cc:bc:fc:ff
    l2TemplateSelector:
      name: "default-region-one"
"""

    # Get namespace
    LOG.info("Namespace name - %s", namespace_name)
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)

    # Create ssh key
    public_key_name = f"{cluster_name}-key"
    LOG.info(f"Public key name - {public_key_name}")
    ns.get_publickey(public_key_name)

    mgmt_version = kaas_manager.get_mgmt_cluster().spec['providerSpec']['value']['kaas']['release']
    LOG.info(f"KaaS mgmt version is:{mgmt_version}")

    # Gather nodes information from yaml file
    child_data = utils.render_child_data(kaas_manager.si_config, {'mgmt_version': mgmt_version})
    ns.create_bm_statics(child_data)
    bm_hosts_data = yaml.load(node_data, Loader=yaml.SafeLoader)

    # Create BMH with invalid data and check it has been in error state
    LOG.info("Create Baremetal host(s)/machine(s) with invalid data")
    machines_to_create = [node for node in bm_hosts_data['nodes'] if 'si-role/fake-node' in node.get('bmh_labels', {})]
    new_machine_names = []

    for node in machines_to_create:
        cred_name = node['name'] + '-cred'

        secret_data = {
            "username": node['ipmi']['username'],
            "password": node['ipmi']['password']
        }
        if "kaas.mirantis.com/baremetalhost-credentials-name" in node.get('bmh_annotations', {}):
            if node['ipmi'].get('monitoringUsername', False):
                secret_data.update({
                    "monitoringPassword": node['ipmi']['monitoringPassword'],
                    "monitoringUsername": node['ipmi']['monitoringUsername']
                })
            if not kaas_manager.api.kaas_baremetalhostscredentials.present(name=cred_name, namespace=ns.name):
                ns.create_baremetalhostcredential(name=cred_name, data=secret_data, region=region,
                                                  provider=settings.BAREMETAL_PROVIDER_NAME)
            else:
                LOG.warning(f'bmhc: {cred_name} already exist, skipping')
        else:
            raise Exception("IPMI credentials supported only over baremetalhostcredentials")

        # Dirty hack, for not copy-paste bmh_name|cred_name across whole test.
        bmh_name = utils.render_bmh_name(
            node['name'], cluster_name, node.get('bootUEFI', True),
            node['bmh_labels'], si_roles=node.get('si_roles', False))
        node.update({'bmh_name': bmh_name, 'cred_name': cred_name})
        new_machine_names.append(bmh_name)
        ns.create_baremetalhost(
            bmh_name=bmh_name, bmh_secret=cred_name, bmh_mac=node['networks'][0]['mac'], bmh_ipmi=node['ipmi'],
            hardwareProfile=node.get('hardwareProfile', False), labels=node['bmh_labels'],
            bootUEFI=node.get('bootUEFI', True), annotations=node.get('bmh_annotations', {}),
            bmhi_credentials_name=cred_name)

    if machines_to_create:
        LOG.info(f"Wating bmh(s):\n{new_machine_names}\nto be in 'registering' state")
        ns.wait_baremetalhosts_statuses(nodes=new_machine_names, wait_status='registering', retries=10, interval=30)

        def wait_operational_status_error(bmhs_names):
            for bmh_name in bmhs_names:
                bmh = ns.get_baremetalhost(name=bmh_name)
                if bmh.read().status.get('operationalStatus', 'Nil') != 'error':
                    LOG.warning(f'Baremetal host {bmh_name} should be in operationalState: error')
                    return False
            return True

        LOG.info(f"Wating bmh(s):\n{new_machine_names}\noperationalStatus to be in 'error' state")
        waiters.wait(lambda: wait_operational_status_error(new_machine_names), timeout=120,
                     timeout_msg=f"Baremetal host should be in 'error' operationalState {new_machine_names}")

    # Create worker machines
    for node in machines_to_create:
        LOG.info('Create BM machine')
        custom_bmhp = False
        if node.get('bmh_profile'):
            custom_bmhp = {'namespace': namespace_name, 'name': node['bmh_profile']}
        cluster.create_baremetal_machine(
            genname=node['bmh_name'],
            node_pubkey_name=public_key_name,
            matchlabels={'kaas.mirantis.com/'
                         'baremetalhost-id': node['bmh_labels']['kaas.mirantis.com/baremetalhost-id']},
            baremetalhostprofile=custom_bmhp,
            labels=node['machine_labels'],
            l2TemplateSelector=node.get('l2TemplateSelector', dict()),
            node_labels=node.get('node_labels', {}),
            distribution=node.get('distribution', None),)

    if machines_to_create:
        LOG.info(f"Waiting bmh(s):\n{new_machine_names}\nto be in 'registering' state")
        ns.wait_baremetalhosts_statuses(nodes=new_machine_names, wait_status='registering', retries=60, interval=60)
        LOG.info('Baremetal machine(s) is succesfully created')

    # Delete machine(s)
    machines_for_delete = []
    for machine in ns.get_machines():
        machine_data = machine.read().to_dict()
        labels = machine_data.get('metadata', {}).get('labels', {})
        if 'si-role/fake-node' in labels:
            machines_for_delete.append(machine)
    assert len(machines_for_delete) >= 1, "Not found Machine(s) with label 'si-role/fake-node'"
    bmh_for_delete = []
    for bmh in ns.get_baremetalhosts():
        labels = bmh.data.get('metadata', {}).get('labels', {})
        if 'si-role/fake-node' in labels:
            bmh_for_delete.append(bmh)
    assert len(bmh_for_delete) >= 1, "Not found BMH(s) with label 'si-role/fake-node'"

    bmh_names_for_delete = [bmh.name for bmh in bmh_for_delete]

    LOG.info("! Machine deletion with associated baremetalhost with error state available after 2-24-4")
    LOG.debug("Machines for delete: %s", ','.join([m.name for m in machines_for_delete]))
    for machine in machines_for_delete:
        LOG.info(f"Delete Machine: {machine.name}")
        machine.delete()
        cluster.wait_machine_deletion(machine.name, retries=10, interval=10)

    if machines_for_delete:
        LOG.info(f"Waiting bmh(s):\n{bmh_names_for_delete}\nto be in 'registering' state")
        ns.wait_baremetalhosts_statuses(nodes=bmh_names_for_delete, wait_status='registering', retries=10, interval=10)

    LOG.info("Delete BMH(s) and check it has been deleted")
    LOG.debug("BMH for delete: %s", ','.join(bmh_names_for_delete))
    bmcs_data = []
    for bmh in bmh_for_delete:
        bmcs_data.append(cluster.get_bmc_data(bmh))
        bmh_name = bmh.name
        ns.delete_baremetalhost(name=bmh_name)
    ns.wait_all_bmhs_deletion(bmhs_for_delete=bmh_for_delete, wait_bmh_cred=True, bm_hosts_data=bm_hosts_data['nodes'])
    LOG.info(f"Next BMHs were deleted successfully:\n{bmh_for_delete}")

    # Check that all machines in 'Ready' status
    cluster.check.check_machines_status()
    cluster.check.check_cluster_nodes()
    cluster.check.check_cluster_readiness()
    cluster.check.check_helmbundles()
    cluster.check.check_k8s_nodes()

    cluster.check.check_bmh_inventory_presense()
