import ast

import pytest

from si_tests import logger, settings
from si_tests.managers import openstack_manager, tungstenfafric_manager

LOG = logger.logger

cluster_name = settings.TARGET_CLUSTER
namespace_name = settings.TARGET_NAMESPACE
tfoperator_name = settings.TF_OPERATOR_CR_NAME


@pytest.mark.usefixtures('log_method_time')
@pytest.mark.usefixtures("collect_downtime_statistics")  # Should be used if ALLOW_WORKLOAD == True
def test_replace_failed_tf_control(kaas_manager, show_step):
    """Replace TF control plane node after the outage.

    Scenario:
        1. Get info about replace node
        2. Identity of cassandra services
        3. Get Cassandra pod IPs from the node to be removed
        4. Simulate power outage (Hard power off)
        5. Wait for NotReady node state
        6. Replace the powered off TF control Machine and it's BMH/BMHI
        7. Check Machines and Nodes
        8. Remove old Cassandra hosts from the cluster configuration
        9. Force delete all pods and PVC from replaced node
        10. Check child cluster conditions
        11. Check openstack resources
    """

    # Get namespace
    LOG.info(f"Namespace name: {namespace_name}")
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)

    # Check or set customHostnamesEnabled flag in Cluster object to match settings.CUSTOM_HOSTNAMES
    cluster.set_custom_hostnames_enabled(flag=settings.CUSTOM_HOSTNAMES)

    # Write kubeconfig
    child_kubeconfig_name, child_kubeconfig = cluster.get_kubeconfig_from_secret()
    with open('child_conf', 'w') as f:
        f.write(child_kubeconfig)
    tf_manager = tungstenfafric_manager.TFManager(kubeconfig='child_conf')
    os_manager = openstack_manager.OpenStackManager(kubeconfig='child_conf')

    show_step(1)

    # get cluster machine (outage candidate)
    machines = cluster.get_machines(machine_status='Ready')
    tf_control = []

    for w in machines:
        if ("si-role/replacetfcontrol" in w.machinelabels and
                [n for n in w.nodeLabels if 'tfcontrol' in n.values()]):
            tf_control.append(w)

    assert len(tf_control) == 1, (
        f"BM Machine not found on the cluster, "
        f"or Machine not in Ready status. Found {len(tf_control)} Machines: {tf_control}")
    target_node = tf_control[0]
    instance_name = target_node.status['instanceName']

    show_step(2)
    # Check if target node has cassandra db (config/analytics) to get the IP
    # addresses of the Cassandra pods that run on the node to be replaced:
    tfconfigdb = False
    tfanalyticsdb = False
    for label in target_node.nodeLabels:
        if 'tfconfigdb' in label.values():
            tfconfigdb = True
            LOG.info("Cassandra tfconfigdb detected")
        if ('tfanalyticsdb' in label.values() and
                tf_manager.is_analytics_enabled()):
            tfanalyticsdb = True
            LOG.info("Cassandra tfanalyticsdb detected")

    show_step(3)

    k8s_node = target_node.get_k8s_node()
    LOG.info(f"K8s node: {k8s_node.name}")
    old_node_k8s_name = k8s_node.name

    # Get Cassandra pod IPs from the node to be removed.
    ip_configdb = None
    ip_analyticsdb = None
    if tfconfigdb:
        cdb_pods = cluster.k8sclient.pods.list_all(
            namespace=tf_manager.tf_namespace,
            name_prefix="tf-cassandra-config"
        )
        pod = [p for p in cdb_pods if p.data['spec']['node_name'] ==
               old_node_k8s_name]
        assert len(pod) == 1, "Can't get IP of cassandra configdb pod from " \
                              "deleted node"
        pod = pod[0]
        cdb_pods.remove(pod)
        ip_configdb = pod.data['status']['pod_ip']
        LOG.info(f"Cassandra configdb node to be removed: {ip_configdb}")
    if tfanalyticsdb:
        adb_pods = cluster.k8sclient.pods.list_all(
            namespace=tf_manager.tf_namespace,
            name_prefix="tf-cassandra-analytics"
        )
        pod = [p for p in adb_pods if p.data['spec']['node_name'] ==
               old_node_k8s_name]
        assert len(pod) == 1, "Can't get IP of cassandra analyticsdb pod " \
                              "from deleted node"
        pod = pod[0]
        adb_pods.remove(pod)
        ip_analyticsdb = pod.data['status']['pod_ip']
        LOG.info(f"Cassandra analyticsdb node to be removed: {ip_analyticsdb}")

    # Get bmh and power off
    # Wait form k8s node status NotReady
    # Remove all labels from k8s node
    show_step(4)
    target_node.set_baremetalhost_power(online=False)

    show_step(5)
    LOG.info("Delete, and start waiting k8s node status `NotReady`")
    cluster.check.wait_k8s_node_status(old_node_k8s_name, expected_status='NotReady')

    show_step(6)
    new_machine = cluster.day2operations.replace_baremetal_machine_and_bmh(
        target_node, machine_deletion_policy="unsafe")

    show_step(7)
    # Waiting for cluster,pods,hb are Ready after machines were added
    cluster.check.check_machines_status()
    cluster.check.check_cluster_nodes()
    cluster.check.check_k8s_nodes()
    cluster.check.check_helmbundles()

    # Check machine runtime
    if settings.DESIRED_RUNTIME:
        machines = cluster.get_machines()
        replaced_machine_for_runtime_check = [machine for machine in machines
                                              if "si-role/replacetfcontrol" in machine.machinelabels][0]
        cluster.check.compare_machines_runtime_with_desired([replaced_machine_for_runtime_check])

    # Check that Machine hostname is created with respect to Cluster flag 'customHostnamesEnabled'
    cluster.check.check_custom_hostnames_on_machines(machines=[cluster.get_machine(name=(new_machine.name))])

    new_node_k8s_name = new_machine.get_k8s_node().name
    LOG.info(f"New node k8s name: {new_node_k8s_name}")

    if tf_manager.is_tf_cli_enabled():
        LOG.info("Optional step. Trying to remove deleted node from TF configuration using TF cli:")
        node_types = ["config-node", "config-database-node"]
        if tfanalyticsdb:
            node_types.append("analytics-node")
        for type in node_types:
            LOG.info(f"Searching for kaas node {instance_name} in {type}s")
            objects = str.split(tf_manager.exec_tf_cli_pod(f"tf-api-cli ls {type} 2>/dev/null"))
            LOG.info(f"Found {len(objects)} objects in configuration.")
            for obj in objects:
                LOG.info(f"Processing {obj}...")
                details = tf_manager.exec_tf_cli_pod(f"tf-api-cli cat {obj} 2>/dev/null")
                data = ast.literal_eval(details)
                if instance_name in data['name']:
                    stdout = tf_manager.exec_tf_cli_pod(f"tf-api-cli -- rm --force {obj}")
                    LOG.info(f"{obj} was deleted from TF configuration:\n{stdout}")
                    break
    else:
        LOG.info("Optional step with remove node from TF configuration was skipped "
                 "because TF tool cli pod isn't enabled.")

    show_step(8)
    if tfconfigdb:
        cmd = ['/bin/sh', '-c', f"id=$(nodetool status | grep {ip_configdb} "
                                "| awk '{{ print $7 }}') && nodetool "
                                "removenode force $id"]
        resutl_cmd = cdb_pods[0].exec(cmd, container='cassandra')
        LOG.info(f"Result of removing configdb node from cassandra cluster:"
                 f"\n{resutl_cmd}")
    if tfanalyticsdb:
        cmd = ['/bin/sh', '-c', f"id=$(nodetool status | grep {ip_analyticsdb} "
                                "| awk '{{ print $7 }}') && nodetool "
                                "removenode force $id"]
        resutl_cmd = adb_pods[0].exec(cmd, container='cassandra')
        LOG.info(f"Result of removing analyticsdb node from cassandra cluster:"
                 f"\n{resutl_cmd}")

    show_step(9)  # Force delete all pv/pvc from old node
    cluster.day2operations.cleanup_pods_and_pvc_from_k8s_node(old_node_k8s_name)

    show_step(10)
    cluster.check.check_cluster_readiness()
    cluster.check.check_k8s_pods()
    cluster.check.check_deploy_stage_success()
    cluster.check.check_diagnostic_cluster_status()

    LOG.info("Check TFOperator health status")
    tf_manager.wait_tfoperator_healthy(timeout=900)
    tf_manager.wait_tf_controllers_healthy()

    show_step(11)
    LOG.info("Check Openstack resources statuses")
    os_manager.wait_all_osdpl_children_status()
    os_manager.wait_openstackdeployment_health_status()
    os_manager.wait_all_os_pods()

    cluster.check.check_bmh_inventory_presense()
