import pytest
from si_tests import logger
from si_tests import settings

from si_tests.deployments.utils import kubectl_utils
from si_tests.utils import waiters

LOG = logger.logger


def get_reboot_warning(m_list, namespace_name):
    """Get reboot warning
    Function for get nodes warning from cli
    we need check that kubectl display warning for nodes (reboot for example)
    """
    kubectl = kubectl_utils.Kubectl()
    nodes_warning = {}
    out = kubectl.get('machines', '-o yaml', namespace_name).result_yaml
    for i in out['items']:
        name = i['metadata']['name']
        warning = i['status']['providerStatus'].get('warnings', [])
        if name in m_list and not nodes_warning.get(name):
            nodes_warning[name] = ("Scheduled for a reboot" in warning or "Reboot is in progress" in warning)
    LOG.info("machines without reboot warning:\n" + str([k for k, v in nodes_warning.items() if not v]))
    return all(nodes_warning.values())


@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.TARGET_CLUSTER}"])
@pytest.mark.usefixtures('mcc_loadtest_prometheus')
@pytest.mark.usefixtures('mcc_loadtest_grafana')
@pytest.mark.usefixtures('mcc_loadtest_alerta')
@pytest.mark.usefixtures('mcc_loadtest_kibana')
@pytest.mark.usefixtures('mcc_loadtest_alertmanager')
def test_graceful_reboot(kaas_manager, _, show_step):
    """ Graceful reboot of all machines of the cluster
    Scenario:
        1. Check init state cluster
        2. Create Graceful Reboot Request
        3. Wait for nodes warning: 'Scheduled for a reboot' or 'Reboot is in progress'
        4. Check all machines are rebooted
        5. Check return cluster to init state
    """
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)

    show_step(1)
    LOG.info(f"Check init state on the {cluster._cluster_type} cluster {cluster.namespace}/{cluster.name}")
    cluster.check.check_machines_status()
    cluster.check.check_cluster_readiness()
    cluster.check.check_k8s_nodes()
    cluster.check.wait_graceful_reboot_request(expected_status=False)

    show_step(2)
    LOG.info(f"Creating Graceful Reboot Request for all machines in cluster {cluster.namespace}/{cluster.name}")
    m_list = cluster.get_machines_names()
    ns.create_gracefulrebootrequest_object(cluster.name, namespace_name, m_list)

    show_step(3)
    waiters.wait(
        lambda: get_reboot_warning(m_list, namespace_name),
        timeout=3600, interval=10,
        timeout_msg="Wait for 'reboot' warning for all machines")

    cluster.check.wait_graceful_reboot_request(expected_status=True)

    show_step(4)
    LOG.info('Waiting for all machines to reboot')
    approx_node_graceful_reboot_time = settings.APPROX_NODE_GRACEFUL_REBOOT_TIME
    boot_time_dict = cluster.get_boot_time_dict(exclude_bastion=True)
    machines_number = len(boot_time_dict.keys())
    # Rebooting BM machines takes about 6-9 minutes
    machines_reboot_timeout = approx_node_graceful_reboot_time * machines_number
    cluster.check.wait_machines_reboot(boot_time_dict, timeout=machines_reboot_timeout)

    show_step(5)
    LOG.info(f"Check cluster {cluster.namespace}/{cluster.name} for init state")
    cluster.check.wait_graceful_reboot_request(expected_status=False, timeout=600)
    cluster.check.check_machines_status()
    cluster.check.check_cluster_readiness()
