import pytest
import re

from si_tests import logger
from si_tests.utils import utils
from si_tests.managers.openstack_client_manager import OpenStackClientManager

LOG = logger.logger


def process_pid(ssh, services_list):
    pids = {}
    result = ssh.check_call("ps -xaeo pid,command")
    for line in result.stdout:
        line = line.decode().strip()
        for service in services_list:
            if service in line:
                pid = line.split()[0]
                if pid:
                    pids[service] = pid
    ssh.close()
    return pids


def get_container_runtime(ssh):
    result = ssh.check_call("ps -xaeo pid,command")
    container_runtime = ''
    for line in result.stdout:
        line = line.decode().strip()
        if './kubelet' in line:
            container_runtime = re.findall(
                r"--container-runtime-endpoint=.+?\s", line)[-1]
    return container_runtime


@pytest.mark.parametrize("services_list", [["ovsdb-server", "ovs-vswitchd"]])
def test_restart_containerd(show_step, services_list):
    """Verify no downtime of processes during container runtime update.

     Scenario:
         1. Get IDs of  processes on one of the compute nodes
         2. Simulate update process by restart containerd service
         on the compute node
         3. Get IDs of the processes on the compute node after
         containerd service restart
         4. Check that PIDs of process don't change

     """
    os_client = OpenStackClientManager()
    compute_node = os_client.os_manager.api.nodes.list(
        label_selector="openstack-compute-node=enabled")[0]

    ssh = utils.ssh_k8s_node(compute_node.name)
    container_runtime = get_container_runtime(ssh)

    if "containerd" not in container_runtime:
        pytest.skip("Skip because container runtime is not containerd")

    show_step(1)
    pids_before_restart = process_pid(ssh, services_list)

    show_step(2)
    ssh.check_call(
        "sudo systemctl restart containerd; sleep 30")

    show_step(3)
    pids_after_restart = process_pid(ssh, services_list)

    show_step(4)

    assert pids_before_restart == pids_after_restart, \
        f"Processes {services_list} restarted during containerd restart"
