import pytest
import re
import time

from si_tests import logger
from si_tests import settings
from si_tests.managers.kaas_manager import Machine, Cluster, Manager, Namespace  # noqa: F401
from si_tests.managers.openstack_manager import OpenStackManager
from si_tests.utils import waiters
from si_tests.utils import exceptions
from si_tests.tests.lcm.test_replace_bm_master_node import (
    wait_ceph_status)
from si_tests.utils.ha_helpers import collect_machine_sequence, TargetMachineData, id_label_node

LOG = logger.logger


@pytest.mark.usefixtures('log_step_time')
@pytest.mark.usefixtures('mcc_loadtest_os_refapp')
@pytest.mark.parametrize("target_machine_data", collect_machine_sequence(), ids=id_label_node)
def test_ha_drive_full(request, kaas_manager, show_step, target_machine_data: TargetMachineData):
    """Validate disk full failover and data replication within a single node.

    Scenario:
        1. Get cluster health before test
        2. Get info for bmh, machine, disks
        3. Find disks we want to fill out
        4. Fill 95 % of available space
        5. Free space
        6. Check cluster readiness
    """
    if request.session.testsfailed:
        pytest.skip("Skip because only one node can be damage")

    ns: Namespace = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    LOG.info(f"Current Namespace name - {ns.name}")
    cluster: Cluster = ns.get_cluster(settings.TARGET_CLUSTER)
    LOG.info(f"Current Cluster name - {cluster.name}")
    machine: Machine = target_machine_data.machine.machine
    percent = settings.HA_FILL_DISK_PERCENT

    show_step(1)
    cluster.check.check_cluster_readiness()

    show_step(2)
    machine_bmh_name = machine.metadata['annotations'].get('metal3.io/BareMetalHost')
    bmh = ns.get_baremetalhost(name=machine_bmh_name.split("/")[1])
    bm_hosts_data = [{}]
    bm_hosts_data[0]['name'] = machine_bmh_name.split("/")[1].split('-')[0]
    bm_hosts_data[0]['bmh_annotations'] = bmh.data['metadata'].get('annotations', {})

    show_step(3)
    LOG.info(f"Find disks that have partitions and mount points on node - {machine.name}")

    # Find disks which have partitions and mount points to fill them
    ansible_extra = ns.get_ansibleextra(name=machine.get_bmh_name())
    target_storage = ansible_extra.data['spec']['target_storage']

    parted_disks = [s for s in target_storage if 'partition_schema' in s]
    lvm_groups = [s for s in target_storage if 'lvm_groups' in s]
    md_groups = [g for g in target_storage if 'md_devices' in g]

    mount_points = []
    for disk in parted_disks:
        partition_schema = disk['partition_schema']
        for partition in partition_schema:
            if 'mount' in partition:
                if partition['filesystem']['type'] in ('ext4', 'xfs'):
                    mount_points.append(partition['mount']['point'])

    for subgroup in lvm_groups:
        for group in subgroup['lvm_groups']:
            if not group['create']:
                continue

            for name in group['lvnames']:
                if not name['create']:
                    continue

                if 'mount' in name:
                    if name['filesystem']['type'] in ('ext4', 'xfs'):
                        mount_points.append(name['mount']['point'])
    for subgroup in md_groups:
        for group in subgroup['md_devices']:
            if not group['create']:
                continue
            if 'mount' in group:
                if group['filesystem']['type'] in ('ext4', 'xfs'):
                    mount_points.append(group['mount']['point'])

    LOG.info("Mount points to fill - %s", mount_points)

    show_step(4)
    machine.exec_pod_cmd("df -h | grep -v /var/lib/docker | grep -v /var/lib/kubelet", verbose=True)
    cmd_available_space = [f"df {mount_point} -h --output=avail -BG | tail -n1" for mount_point in mount_points]
    command = " ; ".join(cmd_available_space)
    available_space = machine.exec_pod_cmd(command, verbose=True)['logs']
    formatted_available_space = re.findall(r'\d+G', available_space.replace("\n", ""))
    LOG.info(f"Available space {formatted_available_space}")
    desired_size = [int(size[:-1]) * percent / 100 for size in formatted_available_space]
    commands = [
        f"sudo fallocate -l {point[0]}G {point[-1]}fill.out"
        for point in [(desired_size[i], mount_points[i])
                      for i in range(min(len(desired_size), len(mount_points)))]]
    commands.append(
        "df -h | grep -v /var/lib/docker | grep -v /var/lib/kubelet"
    )
    command = ' ; '.join(commands)
    LOG.info("Execute command - %s", command)
    try:
        machine.exec_pod_cmd(cmd=command, verbose=True, timeout=30, delete_pod=False)
    except exceptions.TimeoutError:
        LOG.info("Got timeout error. Expected behavior after disk filling")

    time.sleep(120)  # wait to LB ip moves to new node
    cluster.k8sclient.login()  # try to relogin after LB ip moved to new node

    k8s_nodes = cluster.get_k8s_nodes()
    for node in k8s_nodes:
        LOG.info("Node %s has conditions: %s", node['metadata']['name'], node['status']['conditions'])

    show_step(5)
    # delete file
    commands = [
        f"sudo rm {mount_point}fill.out"
        for mount_point in mount_points]
    commands.append(
        "df -h | grep -v /var/lib/docker | grep -v /var/lib/kubelet"
    )
    command = ' ; '.join(commands)
    LOG.info("Execute command - %s", command)
    machine.exec_pod_cmd(cmd=command, verbose=True, timeout=30, delete_pod=False)

    show_step(6)
    if cluster.is_child:
        LOG.info("Ceph cluster rebalancing may take some time. Wait timeout is 1h.")
        # Waiting for actual Ceph status from Ceph tools pod
        LOG.info("Wait Ceph HEALTH_OK status in Ceph tools")
        waiters.wait(lambda: wait_ceph_status(cluster), timeout=3600, interval=30)
        # Wait until KaaS update Cluster kind with Ceph status
        LOG.info("Wait Ceph HEALTH_OK status in cluster object")
        try:
            health_info = cluster.check.get_ceph_health_detail()
            assert health_info['status'] == "HEALTH_OK", f'Health is not OK. Will not proceed. ' \
                                                         f'Current ceph health status: {health_info}'
        except AssertionError:
            cluster.check.wait_ceph_health_status(timeout=600, interval=30)

    # Check/wait for correct docker service replicas in cluster
    ucp_worker_agent_name = cluster.check.get_ucp_worker_agent_name()
    cluster.check.check_actual_expected_docker_services(
        changed_after_upd={'ucp-worker-agent-x': ucp_worker_agent_name})
    cluster.check.check_k8s_pods()
    cluster.check.check_actual_expected_pods(timeout=3200)
    cluster.check.check_cluster_readiness()
    cluster.check.check_deploy_stage_success()

    if cluster.clusterrelease_version.startswith(settings.MOSK_RELEASE_PREFIX) \
            and cluster.is_os_deployed():
        child_kubeconfig_name, child_kubeconfig = cluster.get_kubeconfig_from_secret()
        with open('child_conf', 'w') as f:
            f.write(child_kubeconfig)
        os_manager = OpenStackManager(kubeconfig='child_conf')
        LOG.info("Wait osdpl health status=Ready")
        os_manager.wait_openstackdeployment_health_status(timeout=1800)
        LOG.info("Wait os jobs to success and pods to become Ready")
        os_manager.wait_os_resources(timeout=1800)
