import pytest
import yaml
import exec_helpers
import os
import time

from si_tests.managers import bootstrap_manager
from si_tests import settings
from si_tests import logger
from si_tests.utils import templates, utils, waiters

LOG = logger.logger


@pytest.mark.usefixtures('log_step_time')
@pytest.mark.usefixtures('log_method_time')
@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                               .format(settings.KAAS_CHILD_CLUSTER_NAME)])
@pytest.mark.usefixtures("store_cluster_description")
@pytest.mark.usefixtures('introspect_child_deploy_objects')
def test_kaas_create_child(kaas_manager, openstack_client, proxy_manager,
                           check_sg, show_step, _):
    """This test is designed for creating child cluster and prepare it
    for further openstack deployment. Openstack deployment requires
    ceph cluster and additional networks to be ready. Building on this,
    test creates common child cluster. After cluster is created,
    3 additional nodes (workers) will be added for ceph cluster.
    Ceph config will be generated according to these new nodes
    and applyed. After ceph cluster is ready, some additional networks
    (2 networks) will be added to each of node with openstack controller
    or openstack compute labels. Based on these networks new netplans for
    these nodes will be generated, uploaded and applyed on every compute
    or control node.

    Scenario:
        1. Collecting env data
        2. Creating namespace
        3. Creating public key
        4. Creating OS credential
        5. Set proxy
        6. Create child cluster and machines with SL label
        7. Pod checks (optional)
        8. Creating nodes for ceph cluster
        9. Waiting for Ceph cluster is Ready
        10. Prepare additional networks for openstack deploy

    """

    # Collecting env data
    show_step(1)
    key_file = settings.KAAS_MGMT_CLUSTER_PRIVATE_KEY_FILE
    keys = utils.load_keyfile(key_file)
    pkey = utils.get_rsa_key(keys['private'])
    auth = exec_helpers.SSHAuth(username='mcc-user',
                                password='', key=pkey)
    seed_ip = openstack_client.get_seed_ip()
    bootstrap = bootstrap_manager.BootstrapManager(seed_ip=seed_ip)
    remote = bootstrap.remote_seed()
    kubectl_path = "{}/bin/kubectl".format(
        settings.KAAS_BOOTSTRAP_TARGET_DIR)
    cluster_name = settings.KAAS_CHILD_CLUSTER_NAME
    namespace_name = settings.KAAS_NAMESPACE
    child_conf_path = 'kubeconfig-{}'.format(cluster_name)
    release_name = settings.KAAS_CHILD_CLUSTER_RELEASE_NAME
    external_network = settings.KAAS_EXTERNAL_NETWORK_ID
    LOG.info("Available releases: {}".format(
        kaas_manager.get_clusterrelease_names()))
    assert release_name in kaas_manager.get_clusterrelease_names()
    child_dns = settings.KAAS_CHILD_CLUSTER_DNS
    machine_image_name = settings.KAAS_CHILD_CLUSTER_MACHINE_IMAGE_NAME
    machine_flavor_name = settings.KAAS_CHILD_CLUSTER_MACHINE_FLAVOR_NAME
    ceph_machine_flavor_name = settings\
        .KAAS_CHILD_CLUSTER_CEPH_MACHINE_FLAVOR_NAME
    compute_machine_flavor_name = settings\
        .KAAS_CHILD_CLUSTER_COMPUTE_MACHINE_FLAVOR_NAME
    control_machine_flavor_name = settings\
        .KAAS_CHILD_CLUSTER_CONTROL_MACHINE_FLAVOR_NAME
    machine_az_name = settings.KAAS_CHILD_CLUSTER_AZ_NAME
    slave_nodes_count = int(settings.KAAS_CHILD_CLUSTER_SLAVE_NODES_COUNT)
    nodes_cidr = "10.10.0.0/16"
    ceph_nodes_cnt = int(settings.KAAS_CHILD_CLUSTER_CEPH_NODES_COUNT)
    computes_cnt = int(settings.KAAS_CHILD_CLUSTER_COMPUTE_NODES_COUNT)
    control_cnt = int(settings.KAAS_CHILD_CLUSTER_CONTROL_NODES_COUNT)
    lma_extra_options = yaml.safe_load(settings.KAAS_CHILD_LMA_EXTRA_OPTIONS) \
        if settings.KAAS_CHILD_LMA_EXTRA_OPTIONS else None
    assert slave_nodes_count >= ceph_nodes_cnt + \
        computes_cnt + \
        control_cnt, ("Not enough {} slave nodes for OS deploy. "
                      "Should be >= {}".format(slave_nodes_count,
                                               ceph_nodes_cnt +
                                               computes_cnt + control_cnt))
    master_nodes_count = int(settings.KAAS_CHILD_CLUSTER_MASTER_NODES_COUNT)
    rnd_string = utils.gen_random_string(6)

    # Creating namespace
    show_step(2)
    LOG.info("Namespace name - %s", namespace_name)
    ns = kaas_manager.get_or_create_namespace(namespace_name)

    # Creating public key
    show_step(3)
    public_key_name = "{cluster_name}-{rnd_string}-key".format(
        cluster_name=cluster_name, rnd_string=rnd_string)
    LOG.info("Public key name - {public_key_name}".format(
        public_key_name=public_key_name))
    with open(settings.KAAS_CHILD_CLUSTER_PUBLIC_KEY_FILE) as pub_key:
        pub_key_content = pub_key.read()
    ns.create_publickey(public_key_name, pub_key_content)

    # Creating openstack credential
    show_step(4)
    credential_name = "{cluster_name}-{rnd_string}-cred".format(
        cluster_name=cluster_name, rnd_string=rnd_string)
    LOG.info("Credential name - {credential_name}".format(
        credential_name=credential_name))
    clouds_yaml_str = templates.render_template(settings.CLOUDS_YAML_PATH)
    region = kaas_manager.get_mgmt_cluster().region_name
    ns.create_openstack_credential(credential_name, clouds_yaml_str, region)
    ns.create_openstack_secret(credential_name, clouds_yaml_str, region)
    ns.wait_openstack_resources_available(credential_name)

    helm_releases = []
    for hrelease in settings.KAAS_CHILD_CLUSTER_EXTRA_HELM_RELEASES:
        helm_releases.append({"name": hrelease})

    show_step(5)
    if ((settings.KAAS_EXTERNAL_PROXY_ACCESS_STR or
         settings.KAAS_INSTALL_FAKE_PROXY) and
            not settings.KAAS_OFFLINE_DEPLOYMENT):
        LOG.error("Proxy variables is set but KAAS_OFFLINE_DEPLOYMENT "
                  "is False.Using Proxy on online deployments makes no sense.")
        raise RuntimeError("Proxy variables is set but KAAS_OFFLINE_DEPLOYMENT"
                           " is False")

    if settings.KAAS_OFFLINE_DEPLOYMENT:
        offline_security_group = [check_sg['name']]
        LOG.info('Offline deployment is enabled. External network will be '
                 'restricted on child cluster nodes. Using '
                 'security group: {0}'.format(
                     [check_sg['name']]))
    else:
        offline_security_group = None

    if settings.KAAS_EXTERNAL_PROXY_ACCESS_STR:
        LOG.info("Found KAAS_EXTERNAL_PROXY_ACCESS_STR. Creating "
                 "proxyobject with proxy {0}".
                 format(settings.KAAS_EXTERNAL_PROXY_ACCESS_STR))
        proxy_access_str = settings.KAAS_EXTERNAL_PROXY_ACCESS_STR
        proxy_object = ns.create_proxyobject(
            name=f"{cluster_name}-{settings.KAAS_PROXYOBJECT_NAME}",
            region=region,
            proxy_str=proxy_access_str)
        proxy_object_name = proxy_object.data['metadata']['name']
    elif settings.KAAS_INSTALL_FAKE_PROXY:
        LOG.info("Variable KAAS_EXTERNAL_PROXY_ACCESS_STR is not set but"
                 "KAAS_INSTALL_FAKE_PROXY is enabled. Fake squid proxy will "
                 "be installed on management cluster to use it in child.")
        proxy_manager.deploy_proxy_sts()
        proxy_manager.deploy_proxy_svc()
        proxy_access_str = proxy_manager.generate_proxy_string()
        proxy_manager.check_proxy_conn()
        proxy_object = ns.create_proxyobject(
            name=f"{cluster_name}-{settings.KAAS_PROXYOBJECT_NAME}",
            region=region,
            proxy_str=proxy_access_str)
        proxy_object_name = proxy_object.data['metadata']['name']
    else:
        proxy_object_name = None

    show_step(6)

    cluster = ns.create_cluster(
        cluster_name,
        release_name,
        credential_name,
        external_network_id=external_network,
        region=region,
        provider="openstack",
        services_cidr="10.96.0.0/16",
        pods_cidr="192.168.0.0/16",
        nodes_cidr=nodes_cidr,
        dns=child_dns,
        extra_helm_releases=helm_releases,
        lma_enabled=settings.KAAS_CHILD_CLUSTER_DEPLOY_LMA,
        lma_extra_options=lma_extra_options,
        public_key_name=public_key_name,
        proxy_name=proxy_object_name,
        secure_overlay=settings.KAAS_CHILD_CLUSTER_SECURE_OVERLAY_ENABLED)

    os_control_labels = [
        {'key': 'openstack-control-plane', 'value': 'enabled'},
        {'key': 'openstack-gateway', 'value': 'enabled'},
        {'key': 'openvswitch', 'value': 'enabled'}]

    os_computes_labels = [
        {'key': 'openstack-compute-node', 'value': 'enabled'},
        {'key': 'openvswitch', 'value': 'enabled'}]

    stacklight_enabled_dict = {'key': 'stacklight', 'value': 'enabled'}

    if settings.STACKLIGHT_ON_COMPUTE_NODES:
        os_computes_labels.append(stacklight_enabled_dict)
    else:
        os_control_labels.append(stacklight_enabled_dict)

    for node in range(master_nodes_count):
        cluster.create_os_machine(
            node_type="master",
            region=region,
            node_flavor=machine_flavor_name,
            node_image=machine_image_name,
            node_az=machine_az_name,
            node_sg=offline_security_group)

    os_control_nodes_names = []
    cluster_control_machines = []
    for _ in range(control_cnt):
        machine = cluster.create_os_machine(
            node_type="node",
            region=region,
            node_flavor=control_machine_flavor_name or machine_flavor_name,
            node_image=machine_image_name,
            node_az=machine_az_name,
            node_labels=os_control_labels,
            node_sg=offline_security_group)
        os_node_name = "kaas-node-{}".format(
            machine.uid)
        os_control_nodes_names.append(os_node_name)
        cluster_control_machines.append(
            machine)

    os_compute_nodes_names = []
    cluster_compute_machines = []
    for _ in range(computes_cnt):
        machine = cluster.create_os_machine(
            node_type="node",
            region=region,
            node_flavor=compute_machine_flavor_name or machine_flavor_name,
            node_image=machine_image_name,
            node_az=machine_az_name,
            node_labels=os_computes_labels,
            node_sg=offline_security_group)
        os_node_name = "kaas-node-{}".format(
            machine.uid)
        os_compute_nodes_names.append(os_node_name)
        cluster_compute_machines.append(
            machine)

    cluster.check.check_machines_status()
    cluster.check.check_cluster_nodes()
    cluster.check.check_k8s_nodes()

    show_step(7)
    if settings.RUN_POD_CHECKS:
        # Check/wait for correct docker service replicas in cluster
        cluster.check.check_actual_expected_docker_services()
        cluster.check.check_k8s_pods()
        cluster.check.check_actual_expected_pods(timeout=3200,
                                                 check_all_nss=True)

    show_step(8)
    LOG.info("Creating nodes for ceph cluster")
    ceph_nodes = []
    for ceph_node in range(ceph_nodes_cnt):
        machine = cluster.create_os_machine(
            node_type="node",
            region=region,
            node_flavor=ceph_machine_flavor_name or machine_flavor_name,
            node_image=machine_image_name,
            node_az=machine_az_name,
            node_sg=offline_security_group)
        ceph_nodes.append(machine.name)

    cluster.check.check_machines_status()
    cluster.check.check_cluster_nodes()
    cluster.check.check_k8s_nodes()
    cluster.check.check_cluster_readiness()

    ceph_tmp_version = cluster.clusterrelease_version

    ceph_nodes_config_path_f = \
        "si_tests/templates/ceph/ceph-virtual-{}.yaml"
    ceph_nodes_config_path = \
        ceph_nodes_config_path_f.format(ceph_tmp_version.replace("-rc", ""))
    if not os.path.isfile(ceph_nodes_config_path):
        ceph_nodes_config_path = ceph_nodes_config_path_f.format("default")
    LOG.info("Using the following MiraCeph template file: {}".format(
        ceph_nodes_config_path))
    with open(ceph_nodes_config_path) as f:
        ceph_cluster_conf = yaml.load(f, Loader=yaml.SafeLoader)

    for node in ceph_nodes:
        machine_obj = cluster.get_machine(node)
        existed_disks = machine_obj.run_cmd(
            'lsblk -o NAME --json').stdout_yaml
        disks_names_before = [
            d['name'] for d in existed_disks['blockdevices']]
        ceph_nodes_map = {
            'name': 'machine_name',
            'devices': [{'name': '', 'config': {'deviceClass': 'hdd'}}],
            'roles': ['mon', 'mgr']
        }
        kaas_node_uid = cluster.get_machine(node).uid
        os_node_name = "kaas-node-{}".format(kaas_node_uid)
        os_node_id = openstack_client.get_server_by_name(
            os_node_name).id
        volume = openstack_client.cinder.volumes.create(
            size=50, name="ceph-volume-{}".format(os_node_name))
        volume_id = volume.id
        waiters.wait(
            lambda: openstack_client.cinder.volumes.get(
                volume_id)._info.get('status') == 'available')
        LOG.info("Attaching volume {} to node {}".format(
            volume_id, os_node_id))
        openstack_client.nova.volumes.create_server_volume(
            server_id=os_node_id, volume_id=volume_id)
        # Need to wait some time for disk appears in devices
        time.sleep(20)
        LOG.info("Checking for new disk")
        disks_after_attach = machine_obj.run_cmd(
            'lsblk -o NAME --json').stdout_yaml
        disks_names_after_attach = [
            d['name'] for d in disks_after_attach['blockdevices']]
        new_disk = [d for d in disks_names_after_attach if
                    d not in disks_names_before]
        assert len(new_disk) > 0, ("No new disks were added")
        LOG.info("New disk name: {}".format(new_disk[0]))
        ceph_nodes_map['name'] = os_node_name
        ceph_nodes_map['devices'][0]['name'] = new_disk[0]
        ceph_cluster_conf['spec']['nodes'].append(ceph_nodes_map)
        ceph_cluster_conf['spec']['network']['clusterNet'] = nodes_cidr
        ceph_cluster_conf['spec']['network']['publicNet'] = nodes_cidr
    with open(ceph_nodes_config_path, 'w') as f:
        yaml.dump(ceph_cluster_conf, f)
    LOG.info("Uploading ceph config to seed node")
    ceph_virtual_config_filename = 'ceph_virtual_{}.yaml'.format(
        cluster_name)
    remote.upload(
        ceph_nodes_config_path, ceph_virtual_config_filename)
    secret_name = "{}-kubeconfig".format(cluster_name)
    name, kubeconfig = cluster.get_kubeconfig_from_secret(secret_name)

    remote.execute("cat << EOF >> {} \n{}EOF".format(child_conf_path,
                                                     kubeconfig))
    LOG.info("Applying ceph config to child cluster")
    remote.check_call('{0} --kubeconfig {1} apply -f {2}'.format(
        kubectl_path, child_conf_path, ceph_virtual_config_filename),
        verbose=True)

    show_step(9)
    LOG.info("Waiting for Ceph cluster is Ready")
    client = cluster.k8sclient
    # Waiting for ceph tools pod starts
    waiters.wait(lambda: client.pods.list(
        namespace='rook-ceph', name_prefix='rook-ceph-tools'),
        timeout=1800)
    ceph_tools_pod = client.pods.list(
        namespace='rook-ceph', name_prefix='rook-ceph-tools')[0]
    LOG.info("Ceph tools pod found")
    LOG.info("Waiting for pod is ready")
    ceph_tools_pod.wait_ready()
    LOG.info("Ceph tools pod is READY")

    ceph_health_timeout = 2400
    cmd_status = ['/bin/sh', '-c', 'ceph -s -f json']
    LOG.info("Waiting for ceph cluster HEALTH is OK")

    def check_status():
        json_ret = ceph_tools_pod.exec(cmd_status)
        try:
            ret = yaml.safe_load(json_ret)
            return ret.get('health', {}).get('status') == "HEALTH_OK"
        except yaml.scanner.ScannerError as e:
            LOG.warning(
                "Got error `{err}` during parsing ceph status: '{st}'".format(
                    err=e,
                    st=json_ret))
            return False

    waiters.wait(
        check_status,
        timeout=ceph_health_timeout, interval=30,
        timeout_msg="Ceph status failed to became "
                    "HEALTH_OK in {} sec".format(ceph_health_timeout))

    LOG.info("Ceph cluster deployed. HEALTH is OK")

    show_step(10)
    LOG.info("Prepare additional networks for openstack deploy")
    new_networks = {}
    for i in range(1, 3):
        net_name = 'kaas-net-{}-{}'.format(cluster_name, i)
        # subnet_cidr = '10.1{}.3.0/24'.format(i)
        network = openstack_client.neutron.create_network(
            {'network': {'name': net_name,
                         'description': 'MOSK related network',
                         'admin_state_up': True,
                         'port_security_enabled': False}})
        net_id = network['network']['id']
        new_networks[net_name] = {'net_id': net_id}
        LOG.info("Created network {} with id {}".format(
            net_name, net_id))
    ip_prefix = 1
    for machine in cluster_control_machines + cluster_compute_machines:
        machine = cluster.get_machine(machine.name)
        machine_ip = machine.public_ip
        machine_name = machine.name
        netplan_path = '{}/{}_50-cloud-init.yaml'.format(
            settings.ARTIFACTS_DIR, machine_name)
        remote_netplan_path = '/etc/netplan/50-cloud-init.yaml'
        ssh = exec_helpers.SSHClient(host=machine_ip, port=22, auth=auth)
        ssh.logger.addHandler(logger.console)
        ssh.download(remote_netplan_path, netplan_path)
        with open(netplan_path) as f:
            netplan = yaml.load(f, Loader=yaml.SafeLoader)
        existed_ifaces = netplan['network']['ethernets']
        netplan_map = {
            'network': {
                'ethernets': {'veth-vbmc-br': {},
                              'veth-vbmc': {},
                              'veth-bm-br': {},
                              'veth-bm': {},
                              'veth-br': {},
                              'veth-phy': {}},
                'bridges': {'br-public': {
                    'dhcp4': 'false',
                    'interfaces': ['veth-br'],
                    'macaddress': '',
                    'addresses': []},
                    'br-baremetal': {
                    'dhcp4': 'false',
                    'interfaces': ['veth-bm-br',
                                   'veth-vbmc-br'],
                    'macaddress': '',
                    'addresses': []}}}}

        for k, v in existed_ifaces.items():
            netplan_map['network']['ethernets'][k] = v

        os_machine_obj = openstack_client.get_server_by_name(
            "kaas-node-{}".format(machine.uid))
        machine_id = os_machine_obj.id
        for i in range(1, 3):
            net_id = new_networks['kaas-net-{}-{}'.format(cluster_name, i)]['net_id']
            if i == 1:
                ip = '10.11.1.{}'.format(ip_prefix)
                bridge = 'br-public'
            else:
                ip = '10.12.1.{}'.format(ip_prefix)
                bridge = 'br-baremetal'
            LOG.info("Attaching network {} to instance {}".format(
                net_id, machine_id))
            openstack_client.nova.servers.interface_attach(
                server=machine_id,
                port_id=None,
                net_id=net_id,
                fixed_ip=None)
            iface_hex = machine.run_cmd(
                "lspci | grep Ethernet | tail -n 1 | "
                "cut -d ':' -f2 | cut -d '.' -f1").stdout_yaml
            iface_num = int(str(iface_hex), 16)
            new_iface = "ens{}".format(iface_num)
            mac = machine.run_cmd("cat /sys/class/net/{}/address".format(
                new_iface)).stdout_yaml
            LOG.info("Found new interface {} on node {} with mac {}".format(
                new_iface, machine_id, mac))
            netplan_map['network']['ethernets'][new_iface] = \
                {'match': {'macaddress': mac},
                 'mtu': 9050, 'set-name': new_iface}

            netplan_map['network']['bridges'][bridge]['interfaces'].append(
                new_iface)
            netplan_map['network']['bridges'][bridge]['addresses'].append(
                ip + "/24")
            netplan_map['network']['bridges'][bridge]['macaddress'] = \
                mac
        with open(netplan_path, 'w') as f:
            yaml.dump(netplan_map, f)
        LOG.info("Uploading new netplan to node {}".format(
            machine_id))
        ssh.upload(netplan_path, '/home/mcc-user/50-cloud-init.yaml')
        LOG.info("Configure new NetDevs")
        with ssh.sudo(enforce=True):
            ssh.check_call('mv /home/mcc-user/50-cloud-init.yaml '
                           '/etc/netplan/')
            ssh.check_call(
                "cat << EOF > /etc/systemd/network/10-veth-phy-br.netdev\n"
                "[NetDev]\n"
                "Name=veth-phy\n"
                "Kind=veth\n"
                "[Peer]\n"
                "Name=veth-br\n"
                "EOF"
            )
            ssh.check_call(
                "cat << EOF > /etc/systemd/network/11-veth-bm.netdev\n"
                "[NetDev]\n"
                "Name=veth-bm\n"
                "Kind=veth\n"
                "[Peer]\n"
                "Name=veth-bm-br\n"
                "EOF"
            )
            ssh.check_call(
                "cat << EOF > /etc/systemd/network/12-veth-vbmc.netdev\n"
                "[NetDev]\n"
                "Name=veth-vbmc\n"
                "Kind=veth\n"
                "[Peer]\n"
                "Name=veth-vbmc-br\n"
                "EOF"
            )
            LOG.info("Applying new netplan for node {}".format(machine_id))
            ssh.check_call('netplan apply /etc/netplan/50-cloud-init.yaml',
                           verbose=True)
        ip_prefix += 1

    cluster.check.check_cluster_readiness()
    cluster.check.check_helmbundles()

    # Collecting artifacts
    cluster.store_k8s_artifacts()
    cluster.provider_resources.save_artifact()

    if settings.KAAS_OFFLINE_DEPLOYMENT:
        cluster.check.check_offline_isolation(
            settings.KAAS_EXTERNAL_PROXY_ACCESS_STR)
