import pytest
import yaml
import time
from si_tests import settings
from si_tests import logger
from si_tests.utils import utils, templates

LOG = logger.logger


@pytest.mark.usefixtures('log_step_time')
@pytest.mark.usefixtures('log_method_time')
@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                               .format(settings.KAAS_CHILD_CLUSTER_NAME)])
@pytest.mark.usefixtures("store_cluster_description")
@pytest.mark.usefixtures('introspect_child_deploy_objects')
def test_kaas_create_baremetal_child_ceph_lma(kaas_manager, proxy_manager, _):
    """Deploy child cluster on top of management cluster."""

    raise Exception("Those test require refactor before run")

    # Collecting env data
    cluster_name = settings.KAAS_CHILD_CLUSTER_NAME
    namespace_name = settings.KAAS_NAMESPACE
    release_name = settings.KAAS_CHILD_CLUSTER_RELEASE_NAME
    LOG.info("Available releases: {}".format(
        kaas_manager.get_clusterrelease_names()))
    assert release_name in kaas_manager.get_clusterrelease_names()

    child_dns = settings.KAAS_CHILD_CLUSTER_DNS
    rnd_string = utils.gen_random_string(6)

    region = kaas_manager.get_mgmt_cluster().region_name

    # Creating namespace
    LOG.info("Namespace name - %s", namespace_name)
    ns = kaas_manager.create_namespace(namespace_name)
    # Create ssh key
    public_key_name = "{cluster_name}-{rnd_string}-key".format(
        cluster_name=cluster_name, rnd_string=rnd_string)
    LOG.info("Public key name - {public_key_name}".format(
        public_key_name=public_key_name))
    with open(settings.KAAS_CHILD_CLUSTER_PUBLIC_KEY_FILE) as pub_key:
        pub_key_content = pub_key.read()
    ns.create_publickey(public_key_name, pub_key_content)

    # Gather nodes information from yaml file
    child_data = yaml.safe_load(templates.render_template(
        settings.BM_CHILD_SETTINGS_YAML_PATH))[0]
    bm_hosts_data = child_data.get('nodes')
    child_cluster_data = child_data.get('child_config')
    metallb_config = child_data.get('metallb_config', {})

    # Creating secret for each node
    for node in bm_hosts_data:
        cred_name = node.get('name') + '-cred'
        secret_data = {
            "username": node.get('ipmi').get('username'),
            "password": node.get('ipmi').get('password')
        }
        ns.create_credentials_secret(name=cred_name, data=secret_data,
                                     region=region, provider="baremetal")

    # Split nodes by types
    storage_nodes = [
        node for node in bm_hosts_data if 'child-storage' in node.get(
            'label', '')]
    control_nodes = [
        node for node in bm_hosts_data if 'child-controller' in node.get(
            'label', '')]
    worker_nodes = [
        node for node in bm_hosts_data if 'child-worker' in node.get(
            'label', '')]
    # assert len(storage_nodes) == 3, 'Storage cluster should contain 3 nodes'

    # Creating BAREMETAL HOSTS with label controlplane
    for node in control_nodes:
        cred_name = node.get('name') + '-cred'
        bootUEFI = node.get('bootUEFI', True)
        machine_name = node.get('name') + "-" + node.get('label')
        bmh_mac = node.get('networks')[0].get('mac')
        ns.create_baremetalhost(bmh_name=machine_name,
                                bmh_secret=cred_name,
                                bmh_mac=bmh_mac,
                                bmh_ipmi=node['ipmi'],
                                role='controlplane',
                                bootUEFI=bootUEFI,
                                bmhi_credentials_name=cred_name)
        time.sleep(1)

    # Creating BAREMETAL HOSTS with label worker
    for node in worker_nodes:
        cred_name = node.get('name') + '-cred'
        bootUEFI = node.get('bootUEFI', True)
        machine_name = node.get('name') + "-" + node.get('label')
        bmh_mac = node.get('networks')[0].get('mac')
        ns.create_baremetalhost(bmh_name=machine_name,
                                bmh_secret=cred_name,
                                bmh_mac=bmh_mac,
                                bmh_ipmi=node['ipmi'],
                                role='worker',
                                bootUEFI=bootUEFI,
                                bmhi_credentials_name=cred_name)
        time.sleep(1)

    # Creating BAREMETAL HOSTS with label storage
    for node in storage_nodes:
        cred_name = node.get('name') + '-cred'
        bootUEFI = node.get('bootUEFI', True)
        machine_name = node.get('name') + "-" + node.get('label')
        bmh_mac = node.get('networks')[0].get('mac')
        ns.create_baremetalhost(bmh_name=machine_name,
                                bmh_secret=cred_name,
                                bmh_mac=bmh_mac,
                                bmh_ipmi=node['ipmi'],
                                role='storage',
                                bootUEFI=bootUEFI,
                                bmhi_credentials_name=cred_name)
        time.sleep(1)

    # Wait for node commissioning
    ns.wait_baremetalhosts_statuses(
        wait_status='ready', retries=25, interval=60)

    # Create cluster
    helm_releases = [{"name": "ceph-controller"}]

    if ((settings.KAAS_EXTERNAL_PROXY_ACCESS_STR or
         settings.KAAS_INSTALL_FAKE_PROXY) and
            not settings.KAAS_OFFLINE_DEPLOYMENT):
        LOG.error("Proxy variables is set but KAAS_OFFLINE_DEPLOYMENT "
                  "is False.Using Proxy on online deployments makes no sense.")
        raise RuntimeError("Proxy variables is set but KAAS_OFFLINE_DEPLOYMENT"
                           " is False")

    if settings.KAAS_EXTERNAL_PROXY_ACCESS_STR:
        LOG.info("Found KAAS_EXTERNAL_PROXY_ACCESS_STR. Creating "
                 "proxyobject with proxy {0}".
                 format(settings.KAAS_EXTERNAL_PROXY_ACCESS_STR))
        proxy_access_str = settings.KAAS_EXTERNAL_PROXY_ACCESS_STR
        proxy_object = ns.create_proxyobject(
            name=f"{cluster_name}-{settings.KAAS_PROXYOBJECT_NAME}",
            region=region,
            proxy_str=proxy_access_str)
        proxy_object_name = proxy_object.data['metadata']['name']
    elif settings.KAAS_INSTALL_FAKE_PROXY:
        LOG.info("Variable KAAS_EXTERNAL_PROXY_ACCESS_STR is not set but"
                 "KAAS_INSTALL_FAKE_PROXY is enabled. Fake squid proxy will "
                 "be installed on management cluster to use it in child.")
        proxy_manager.deploy_proxy_sts()
        proxy_manager.deploy_proxy_svc()
        proxy_access_str = proxy_manager.generate_proxy_string()
        proxy_manager.check_proxy_conn()
        proxy_object = ns.create_proxyobject(
            name=f"{cluster_name}-{settings.KAAS_PROXYOBJECT_NAME}",
            region=region,
            proxy_str=proxy_access_str)
        proxy_object_name = proxy_object.data['metadata']['name']
    else:
        proxy_object_name = None

    if metallb_config:
        metallb_ip_range = None
    else:
        metallb_ip_range = child_cluster_data.get('metallb_ip_range', None)

    cluster = ns.create_cluster(
        cluster_name,
        release_name,
        region=region,
        provider="baremetal",
        loadbalancer_host=child_cluster_data.get('loadbalancer_host', None),
        metallb_ip_range=metallb_ip_range,
        services_cidr="10.96.0.0/18",
        pods_cidr="192.168.0.0/16",
        nodes_cidr="10.10.10.0/24",
        dns=child_dns,
        lma_enabled=settings.KAAS_CHILD_CLUSTER_DEPLOY_LMA,
        extra_helm_releases=helm_releases,
        dedicated_controlplane=False,
        public_key_name=public_key_name,
        proxy_name=proxy_object_name,
        secure_overlay=settings.KAAS_CHILD_CLUSTER_SECURE_OVERLAY_ENABLED)

    # Create control plane machines
    for node in range(len(control_nodes)):
        cluster.create_baremetal_machine(
            node_type="controlplane",
            node_pubkey_name=public_key_name)
        time.sleep(1)

    # Create worker machines
    for node in range(len(worker_nodes)):
        cluster.create_baremetal_machine(
            node_type="worker",
            node_pubkey_name=public_key_name)
        time.sleep(1)

    # Create Storage machines
    for node in range(len(storage_nodes)):
        cluster.create_baremetal_machine(
            node_type="storage",
            node_pubkey_name=public_key_name)
        time.sleep(1)

    # Wait Baremetal hosts be provisioned
    ns.wait_baremetalhosts_statuses(
        wait_status='provisioned', retries=25, interval=60)

    # Waiting for machines are Ready and helmbundles are deployed
    cluster.check.check_machines_status()
    cluster.check.check_cluster_nodes()
    cluster.check.check_cluster_readiness()
    cluster.check.check_helmbundles()
    cluster.check.check_k8s_nodes()

    # Create Ceph cluster
    ceph_network = child_cluster_data.get('ceph_network_config')
    ns.create_ceph_cluster(name='ceph-' + cluster_name + rnd_string,
                           cluster_name=cluster_name,
                           dedicated_controlplane=False,
                           network=ceph_network)
    # Add timeout to get Ceph cluster build namespaces
    time.sleep(10)

    # Collecting artifacts
    cluster.store_k8s_artifacts()

    if cluster.lcm_type_is_ucp:
        # Check/wait for correct docker service replicas in cluster
        cluster.check.check_actual_expected_docker_services(timeout=600)
    # Check/wait for correct pods statuses in child cluster
    cluster.check.check_k8s_pods()
    cluster.check.check_actual_expected_pods(timeout=600,
                                             check_all_nss=True)

    if settings.KAAS_OFFLINE_DEPLOYMENT:
        cluster.check.check_offline_isolation(
            settings.KAAS_EXTERNAL_PROXY_ACCESS_STR)
