import base64

import pytest
import yaml

from si_tests import settings
from si_tests import logger
from si_tests.utils import templates
from si_tests.utils import utils
from si_tests.utils import waiters

LOG = logger.logger


@pytest.mark.usefixtures('log_step_time')
@pytest.mark.usefixtures('log_method_time')
@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                               .format(settings.KAAS_CHILD_CLUSTER_NAME)])
@pytest.mark.usefixtures("store_cluster_description")
@pytest.mark.usefixtures('introspect_child_deploy_objects')
def test_kaas_create_os_child(kaas_manager, check_sg, proxy_manager, show_step, openstack_client, _):
    """Deploy child cluster on top of management cluster.

    Scenario:
        1. Collecting env data
        2. Creating namespace
        3. Creating public key
        4. Creating openstack credential
        5. Set proxy
        6. Configure bastion (optional)
        7. Create child cluster and machines with SL label
        8. Check cluster readiness
        9. Pod checks (optional)
        10. Check bastion VM (optional)

    """

    # Collecting env data
    show_step(1)
    cluster_name = settings.KAAS_CHILD_CLUSTER_NAME
    namespace_name = settings.KAAS_NAMESPACE
    release_name = settings.KAAS_CHILD_CLUSTER_RELEASE_NAME
    external_network = settings.KAAS_EXTERNAL_NETWORK_ID
    LOG.info("Available releases: {}".format(
        kaas_manager.get_clusterrelease_names()))
    assert release_name in kaas_manager.get_clusterrelease_names()
    child_dns = settings.KAAS_CHILD_CLUSTER_DNS
    machine_image_names = settings.KAAS_CHILD_CLUSTER_MACHINE_IMAGE_NAME.split(',')
    machine_flavor_name = settings.KAAS_CHILD_CLUSTER_MACHINE_FLAVOR_NAME
    machine_az_name = settings.KAAS_CHILD_CLUSTER_AZ_NAME
    slave_nodes_count = int(settings.KAAS_CHILD_CLUSTER_SLAVE_NODES_COUNT)
    lma_extra_options = yaml.safe_load(settings.KAAS_CHILD_LMA_EXTRA_OPTIONS) \
        if settings.KAAS_CHILD_LMA_EXTRA_OPTIONS else None
    stacklight_machine_flavor = settings.KAAS_CHILD_CLUSTER_SLAVE_SL_FLAVOR_NAME or machine_flavor_name
    master_nodes_count = int(settings.KAAS_CHILD_CLUSTER_MASTER_NODES_COUNT)
    boot_from_volume = settings.OPENSTACK_BOOT_FROM_VOLUME
    boot_volume_size = settings.OPENSTACK_BOOT_VOLUME_SIZE
    custom_indexes = settings.KAAS_CHILD_CLUSTER_SET_CUSTOM_INDEXES
    indexes = utils.generate_upgrade_indexes()
    rnd_string = utils.gen_random_string(6)
    region = kaas_manager.get_mgmt_cluster().region_name

    # Creating namespace
    show_step(2)
    LOG.info("Namespace name - %s", namespace_name)
    ns = kaas_manager.get_or_create_namespace(namespace_name)

    # Creating public key
    show_step(3)
    public_key_name = "{cluster_name}-{rnd_string}-key".format(
        cluster_name=cluster_name, rnd_string=rnd_string)
    LOG.info("Public key name - {public_key_name}".format(
        public_key_name=public_key_name))
    with open(settings.KAAS_CHILD_CLUSTER_PUBLIC_KEY_FILE) as pub_key:
        pub_key_content = pub_key.read()
    ns.create_publickey(public_key_name, pub_key_content)

    # Creating openstack credential
    show_step(4)
    credential_name = "{cluster_name}-{rnd_string}-cred".format(
        cluster_name=cluster_name, rnd_string=rnd_string)
    LOG.info("Credential name - {credential_name}".format(
        credential_name=credential_name))
    clouds_yaml_str = templates.render_template(settings.CLOUDS_YAML_PATH)
    ns.create_openstack_credential(credential_name, clouds_yaml_str, region)
    ns.wait_openstack_resources_available(credential_name)

    helm_releases = []
    for hrelease in settings.KAAS_CHILD_CLUSTER_EXTRA_HELM_RELEASES:
        helm_releases.append({"name": hrelease})

    if ((settings.KAAS_EXTERNAL_PROXY_ACCESS_STR or settings.KAAS_INSTALL_FAKE_PROXY)
            and not settings.KAAS_OFFLINE_DEPLOYMENT and settings.KAAS_SSL_PROXY_CERTIFICATE_FILE):
        LOG.info("Mitm. Using proxy with custom certificate and not in offline mode")
    elif ((settings.KAAS_EXTERNAL_PROXY_ACCESS_STR or settings.KAAS_INSTALL_FAKE_PROXY) and
            not settings.KAAS_OFFLINE_DEPLOYMENT):
        LOG.error("Proxy variables is set but KAAS_OFFLINE_DEPLOYMENT "
                  "is False.Using Proxy on online deployments makes no sense.")
        raise RuntimeError("Proxy variables is set but KAAS_OFFLINE_DEPLOYMENT"
                           " is False")

    if settings.KAAS_OFFLINE_DEPLOYMENT:
        offline_security_group = [check_sg['name']]
        LOG.info('Offline deployment is enabled. External network will be '
                 'restricted on child cluster nodes. Using '
                 'security group: {0}'.format(
                     [check_sg['name']]))
    else:
        offline_security_group = None

    show_step(5)
    if settings.KAAS_EXTERNAL_PROXY_ACCESS_STR:
        LOG.info("Found KAAS_EXTERNAL_PROXY_ACCESS_STR. Creating "
                 f"proxyobject with proxy {settings.KAAS_EXTERNAL_PROXY_ACCESS_STR}")
        if settings.KAAS_SSL_PROXY_CERTIFICATE_FILE:
            with open(settings.KAAS_SSL_PROXY_CERTIFICATE_FILE, 'r') as f:
                proxy_cert_content = base64.b64encode(f.read().encode("utf-8")).decode()
                LOG.info("Using proxy with custom certificate")
        else:
            proxy_cert_content = None
        proxy_access_str = settings.KAAS_EXTERNAL_PROXY_ACCESS_STR
        proxy_object = ns.create_proxyobject(
            name=f"{cluster_name}-{settings.KAAS_PROXYOBJECT_NAME}",
            region=region,
            proxy_str=proxy_access_str,
            ca_cert=proxy_cert_content)
        proxy_object_name = proxy_object.data['metadata']['name']
    elif settings.KAAS_INSTALL_FAKE_PROXY:
        LOG.info("Variable KAAS_EXTERNAL_PROXY_ACCESS_STR is not set but"
                 "KAAS_INSTALL_FAKE_PROXY is enabled. Fake squid proxy will "
                 "be installed on management cluster to use it in child.")
        proxy_manager.deploy_proxy_sts()
        proxy_manager.deploy_proxy_svc()
        proxy_access_str = proxy_manager.generate_proxy_string()
        proxy_manager.check_proxy_conn()
        proxy_object = ns.create_proxyobject(
            name=f"{cluster_name}-{settings.KAAS_PROXYOBJECT_NAME}",
            region=region,
            proxy_str=proxy_access_str)
        proxy_object_name = proxy_object.data['metadata']['name']
    else:
        proxy_object_name = None

    show_step(6)
    if settings.KAAS_BASTION_CONFIG:
        bastion = settings.KAAS_BASTION_OS_CONFIG
        LOG.info("Use custom bastion config:\n{0}".format(yaml.dump(bastion)))
    else:
        LOG.info("Deploy child cluster without additional bastion config")
        bastion = None

    show_step(7)
    cluster = ns.create_cluster(
        cluster_name,
        release_name,
        credential_name,
        external_network_id=external_network,
        region=region,
        provider="openstack",
        services_cidr="10.96.0.0/16",
        pods_cidr="192.168.0.0/16",
        nodes_cidr="10.10.10.0/24",
        dns=child_dns,
        extra_helm_releases=helm_releases,
        lma_enabled=settings.KAAS_CHILD_CLUSTER_DEPLOY_LMA,
        lma_extra_options=lma_extra_options,
        public_key_name=public_key_name,
        boot_from_volume=boot_from_volume,
        boot_volume_size=boot_volume_size,
        proxy_name=proxy_object_name,
        dedicated_controlplane=settings.KAAS_OS_CHILD_DEDICATED_CONTROLPLANE,
        secure_overlay=settings.KAAS_CHILD_CLUSTER_SECURE_OVERLAY_ENABLED,
        bastion=bastion)

    # Check or set customHostnamesEnabled flag in Cluster object to match settings.CUSTOM_HOSTNAMES
    cluster.set_custom_hostnames_enabled(flag=settings.CUSTOM_HOSTNAMES)

    labels = []
    # Add os labels if CR name is not MKE
    os_control_labels = []
    os_computes_labels = []

    sl_placement_enabled = settings.STACKLIGHT_ENABLE_HA

    if slave_nodes_count < 3 and sl_placement_enabled:
        raise Exception("Not enough worker nodes for LMA HA configuration")

    day1_provisioning_mode = 'auto'
    if settings.FEATURE_FLAGS.enabled('machine-pauses'):
        cluster_release_version = cluster.get_desired_clusterrelease_version()
        if utils.clusterrelease_version_greater_than_or_equal_to_kaas_2_30_0(cluster_release_version) and \
                settings.MACHINE_PAUSE_DURING_CREATION_ENABLED:
            day1_provisioning_mode = 'manual'

    if custom_indexes:
        LOG.info("Custom indexes is enabled. Random index will be set during machine creation")
    if 'mke' not in release_name:
        # for MOS child we will create 3 os control nodes
        # (default value of KAAS_CHILD_CLUSTER_SLAVE_NODES_COUNT)
        # and same number of compute nodes
        os_control_labels_names = [
            {'key': 'openstack-control-plane', 'value': 'enabled'},
            {'key': 'openstack-gateway', 'value': 'enabled'},
            {'key': 'openvswitch', 'value': 'enabled'}]
        for name in os_control_labels_names:
            os_control_labels.append(name)

        os_computes_labels_names = [
            {'key': 'openstack-compute-node', 'value': 'enabled'},
            {'key': 'openvswitch', 'value': 'enabled'}]
        for name in os_computes_labels_names:
            os_computes_labels.append(name)
        for n in range(slave_nodes_count):
            cluster.create_os_machine(
                node_type="node",
                region=region,
                node_flavor=machine_flavor_name,
                node_image=machine_image_names[n % len(machine_image_names)],
                node_az=machine_az_name,
                node_labels=os_control_labels,
                boot_from_volume=boot_from_volume,
                boot_volume_size=boot_volume_size,
                node_sg=offline_security_group,
                day1_deployment=day1_provisioning_mode,
            )
            cluster.create_os_machine(
                node_type="node",
                region=region,
                node_flavor=machine_flavor_name,
                node_image=machine_image_names[n % len(machine_image_names)],
                node_az=machine_az_name,
                node_labels=os_computes_labels,
                boot_from_volume=boot_from_volume,
                boot_volume_size=boot_volume_size,
                node_sg=offline_security_group,
                day1_deployment=day1_provisioning_mode,
            )
    else:
        labels = \
            [{'key': k, 'value': v} for k, v in
             cluster.get_allowed_node_labels().items()]

        # we define 2 SL nodes for both ha and non-ha cases
        # for non-ha nothing else is needed
        # however for ha 1 addtional node is required
        stacklight_nodes_count = 2

        for n in range(stacklight_nodes_count):
            machine_index = indexes.__next__() if custom_indexes else None
            cluster.create_os_machine(
                node_type="node",
                region=region,
                node_flavor=stacklight_machine_flavor,
                node_image=machine_image_names[n % len(machine_image_names)],
                node_az=machine_az_name,
                node_sg=offline_security_group,
                boot_from_volume=boot_from_volume,
                boot_volume_size=boot_volume_size,
                node_labels=labels,
                upgrade_index=machine_index,
                day1_deployment=day1_provisioning_mode,
            )
        slave_nodes_count -= stacklight_nodes_count

        for n in range(slave_nodes_count):
            machine_index = indexes.__next__() if custom_indexes else None
            cluster.create_os_machine(
                node_type="node",
                region=region,
                node_flavor=machine_flavor_name,
                node_image=machine_image_names[n % len(machine_image_names)],
                node_az=machine_az_name,
                boot_from_volume=boot_from_volume,
                boot_volume_size=boot_volume_size,
                node_sg=offline_security_group,
                upgrade_index=machine_index,
                day1_deployment=day1_provisioning_mode,
            )

    for n in range(master_nodes_count):
        machine_index = indexes.__next__() if custom_indexes else None
        cluster.create_os_machine(
            node_type="master",
            region=region,
            node_flavor=machine_flavor_name,
            node_image=machine_image_names[n % len(machine_image_names)],
            node_az=machine_az_name,
            boot_from_volume=boot_from_volume,
            boot_volume_size=boot_volume_size,
            node_sg=offline_security_group,
            upgrade_index=machine_index,
            day1_deployment=day1_provisioning_mode)

    if day1_provisioning_mode == 'manual':
        LOG.info("Waiting for machines to be paused before deployment")
        cluster.check.check_machines_status(expected_status='AwaitsDeployment')
        machines = cluster.get_machines_uncached()
        LOG.info("Removing pause before deployment from machines")
        cluster.set_day1_deployment('auto', machines)
        machines = cluster.get_machines_uncached()
        cluster.check.check_day1_modes(machines, deployment='auto')

    cluster.check.check_machines_status()
    cluster.check.check_cluster_nodes()

    if settings.KAAS_OFFLINE_DEPLOYMENT:
        cluster.check.check_offline_isolation(
            settings.KAAS_EXTERNAL_PROXY_ACCESS_STR)

    # Check SL was not deployed if there is not enough SL labels
    if sl_placement_enabled:
        def _sl_release_exists():
            helmbundle = cluster.get_helmbundle().data
            for chart in helmbundle['spec']['releases']:
                if chart['name'] == 'stacklight':
                    LOG.info("Stacklight release is present in helmbundle")
                    return True
            return False

        LOG.info("Checking that Stacklight was not deployed")
        assert not _sl_release_exists(), \
            'StackLight HA was deployed on cluster with not ' \
            'enough SL labels on worker machines'

        LOG.info("Adding one more worker node with SL label")
        machine_index = indexes.__next__() if custom_indexes else None
        scale_m = cluster.create_os_machine(
            node_type="node",
            region=region,
            node_flavor=stacklight_machine_flavor,
            node_image=machine_image_names[0],
            node_az=machine_az_name,
            node_sg=offline_security_group,
            boot_from_volume=boot_from_volume,
            boot_volume_size=boot_volume_size,
            node_labels=labels,
            upgrade_index=machine_index)

        stacklight_nodes_count += 1

        cluster.check.wait_machine_status_by_name(machine_name=scale_m.name,
                                                  expected_status='Ready')
        cluster.check.check_machines_status()
        cluster.check.check_cluster_nodes()

        LOG.info("Waiting for helm releases (including Stacklight) to be "
                 "deployed")
        cluster.check.check_helmbundles(timeout=1800)

    if labels:
        machines = [
            x for x in cluster.get_machines(machine_type="worker")
            if 'nodeLabels' in x.spec['providerSpec']['value'].keys() and
               labels == x.nodeLabels
        ]
        assert len(machines) == stacklight_nodes_count, \
            f"Not all machines have expected " \
            f"nodeLabels expected: {stacklight_nodes_count} " \
            f"actual: {machines}"
        for machine in machines:
            waiters.wait(
                lambda: machine.check_k8s_nodes_has_labels(),
                timeout=1200, interval=60)

    # Check that all machines booted from volume successfully if needed
    if boot_from_volume:
        LOG.info("Start checking that all machines booted from volume")
        failed_bfv_machines = {}
        all_machines = cluster.get_machines()
        for machine in all_machines:
            try:
                cluster.check.check_boot_from_volume_by_machine_name(machine_name=machine.name,
                                                                     openstack_client=openstack_client,
                                                                     boot_volume_size=boot_volume_size)
            except (AssertionError, Exception) as error:
                failed_bfv_machines.update({machine.name: error})
        if failed_bfv_machines:
            raise Exception(f'bootFromVolume checked failed, list of failed machines with errors: '
                            f'\n {yaml.dump(failed_bfv_machines)}')
        else:
            LOG.info("All machines from cluster deployed with option bootFromVolume successfully")
    if settings.DESIRED_RUNTIME:
        cluster.check.compare_cluster_runtime_with_desired()

    # Collecting artifacts
    cluster.store_k8s_artifacts()

    show_step(8)
    cluster.check.check_k8s_nodes()
    cluster.check.check_kubeconfig_stored_in_userdata()
    cluster.check.check_cluster_readiness()
    cluster.check.check_helmbundles(timeout=1800)
    cluster.check.check_deploy_stage_success()

    # Check that Machines hostnames are created with respect to Cluster flag 'customHostnamesEnabled'
    cluster.check.check_custom_hostnames_on_machines()

    show_step(9)
    if settings.RUN_POD_CHECKS:
        # Check/wait for correct docker service replicas in cluster
        cluster.check.check_actual_expected_docker_services()
        cluster.check.check_k8s_pods()
        cluster.check.check_actual_expected_pods(timeout=3200,
                                                 check_all_nss=True)

    cluster.check.check_overlay_encryption_functionality()
    cluster.check.check_audit()
    cluster.provider_resources.save_artifact()

    show_step(10)
    if bastion:
        LOG.info(f"New bastion IP = {cluster.bastion_ip} for cluster {cluster.name}")
        LOG.info("Check ssh to KaaS machine vie bastion VM")
        machines = cluster.get_machines()
        target_machine = machines[0]
        cluster.check.check_connection_via_bastion(target_machine)
        LOG.info("Check bastion parameters VM in OpenStack")
        cluster_status = cluster.data.get('status') or {}
        vm_id = cluster_status.get('providerStatus', {}).get('bastion', {}).get('serverUUID')
        cluster.check.check_os_vm(openstack_client=openstack_client,
                                  vm_id=vm_id,
                                  image=bastion['image'],
                                  flavor=bastion['flavor'],
                                  az_name=bastion['availabilityZone'],
                                  metadata=bastion['serverMetadata'],
                                  boot_from_volume=bastion['bootFromVolume']['enabled'],
                                  volume_size=bastion['bootFromVolume']['volumeSize'])
    else:
        LOG.info("Current cluster does not have own bastion node")
