import base64

import pytest

import os
import time
from exec_helpers import SSHClient, SSHAuth
from si_tests import settings
from si_tests import logger
# from si_tests.utils.gen_upgrade_paths import get_child_cluster_release_name
from si_tests.utils import waiters, utils, exceptions

LOG = logger.logger


def check_networks(ns, seed_ip):
    # The Plan
    # 1. get all ipaddr objects in ns and extract IP addresses from them
    # 2. get all ipamhost objects in ns
    # 3. For each ipamhost:
    # 3.1 get PXE address and make ssh_client to it
    # 3.2 get corresponding l2template
    # 3.2.1 Get si/expected_ips from labels of the l2template
    # 3.2.2 If there is no expected_ips skip test for the ipamhost
    # 3.3 sort out IP addresses that belongs to the ipamhost from list
    #     of all IPs
    # 3.4 run ping for remaining IP addresses on the ipamhost
    ipaddrs = [i.read().to_dict()['status']['address']
               for i in ns.get_ipam_ipaddrs()]
    LOG.debug(f"Got IP addresses: {ipaddrs}")
    ipamhosts = [i.read().to_dict() for i in ns.get_ipam_hosts()]
    LOG.debug(f"Got IpamHosts: {ipamhosts}")

    seed_key_file = os.path.expanduser(settings.SEED_SSH_PRIV_KEY_FILE)
    seed_key = utils.load_keyfile(seed_key_file)
    seed_key_rsa = utils.get_rsa_key(seed_key["private"])
    remote_seed = SSHClient(
        host=seed_ip,
        port=22,
        # verbose=True,
        auth=SSHAuth(
            username=settings.SEED_SSH_LOGIN,
            key=seed_key_rsa))
    remote_seed.logger.addHandler(logger.console)
    LOG.debug(f"SSH client to seed node: {remote_seed}")
    child_key_file = os.path.expanduser(
        settings.KAAS_CHILD_CLUSTER_PRIVATE_KEY_FILE)
    child_key = utils.load_keyfile(child_key_file)
    child_key_rsa = utils.get_rsa_key(child_key["private"])

    def get_bastion_ip(ipamhost):
        ipamhost_name = ipamhost['metadata']['name']
        status = ipamhost['status']
        if 'osMetadataNetwork' in status:
            return status['osMetadataNetwork']['networks'][0]['ip_address']

        if 'nicMACmap' in status:
            primary = next(filter(lambda x: x.get('primary', False),
                                  status['nicMACmap']))
            if primary:
                return primary['ip'].split('/')[0]
            else:
                raise Exception(
                    f"No primary interface in nicMACmap for {ipamhost_name}")

    failed_pings = {}
    for ipamhost in ipamhosts:
        ipamhost_name = ipamhost['metadata']['name']
        # Build ssh client to the ipamhost through its PXE address
        bastion_ip = get_bastion_ip(ipamhost)
        if not bastion_ip:
            raise Exception(f"No bastion ip for {ipamhost_name}")
        LOG.debug("Check networking on {0} ({1})".format(
            ipamhost_name, bastion_ip))
        # proxy_to() method available only in exec-helpers>=6.0.0
        bastion_kwargs = {
            'hostname': bastion_ip,
            # 'verbose': True,
            'auth': SSHAuth(
                username=settings.KAAS_CHILD_CLUSTER_SSH_LOGIN,
                key=child_key_rsa),
        }
        LOG.debug(f"SSH params for {ipamhost_name}: {bastion_kwargs}")
        # Get netconfig
        netconfig = utils.get_np_struct_from_netconfigfiles(ipamhost['status'].get('netconfigFiles', []))
        # Get corresponding l2template
        l2template_ref = ipamhost['status'].get('l2TemplateRef', '')
        if not (netconfig and l2template_ref):
            LOG.info("No l2template applied to imaphost '%s'. "
                     "Skip extra network checking", ipamhost_name)
            continue
        l2template_name = l2template_ref.split('/')[1]
        l2template = ns.get_l2template(l2template_name).read().to_dict()
        LOG.debug("Got l2template for ipamhost '%s': %s",
                  ipamhost_name, l2template)
        expected_ips = int(l2template['metadata'].get('labels', {})
                           .get('si/expected_ips', 0))
        LOG.debug("Got expected_ips for ipamhost '%s': %s",
                  ipamhost_name, expected_ips)
        if not expected_ips:
            LOG.info("No 'si/expected_ips' in labels of l2template '%s'. "
                     "Skip extra network checking for ipamhost '%s'",
                     l2template_name, ipamhost_name)
            continue

        # Get ip addresses owned by the ipamhost
        own_ipaddrs = []
        for ifaces in netconfig.values():
            if isinstance(ifaces, dict):
                for iface in ifaces.values():
                    own_ipaddrs += [ip.split('/')[0]
                                    for ip in iface.get('addresses', [])
                                    if isinstance(iface, dict)]
            else:
                LOG.debug("Incorrect intarface object in netconfigFiles[netplan]: "
                          f"{ifaces}")
        LOG.debug("IP addresses owned by {0}: {1}".format(
            ipamhost_name, own_ipaddrs))
        # Ipamhost should have expected by l2template number of IPs
        assert len(own_ipaddrs) == expected_ips, \
            "Incorrect number of IP addresses ({0}) for '{2}' ipamhost. " \
            "Expecting to have {1} IPs".format(
                len(own_ipaddrs), expected_ips, ipamhost_name)
        # Ping all IP addresses except its own
        for ipaddr in set(ipaddrs).difference(own_ipaddrs):
            LOG.debug("Run ping on {0} to check availability "
                      "of {1}".format(ipamhost_name, ipaddr))
            ping_success = waiters.icmp_ping(
                host=ipaddr,
                ssh_client=remote_seed,
                bastion_kwargs=bastion_kwargs)
            if not ping_success:
                failed_pings.setdefault(bastion_ip, []).append(ipaddr)
            LOG.debug(f"Was ping of {ipaddr} from {ipamhost_name} "
                      f"({bastion_ip}) successfull: %s", ping_success)
    # All pings should be successfull
    LOG.info(f"Failed pings: {failed_pings}")
    return not failed_pings


def check_child_ceph_cluster(ch_cluster,
                             wait_status='Created',
                             check_health=False):
    """
    ch_cluster = child cluster object
    """
    try:
        # switch to child KUBECONFIG api
        rookceph = ch_cluster.k8sclient.rookcephclusters.get(
            name='rook-ceph', namespace='rook-ceph').data or {}
        rookceph_status = rookceph.get('status') or {}
        rookcephstate = rookceph_status.get('state', 'NotExist')
        LOG.info("Rookceph status is {}".format(rookcephstate))
        if wait_status != rookcephstate:
            return False
        if check_health:
            rookceph_health = rookceph_status.get('ceph', {}).get('health', 'NotExist')
            rookceph_health_detail = rookceph.get(
                'status', {}).get('ceph', {}).get('details', '')
            LOG.info(f'Rookceph health is: {rookceph_health}\n'
                     f'Details: {rookceph_health_detail}')
            if rookceph_health not in ['HEALTH_OK', 'HEALTH_WARN']:
                return False
        return True
    except Exception:
        LOG.warning('rook-ceph cluster does not exist')
        return False


def get_ceph_nodes(data, ns, _bm_nodes, actual_cluster):
    LOG.info("Getting CEPH nodes")
    if actual_cluster.workaround.skip_kaascephcluster_usage():
        c_nodes_data = []
    else:
        c_nodes_data = {}
    if data.get('cephClusterMapping', False) != 'profiled':
        return c_nodes_data
    # Collect all machines, that supposed to be used in ceph
    for_storage_exp = ns._get_storage_machines()
    # Check, that we have mapping for those node
    for machine in for_storage_exp:
        hwid = machine['spec']['providerSpec']['value'].get(
            'hostSelector', {}).get('matchLabels', {}).get(
            'kaas.mirantis.com/baremetalhost-id', False)
        if not hwid:
            LOG.info(f"Machine {machine['metadata']['name']} "
                     f"not supposed "
                     f"to be used in ceph,since don't have"
                     f"'kaas.mirantis.com/baremetalhost-id' hostSelector,"
                     f"continue..")
            continue
        for _node in _bm_nodes:
            if _node.get('name') == hwid and _node.get(
                    'ceph_cluster_node'):
                LOG.info(f"Found custom ceph mapping:\n"
                         f"{_node['ceph_cluster_node']}\n"
                         f"for node: {_node['name']}")
                if actual_cluster.workaround.skip_kaascephcluster_usage():
                    _node.get('ceph_cluster_node')["name"] = machine.get('status', {}).get(
                        'instanceName', {})
                    c_nodes_data.append(_node.get('ceph_cluster_node'))
                else:
                    c_nodes_data[machine['metadata']['name']] = \
                        _node.get('ceph_cluster_node')
    LOG.info(f"Ceph node data for cluster:\n{c_nodes_data}")
    return c_nodes_data


def is_controller_enabled(name, helm_releases):
    for release in helm_releases:
        if release['name'] == name:
            return release.get("enabled", True)
    return False


@pytest.mark.usefixtures('log_step_time')
@pytest.mark.usefixtures('log_method_time')
@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                         .format(settings.TARGET_CLUSTER)])
@pytest.mark.usefixtures("store_cluster_description")
@pytest.mark.usefixtures('introspect_child_target_objects')
@pytest.mark.usefixtures('introspect_no_PRODX_51933_after_lcm')
def test_kaas_create_baremetal_child_lma_istio(kaas_manager, proxy_manager, show_step, _):
    """Deploy child cluster on top of management cluster.

    Scenario:
        1. Collecting env data
        2. Creating namespace
        3. Creating public key
        4. Creating secret for each BMH profile
        5. Creating BMH profiles
        6. Set proxy
        7. Create BM machine
        8. Wait BM hosts to be provisioned
        9. Waiting for machines are Ready and helmbundles are deployed
        10. Install DPDK driver on compute (optional)
        11. Create Ceph cluster
        12. Waiting for KaaSCephCluster Ready
        13. Check cluster readiness
        14. Check offline isolation (optional)

    """

    # Allow test to re-use already defined `namespaces` with some
    # depended on resources, instead failing.
    # Should be used only for debug purposes.
    reuse_resources = False

    # Collecting env data
    show_step(1)

    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    custom_indexes = settings.KAAS_CHILD_CLUSTER_SET_CUSTOM_INDEXES
    indexes = utils.generate_upgrade_indexes()
    release_name = kaas_manager.si_config.version_generators.get_child_cluster_release_name()

    LOG.info("Release name: {}".format(release_name))
    LOG.info("Available releases: {}".format(kaas_manager.get_clusterrelease_names()))
    assert release_name in kaas_manager.get_clusterrelease_names()

    mgmt_cluster = kaas_manager.get_mgmt_cluster()
    mgmt_version = mgmt_cluster.spec['providerSpec']['value']['kaas']['release']
    region = mgmt_cluster.region_name
    with open(settings.KAAS_CHILD_CLUSTER_PUBLIC_KEY_FILE) as pub_key:
        pub_key_content = pub_key.read()

    LOG.info(f"KaaS mgmt version is:{mgmt_version}")
    child_dns = settings.KAAS_CHILD_CLUSTER_DNS
    # NIT: don't rename key-name schema, since it is used across whole tests
    public_key_name = f"{cluster_name}-key"
    allowed_distros = kaas_manager.get_allowed_distributions(release_name)
    render_opts = {
        "mgmt_version": mgmt_version,
        "target_region": region,
        "target_cluster": cluster_name,
        "target_namespace": namespace_name,
        "machine_distribution": allowed_distros.get(settings.DISTRIBUTION_RELEVANCE, {}).get('id', ''),
        "pub_key_content": pub_key_content,
    }
    child_data = utils.render_child_data(kaas_manager.si_config, render_opts)

    bm_hosts_data = child_data.get('nodes', [])
    bm_update_group_data = child_data.get('groups', [])
    LOG.info(f"Update group data {bm_update_group_data}")
    child_cluster_data = child_data.get('child_config', {})
    metallb_config = child_data.get('metallb_config', {})
    tf = child_cluster_data.get('child_config', {}).get('tungstenfabric', {})
    osdpl_node_overrides = dict()
    ceph_cluster_pools = None
    # Creating namespace
    show_step(2)
    LOG.info("Namespace name - %s", namespace_name)
    namespaces = [n.name for n in kaas_manager.get_namespaces()]
    if namespace_name not in namespaces:
        if child_data.get('namespace_config_raw'):
            ns = kaas_manager.create_namespace_raw(child_data['namespace_config_raw'])
        else:
            ns = kaas_manager.create_namespace(namespace_name)
    elif namespace_name in namespaces and reuse_resources:
        ns = kaas_manager.get_namespace(namespace_name)
    else:
        raise Exception(f'Namespace {namespace_name} already exist!')

    ns.create_bm_statics(child_data)
    # Create ssh key
    show_step(3)
    exist_pub_k_names = [k.name for k in ns.get_publickeys()]
    if not child_data.get('public_keys_raw'):
        if public_key_name not in exist_pub_k_names:
            LOG.info(f"Public key name - {public_key_name}")
            ns.create_publickey(public_key_name, pub_key_content)
    else:
        for i in child_data['public_keys_raw']:
            if i['metadata']['name'] not in exist_pub_k_names:
                ns.create_publickey_raw(i)

    show_step(4)
    LOG.info('Creating secret for each BMH')
    bm_nodes = []
    bm_ctrls = []
    if not (bool(child_data.get('bmh_raw', False)) or bool(child_data.get('bmhi_raw', False))):
        existing_bmhc = [bmhc.name for bmhc in ns.get_baremetalhostcredentials()]
        # START BMHC\BMH generated method
        for node in bm_hosts_data:
            if not any(item in node.get('si_roles', []) for item in ['nodeforscale', 'child-scale-controller']):
                # Dirty hack, for not copy-paste bmh_name|cred_name across whole
                # test.
                bmh_name = utils.render_bmh_name(node['name'],
                                                 cluster_name,
                                                 node.get('bootUEFI', True),
                                                 node['bmh_labels'],
                                                 si_roles=node.get('si_roles',
                                                                   False))
                cred_name = node['name'] + '-cred'
                _node = node.copy()
                _node.update({'bmh_name': bmh_name,
                              'cred_name': cred_name})
                bm_nodes.append(_node)
                secret_data = {
                    "username": node['ipmi']['username'],
                    "password": node['ipmi']['password']
                }

                if "kaas.mirantis.com/baremetalhost-credentials-name" in node.get('bmh_annotations', {}):
                    if node['ipmi'].get('monitoringUsername', False):
                        secret_data.update({
                            "monitoringPassword": node['ipmi']['monitoringPassword'],
                            "monitoringUsername": node['ipmi']['monitoringUsername']
                        })
                    if cred_name not in existing_bmhc:
                        ns.create_baremetalhostcredential(name=cred_name, data=secret_data, region=region,
                                                          provider="baremetal")
                    else:
                        LOG.warning(f'bmhc: {cred_name} already exist, skipping')
                else:
                    raise Exception("IPMI credentials supported only over baremetalhostcredentials")

        show_step(5)

        existing_bmh = [bmh.name for bmh in ns.get_baremetalhosts()]
        for node in bm_nodes:
            bmh_name = node['bmh_name']
            if bmh_name not in existing_bmh:
                if node['bmh_labels'].get("hostlabel.bm.kaas.mirantis.com/controlplane"):
                    bm_ctrls.append(bmh_name)

                if settings.KAAS_BM_DUMMY_PROVISION:
                    bmh_annotations = node.get('bmh_annotations', {})
                    bmh_annotations["baremetalhost.metal3.io/paused"] = "true"
                    node['bmh_annotations'] = bmh_annotations

                node_override = node.get('osdpl_node_override')
                if node_override:
                    osdpl_node_overrides[node['name']] = node_override
                ns.create_baremetalhost(bmh_name=bmh_name,
                                        bmh_secret=node['cred_name'],
                                        bmh_mac=node['networks'][0]['mac'],
                                        bmh_ipmi=node['ipmi'],
                                        hardwareProfile=node.get('hardwareProfile', False),
                                        labels=node['bmh_labels'],
                                        annotations=node.get('bmh_annotations', {}),
                                        bootUEFI=node.get('bootUEFI', True),
                                        bmhi_credentials_name=node['cred_name'])
            else:
                LOG.warning(f'BaremetalHost: {bmh_name} already exist, skipping')
                continue

            # dnsmasq bug PRODX-7751; epic with workaround PRODX-8106
            # dnsmasq DHCP provides IPs one by one with ping-check
            # for each IP for about 12sec.
            time.sleep(settings.KAAS_BM_CHILD_SLEEP_PRODX_7751)
            # END BMHC\BMH generated method
    else:
        LOG.info("Create bmhc|bmh|bmhi using raw methods")
        existing_bmhc = [bmhc.name for bmhc in ns.get_baremetalhostcredentials()]
        existing_bmh = [bmh.name for bmh in ns.get_baremetalhosts()]
        bm_ctrls = [i['metadata']['name'] for i in child_data.get('bmh_raw', []) if
                    'hostlabel.bm.kaas.mirantis.com/controlplane' in i['metadata']['labels'].keys()]
        bm_ctrls += [i['metadata']['name'] for i in child_data.get('bmhi_raw', []) if
                     'hostlabel.bm.kaas.mirantis.com/controlplane' in i['metadata']['labels'].keys()]
        for bmhc in child_data.get('bmhc_raw', []):
            if bmhc['metadata']['name'] not in existing_bmhc:
                if not any(item in bmhc['metadata']['labels'].keys() for item in
                           ['si-role/nodeforscale', 'si-role/child-scale-controller']):
                    ns.create_baremetalhostcredential_raw(bmhc)
        for bmh in child_data.get('bmh_raw', []):
            if bmh['metadata']['name'] not in existing_bmh:
                if not any(item in bmh['metadata']['labels'].keys() for item in
                           ['si-role/nodeforscale', 'si-role/child-scale-controller']):
                    ns.create_baremetalhost_raw(bmh)
                    time.sleep(settings.KAAS_BM_CHILD_SLEEP_PRODX_7751)
        for bmhi in child_data.get('bmhi_raw', []):
            if bmhi['metadata']['name'] not in existing_bmh:
                if not any(item in bmhi['metadata']['labels'].keys() for item in
                           ['si-role/nodeforscale', 'si-role/child-scale-controller']):
                    ns.create_baremetalhostinventories_raw(bmhi)
                    time.sleep(settings.KAAS_BM_CHILD_SLEEP_PRODX_7751)

    LOG.info('Wait baremetal hosts commissioning')
    # Commissioning timeout has been increased, since after 2.6 release
    # bm-operator limited to PROVISIONING_LIMIT=6, so only 6 nodes operated
    # at once. Same for provision stage. Otherwise, we will monitor only
    # control nodes - cluster provision can be started when we have
    # control nodes only.
    if not settings.KAAS_BM_DUMMY_PROVISION:
        if not settings.ASYNC_BMH_CONTROL_NODES_WAIT:
            bm_ctrls = []
        else:
            LOG.warning(f'Bmh provision wait scope limited to:{bm_ctrls}')
        if not reuse_resources:
            ns.wait_baremetalhosts_statuses(wait_status='ready', autotime=True, nodes=bm_ctrls)

    LOG.info('Create cluster')
    helm_releases = []
    for hrelease in settings.KAAS_CHILD_CLUSTER_EXTRA_HELM_RELEASES:
        helm_releases.append({"name": hrelease.strip()})

    if settings.KAAS_CHILD_CLUSTER_EXTRA_HELM_RELEASES_JSON:
        helm_releases.extend(settings.KAAS_CHILD_CLUSTER_EXTRA_HELM_RELEASES_JSON)

    if settings.MOSK_CHILD_DEPLOY_CEPH and not is_controller_enabled(
            'ceph-controller', helm_releases):
        LOG.error("The MOSK_CHILD_DEPLOY_CEPH is enabled and ceph controller is "
                  "not enabled.")
        raise RuntimeError("Deploying ceph without ceph-controller is not allowed")

    LOG.info('helm_releases: {0}'.format(helm_releases))
    LOG.info('lma_enabled: {0}'.format(settings.KAAS_CHILD_CLUSTER_DEPLOY_LMA))

    show_step(6)
    if ((settings.KAAS_EXTERNAL_PROXY_ACCESS_STR or
         settings.KAAS_INSTALL_FAKE_PROXY) and
            not settings.KAAS_OFFLINE_DEPLOYMENT):
        LOG.error("Proxy variables is set but KAAS_OFFLINE_DEPLOYMENT "
                  "is False.Using Proxy on online deployments makes no sense.")
        raise RuntimeError("Proxy variables is set but KAAS_OFFLINE_DEPLOYMENT"
                           " is False")

    if settings.KAAS_EXTERNAL_PROXY_ACCESS_STR:
        LOG.info("Found KAAS_EXTERNAL_PROXY_ACCESS_STR. Creating "
                 f"proxyobject with proxy {settings.KAAS_EXTERNAL_PROXY_ACCESS_STR}")
        if settings.KAAS_SSL_PROXY_CERTIFICATE_FILE:
            with open(settings.KAAS_SSL_PROXY_CERTIFICATE_FILE, 'r') as f:
                proxy_cert_content = base64.b64encode(f.read().encode("utf-8")).decode()
                LOG.info("Using proxy with custom certificate")
        else:
            proxy_cert_content = None
        proxy_access_str = settings.KAAS_EXTERNAL_PROXY_ACCESS_STR
        proxy_object = ns.create_proxyobject(
            name=f"{cluster_name}-{settings.KAAS_PROXYOBJECT_NAME}",
            region=region,
            proxy_str=proxy_access_str,
            ca_cert=proxy_cert_content)
        proxy_object_name = proxy_object.data['metadata']['name']
    elif settings.KAAS_INSTALL_FAKE_PROXY:
        LOG.info("Variable KAAS_EXTERNAL_PROXY_ACCESS_STR is not set but"
                 "KAAS_INSTALL_FAKE_PROXY is enabled. Fake squid proxy will "
                 "be installed on management cluster to use it in child.")
        proxy_manager.deploy_proxy_sts()
        proxy_manager.deploy_proxy_svc()
        proxy_access_str = proxy_manager.generate_proxy_string()
        proxy_manager.check_proxy_conn()
        proxy_object = ns.create_proxyobject(
            name=f"{cluster_name}-{settings.KAAS_PROXYOBJECT_NAME}",
            region=region,
            proxy_str=proxy_access_str)
        proxy_object_name = proxy_object.data['metadata']['name']
    else:
        proxy_object_name = None

    show_step(7)
    existing_clusters = [_cluster.name for _cluster in ns.get_clusters()]
    if cluster_name not in existing_clusters:
        if metallb_config:
            metallb_ip_range = None
        else:
            metallb_ip_range = child_cluster_data.get('metallb_ip_range', None)

        if not child_data.get('cluster_raw'):
            cluster = ns.create_cluster(
                cluster_name,
                release_name,
                region=region,
                provider="baremetal",
                loadbalancer_host=child_cluster_data.get('loadbalancer_host', None),
                metallb_ip_range=metallb_ip_range,
                metallb_pool_protocol=child_cluster_data.get("metallb_pool_protocol", "layer2"),
                metallb_peers=child_cluster_data.get("metallb_peers"),
                services_cidr="10.232.0.0/18",
                pods_cidr="192.168.0.0/16",
                nodes_cidr="10.12.10.0/24",
                dns=child_dns,
                extra_helm_releases=helm_releases,
                lma_enabled=settings.KAAS_CHILD_CLUSTER_DEPLOY_LMA,
                dedicated_controlplane=child_cluster_data.get(
                    'dedicated_controlplane', True),
                public_key_name=public_key_name,
                proxy_name=proxy_object_name,
                secure_overlay=settings.KAAS_CHILD_CLUSTER_SECURE_OVERLAY_ENABLED,
                metallb_speaker=child_cluster_data.get("metallb_speaker"),
                use_bgp_announcement=child_cluster_data.get(
                    "use_bgp_announcement"),
                calico_mtu=child_cluster_data.get('calico_mtu', None),
            )
        else:
            LOG.info("Create cluster using raw methods")
            cluster = ns.create_cluster_raw(child_data['cluster_raw'])
    else:
        cluster = ns.get_cluster(cluster_name)
        LOG.warning(f'Re-use existing cluster: {cluster_name}')
    if custom_indexes:
        LOG.info("Custom indexes is enabled. Random index will be set during machine creation")

    # Check or set customHostnamesEnabled flag in Cluster object to match settings.CUSTOM_HOSTNAMES
    cluster.set_custom_hostnames_enabled(flag=settings.CUSTOM_HOSTNAMES)

    # Create update groups based on labels
    upd_groups_name = []
    for node in bm_hosts_data:
        if node['machine_labels'].get("kaas.mirantis.com/update-group"):
            upd_groups_name.append(node['machine_labels'].get("kaas.mirantis.com/update-group"))
        LOG.info(f"Update group names are found and be created: {upd_groups_name}")
    for group in bm_update_group_data:
        if group.get('name') in upd_groups_name:
            cluster.create_update_group(
                cluster_name=settings.TARGET_CLUSTER,
                namespace=settings.TARGET_NAMESPACE,
                name=group.get('name'),
                index=group.get('index'),
                concurrent_updates=group.get('concurrentUpdates'))

    # Create update groups, _raw case
    for upd_g in child_data.get('updategroups_raw', list()):
        upd_g_name = upd_g['metadata']['name']
        if upd_g_name in [x.name for x in cluster.get_update_groups()]:
            LOG.info(f"updategroup obj '{upd_g_name}' already exists.")
        else:
            cluster.create_update_group_raw(upd_g)
            LOG.info(f"updategroup obj '{upd_g_name}' has been created")

    day1_machine_pauses_feature = False
    day1_provisioning_mode = 'auto'
    if settings.FEATURE_FLAGS.enabled('machine-pauses'):
        cluster_release_version = cluster.get_desired_clusterrelease_version()
        if utils.clusterrelease_version_greater_than_or_equal_to_kaas_2_30_0(cluster_release_version):
            day1_machine_pauses_feature = True
            if settings.MACHINE_PAUSE_DURING_CREATION_ENABLED:
                day1_provisioning_mode = 'manual'

    if not (bool(child_data.get('bmh_raw', False)) or bool(child_data.get('bmhi_raw', False))):
        for node in bm_nodes:
            distribution = utils.get_distribution_for_node(kaas_manager, node, release_name)
            LOG.info('Create BareMetal machine')
            custom_bmhp = False
            if node.get('bmh_profile'):
                custom_bmhp = {
                    'namespace': namespace_name,
                    'name': node['bmh_profile']
                }
            machine_index = indexes.__next__() if custom_indexes else None
            cluster.create_baremetal_machine(
                genname=node['bmh_name'],
                node_pubkey_name=public_key_name,
                matchlabels={'kaas.mirantis.com/'
                             'baremetalhost-id':
                                 node['bmh_labels']
                                 ['kaas.mirantis.com/baremetalhost-id']},
                baremetalhostprofile=custom_bmhp,
                l2TemplateSelector=node.get('l2TemplateSelector', dict()),
                labels=node['machine_labels'],
                node_labels=node.get('node_labels', {}),
                distribution=distribution,
                upgrade_index=machine_index,
                day1_provisioning=day1_provisioning_mode,
                day1_deployment=day1_provisioning_mode,
            )
            # dnsmasq bug PRODX-7751; epic with workaround PRODX-8106
            # dnsmasq DHCP provides IPs one by one with ping-check
            # for each IP for about 12sec.
            time.sleep(settings.KAAS_BM_CHILD_SLEEP_PRODX_7751)
    else:
        existing_machines = [m.name for m in ns.get_machines()]
        LOG.info("Create machines using raw methods")
        for m in child_data['machines_raw']:
            if m['metadata']['name'] not in existing_machines:
                if not any(item in m['metadata']['labels'].keys() for item in
                           ['si-role/nodeforscale', 'si-role/child-scale-controller']):
                    cluster.create_baremetal_machine_raw(m)
    show_step(8)
    machines = cluster.get_machines_uncached()
    if day1_provisioning_mode == 'manual':
        LOG.info("Waiting for machines to be paused before provisioning")
        cluster.check.check_machines_status(expected_status='AwaitsProvisioning')
        LOG.info("Checking baremetal hosts: bmh must have 'available' status")
        ns.wait_baremetalhosts_statuses(wait_status='available', autotime=True, nodes=bm_ctrls)
    if day1_machine_pauses_feature:
        LOG.info("Removing pause before provisioning from machines")
        cluster.set_day1_provisioning('auto', machines)
        machines = cluster.get_machines_uncached()
        cluster.check.check_day1_modes(machines, provisioning='auto')

    if settings.KAAS_BM_DUMMY_PROVISION:
        LOG.info('Wait baremetal hosts to be provisioning/provisioned')
        LOG.info('Unlock nodes by bunches, for provision')

        bmh_names = [v['bmh_name'] for v in bm_nodes]
        for i in range(0, len(bmh_names), settings.KAAS_BM_DUMMY_PROVISION_SIZE):
            batch_bmh_names = bmh_names[i:i + settings.KAAS_BM_DUMMY_PROVISION_SIZE]
            for bmh_name in batch_bmh_names:
                LOG.info(f"unlocking {bmh_name}")
                bmh = ns.get_baremetalhost(name=bmh_name)
                utils.remove_k8s_obj_annotations(bmh, ['baremetalhost.metal3.io/paused'])

            ns.wait_baremetalhosts_statuses(
                wait_status={'provisioning', 'provisioned'},
                autotime=True,
                nodes=batch_bmh_names
            )

    else:
        LOG.info('Wait baremetal hosts to be provisioned')
        ns.wait_baremetalhosts_statuses(wait_status='provisioned', autotime=True, nodes=bm_ctrls)

    show_step(9)
    if day1_provisioning_mode == 'manual':
        LOG.info("Waiting for machines to be paused before deployment")
        cluster.check.check_machines_status(expected_status='AwaitsDeployment')
    if day1_machine_pauses_feature:
        LOG.info("Removing pause before deployment from machines")
        cluster.set_day1_deployment('auto', machines)
        machines = cluster.get_machines_uncached()
        cluster.check.check_day1_modes(machines, deployment='auto')
    LOG.info('Waiting for machines are Ready and helmbundles are deployed')
    try:
        cluster.check.check_machines_status()
        cluster.check.check_cluster_nodes()
        cluster.check.check_helmbundles(timeout=3600)
        cluster.check.check_k8s_nodes()
    finally:
        cluster.store_machines_artifacts()
    # This WA to set labels for OpenStack nodes. Once
    # KaaS can set labels this WA should be removed.
    cluster.apply_hotfix_labels(bm_nodes=bm_nodes)
    cluster.apply_nodes_annotations(bm_nodes=bm_nodes)
    cluster.check_bmhp_mapping()

    show_step(10)
    # TODO(alexz) most probably - dead code due get_machines().
    # probably here should be get_nodes()
    machines = ns.get_cluster(cluster_name).get_machines()
    if settings.OPENSTACK_DEPLOY_DPDK:
        LOG.info("Install DPDK driver on compute")
        lkey, lvalue = settings.OPENSTACK_NODES_WITH_HUGEPAGES_LABEL.split('=')
        for machine in machines:
            node_labels = machine.get_k8s_node().read().metadata.labels
            for k, v in node_labels.items():
                if k == lkey and v == lvalue:
                    LOG.info("Get DPDK driver package")
                    distrib_codename = machine.get_distribution().split('/')[-1]
                    openstack_dpdk_driver_package = utils.get_dpdk_driver_package(distrib_codename)
                    LOG.info(f"Install DPDK driver {openstack_dpdk_driver_package}")
                    machine.run_cmd(
                        "sudo apt install -y "
                        f"{openstack_dpdk_driver_package}",
                        timeout=600)
                    break

    # Check that networks have been correctly configured on machines so they
    # can reach each other over all configured network interfaces
    if 'seed_ip' in child_cluster_data:
        LOG.info('Check network connectivity')
        try:
            waiters.wait(
                check_networks,
                predicate_args=[ns, child_cluster_data['seed_ip']],
                interval=5, timeout=60 * 30)
        except exceptions.TimeoutError as e:
            ns.print_events()
            LOG.error("Tired waiting for network connectivity check")
            raise e
        LOG.info('Network connectivity between all machines is OK')
    else:
        LOG.error(
            "Cannot verify network connectivity between child "
            "nodes without seed IP. "
            "Check that 'seed_ip' field is set in child_config.")

    show_step(11)
    if settings.MOSK_CHILD_DEPLOY_CEPH:
        nodes_data = get_ceph_nodes(child_cluster_data, ns, bm_nodes, actual_cluster=cluster)
        ceph_cluster_pools = child_cluster_data.get('ceph_pools_config')

        ceph_rgw_config = None
        if settings.CEPH_DEPLOY_RGW:
            ceph_rgw_config = child_cluster_data.get('ceph_rgw_config')
            LOG.info(f"Adding Ceph rgw metadata {ceph_rgw_config}")

        cephfs_config = None
        if settings.CEPH_DEPLOY_MDS:
            cephfs_config = child_cluster_data.get('cephfs_config')
            LOG.info(f"Adding CephFS metadata {cephfs_config}")
            # Add mds roles only on nodes with mon roles
            for name, node_spec in nodes_data.items():
                if "mon" in node_spec.get('roles', []):
                    nodes_data[name]['roles'].append('mds')
                    LOG.info(f"Adding mds role to Ceph {name} node spec")

        ceph_rook_config = child_cluster_data.get('ceph_rook_config')

        LOG.info('Create Ceph cluster')
        if not child_data.get('kaascephcluster_raw'):
            ns.create_ceph_cluster(
                name=settings.CEPH_CLUSTER_NAME,
                cluster_name=cluster_name,
                hdd_type=child_cluster_data.get('cephClusterDeviceClass', 'hdd'),
                nodes_data=nodes_data,
                pools=ceph_cluster_pools,
                rgw=ceph_rgw_config,
                rook_config=ceph_rook_config,
                network=child_cluster_data.get('ceph_network_config', None),
                tolerations=child_cluster_data.get('ceph_tolerations', None),
                cephfs=cephfs_config)
        else:
            LOG.info("Create kaasCephCluster using raw methods")
            ns.create_ceph_cluster_raw(child_data['kaascephcluster_raw'])
        try:
            waiters.wait(lambda: check_child_ceph_cluster(ch_cluster=cluster,
                                                          check_health=True),
                         timeout=90 * 60, interval=60)
        except exceptions.TimeoutError as e:
            LOG.error("Tired waiting for ceph cluster correct status,"
                      "events from management cluster:")
            ns.print_events()
            # TODO(alexz) extend cluster.k8sclient:
            # LOG.error(f'Tired waiting for ceph cluster correct status,'
            #           'events from child cluster {cluster.name} cluster:')
            # cluster.k8sclient.print_events(
            #     cluster.k8sclient.get_events(namespace=cluster.namespace))
            raise e
    else:
        LOG.info('Ceph cluster create skipped due to there is no ceph-controller on cluster')

    if tf:
        LOG.info("Enable TF monitoring ")
        child_pr_spec = cluster.data['spec']['providerSpec']
        LOG.info("Current stacklight spec: {}".format(child_pr_spec))

        body = {
            "spec": {
                "providerSpec": {
                    "value": {
                        "helmReleases": cluster.data.get(
                            "spec").get("providerSpec").get("value").get(
                            "helmReleases")
                    }
                }
            }
        }
        lma_body = [x for x in
                    body['spec']['providerSpec']['value']['helmReleases']
                    if x['name'] == "stacklight"][0]
        lma_body['values'].update(
            {'tungstenFabricMonitoring': {'enabled': True}})
        kaas_manager.api.kaas_clusters.update(
            name=cluster_name, namespace=namespace_name, body=body)

        LOG.info("Cluster providerSpec after update {}".format(
            cluster.data['spec']['providerSpec']))

    # openstack-controller requires ceph cluster
    cluster.add_osdpl_overrides(osdpl_node_overrides)

    # Collecting artifacts
    cluster.store_k8s_artifacts()

    show_step(12)
    if settings.MOSK_CHILD_DEPLOY_CEPH:
        # If ceph cluster is created and health is OK, it doesn't mean that ceph is ready
        # We need to wait kaascephcluster ready state before checking ceph_pvc
        if cluster.workaround.skip_kaascephcluster_usage():
            LOG.info("Waiting for state Ready in Miraceph cluster")
            cluster.check.wait_miracephhealth_state(timeout=4000)
        else:
            LOG.info("Waiting for KaaSCephCluster Ready")
            cluster.check.wait_kaascephcluster_state(timeout=4000)
        if ceph_cluster_pools is None:
            ceph_crd = cluster.get_cephcluster().read()
            ceph_cluster_pools = ceph_crd.spec['cephClusterSpec']['pools']
        cluster.check.check_actual_expected_pods(timeout=2400, interval=30)
        cluster.check.wait_default_storage(ceph_cluster_pools=ceph_cluster_pools, timeout=1200)
        # Check Ceph pvc
        cluster.check.check_ceph_pvc()
    else:
        LOG.info('Ceph cluster readiness check skipped due to there is no ceph-controller on cluster')

    show_step(13)
    cluster.check.check_cluster_readiness(timeout=3600, interval=90)
    cluster.check.check_diagnostic_cluster_status()

    # Check helmbundles after ceph cluster deployed (e.g. Stacklight)
    cluster.check.check_helmbundles()
    cluster.check.check_deploy_stage_success()

    # Check/wait for correct docker service replicas in cluster
    cluster.check.check_actual_expected_docker_services()
    # Check/wait for correct pods statuses in child cluster
    cluster.check.check_k8s_pods()
    cluster.check.check_actual_expected_pods(timeout=2400, interval=60,
                                             check_all_nss=True)
    cluster.check.check_actual_expected_kernel_versions()
    cluster.check.check_overlay_encryption_functionality()
    cluster.check.check_audit()

    # Check that Machines hostnames are created with respect to Cluster flag 'customHostnamesEnabled'
    cluster.check.check_custom_hostnames_on_machines()
    cluster.check.check_actual_expected_distribution()

    # Check that default container runtime for cluster
    if settings.DESIRED_RUNTIME:
        cluster.check.compare_cluster_runtime_with_desired()

    # Check that hoc installed, and applied
    for hoc in child_data.get('hostosconfiguration_raw', list()):
        existing_hoc_names = [x.name for x in ns.get_hostosconfigurations()]
        LOG.info(f'Found all hostosconfiguration:\n{existing_hoc_names}\n in ns:{ns.name}')
        hoc_name = hoc['metadata']['name']
        hoc_obj = ns.get_hostosconfiguration(hoc_name)
        LOG.info(f'Simple check for applied hostosconfiguration:{hoc_name}')
        cluster.check.wait_lcmmachine_day2_stateitems(hoc_obj, [])
        cluster.check.get_hostosconfig_machines_status(hoc_obj)

    show_step(14)
    if settings.KAAS_OFFLINE_DEPLOYMENT:
        cluster.check.check_offline_isolation(
            settings.KAAS_EXTERNAL_PROXY_ACCESS_STR)
    if settings.ETCD_STORAGE_QOUTA:
        assert cluster.check.check_etcd_quota_applied()
        cluster.check.check_etcd_storage_quota_negative()

    cluster.check.check_bmh_inventory_presense()
