#    Copyright 2022 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
import pytest

from si_tests import logger
from si_tests import settings

from si_tests.managers import bootstrap_manager
from si_tests.managers.kaas_manager import Manager
from si_tests.managers.bootstrapcheck_manager import BoostrapCheckMgr

LOG = logger.logger


class TestProvisionBootstrapRegions:

    def create_bootstrapregion_and_check(self, openstack_client, show_step, provider=None):
        seed_ip = openstack_client.get_seed_ip()
        bootstrap = bootstrap_manager.BootstrapManager(seed_ip=seed_ip)
        kind_kubeconfig_path = f"{settings.ARTIFACTS_DIR}/kubeconfig-kind"

        show_step(1)
        kind_exposed = bootstrap.is_kind_exposed(kind_kubeconfig_path)
        if not kind_exposed:
            raise Exception('Could not connect to KinD. (Not exposed or not running')
        kind_mgr = Manager(kind_kubeconfig_path)
        bvcheckmgr = BoostrapCheckMgr(kind_mgr)

        show_step(2)
        if len(kind_mgr.list_all_bootstrapregions()) != 0:
            # only one br is possible to hold at one time
            br = kind_mgr.list_all_bootstrapregions()[0]
            br.delete()
            bvcheckmgr.check_actual_expected_pods_in_boostrap()

        show_step(3)
        kind_mgr.create_bootstrapregion('test-region', provider)

        show_step(4)
        bvcheckmgr.check_actual_expected_pods_in_boostrap()

    def test_create_bootstrapregion_openstack(self, openstack_client, show_step):
        """Create OS BootstapRegion inside BV2 cluster

        Scenario:
            1. Check bv2 cluster is running and exposed
            2. Verify absence of any bootstrapregions (remove if exist and check resources)
            3. Create os bootstrapregion
            4. Verify required objects created
        """

        self.create_bootstrapregion_and_check(openstack_client, show_step, provider='openstack')

    def test_create_bootstrapregion_aws(self, openstack_client, show_step):
        """Create AWS BootstapRegion inside BV2 cluster

        Scenario:
            1. Check bv2 cluster is running and exposed
            2. Verify absence of any bootstrapregions (remove if exist and check resources)
            3. Create aws bootstrapregion
            4. Verify required objects created
        """

        self.create_bootstrapregion_and_check(openstack_client, show_step, provider='aws')

    def test_create_bootstrapregion_azure(self, openstack_client, show_step):
        """Create Azure BootstapRegion inside BV2 cluster

        Scenario:
            1. Check bv2 cluster is running and exposed
            2. Verify absence of any bootstrapregions (remove if exist and check resources)
            3. Create azure bootstrapregion
            4. Verify required objects created
        """
        msg = 'Azure provider deprecated and disabled. Test will be skipped.'
        LOG.warning(msg)
        pytest.skip(msg)
        self.create_bootstrapregion_and_check(openstack_client, show_step, provider='azure')

    def test_create_bootstrapregion_vsphere(self, openstack_client, show_step):
        """Create vSphere BootstapRegion inside BV2 cluster

        Scenario:
            1. Check bv2 cluster is running and exposed
            2. Verify absence of any bootstrapregions (remove if exist and check resources)
            3. Create vsphere bootstrapregion
            4. Verify required objects created
        """

        self.create_bootstrapregion_and_check(openstack_client, show_step, provider='vsphere')

    def test_create_bootstrapregion_equinixmetalv2(self, openstack_client, show_step):
        """Create EMv2 BootstapRegion inside BV2 cluster

        Scenario:
            1. Check bv2 cluster is running and exposed
            2. Verify absence of any bootstrapregions (remove if exist and check resources)
            3. Create emv2 bootstrapregion
            4. Verify required objects created
        """

        self.create_bootstrapregion_and_check(openstack_client, show_step, provider='equinixmetalv2')

    def test_create_bootstrapregion_baremetal(self, openstack_client, show_step):
        """Create BM BootstapRegion inside BV2 cluster

        Scenario:
            1. Check bv2 cluster is running and exposed
            2. Verify absence of any bootstrapregions (remove if exist and check resources)
            3. Create bm bootstrapregion
            4. Verify required objects created
        """

        LOG.warning("Temporary skip BM from verification because its workflow is under development")
        # self.create_bootstrapregion_and_check(openstack_client, show_step, provider='baremetal')
