#!/usr/bin/env python
"""Utilty script to clean up vSphere lab."""
from argparse import Namespace
from typing import List, Iterable, Optional, Tuple

import argparse
import time
import sys

from vconnector.core import VConnector  # type: ignore

from pyVmomi import vim, vmodl
from retry import retry


def run_cli(args):
    """Define and read CLI arguments."""
    cli = argparse.ArgumentParser(
        description="Command line tool for generate summary report"
    )

    cli.add_argument("--user", help="Define user name", dest="user", required=True)
    cli.add_argument("--password", help="Define user password", required=True)
    cli.add_argument("--host", help="Define hostname or ip", required=True)
    cli.add_argument(
        "--resource-pool-name",
        "-P",
        nargs="+",
        help="Resource pool name",
        required=True)
    cli.add_argument(
        "--machine-folder-path",
        "-F",
        nargs="+",
        help="Full machine folder(s) path(s)",
        required=False,
    )
    cli.add_argument(
        "--add-vm-templates",
        help="Additionaly add VM templates to VMs list",
        action="store_true",  # it's reverted, no flag == false, having flag == true
        required=False,
    )

    commands = cli.add_subparsers(title="Operation commands")
    list_vms = commands.add_parser(
        "list-vms",
        help="List of available VMs",
        description="Show VMs which are available in vSphere",
    )
    list_vms.add_argument(
        "--prefix", help="Use name prefix for filter VMs", default=None
    )
    output_group = list_vms.add_mutually_exclusive_group()
    output_group.add_argument(
        "--show-name", help="Show names of VMs", default=True, action="store_true"
    )
    output_group.add_argument(
        "--show-id", help="Show ids of VMs", default=False, action="store_true"
    )
    list_vms.set_defaults(func=run_list_vms)

    destroy_vms = commands.add_parser(
        "destroy-vms",
        help="Destroy selected or all VMs",
        description="Stop and delete VMs in vSphere",
    )
    destroy_vms.add_argument("--ids", help="Define VMs IDs to destroy")
    destroy_vms.add_argument(
        "--names", nargs="+", help="Define names of VMs to destroy"
    )
    destroy_vms.add_argument(
        "--destroy-all", action="store_true", default=False, help="Destroy all VMs"
    )
    destroy_vms.set_defaults(func=run_destroy_vms)

    return cli.parse_args(args=args)


def get_client(user: str, pwd: str, host: str) -> VConnector:
    """Create vSphere client instance."""
    return VConnector(user, pwd, host)


def get_resource_pools(client: VConnector, pool_name) -> Optional[List[vim.ResourcePool]]:
    """Find pool_name."""
    if "/" in pool_name:
        pool_name = pool_name.split("/")[-1]
    pools_view = client.get_resource_pool_view()
    pools = [p for p in pools_view.view if p.name == pool_name]
    return pools


def get_vm_path(vm):
    folder_name = None
    folder = None
    try:
        folder = vm.parent
    except vmodl.fault.ManagedObjectNotFound as ex:
        print("VM object has been deleted: %s" % ex.obj)
    if folder:
        folder_name = folder.name
        fp = folder.parent
        while fp.name != "Datacenters":
            folder_name = fp.name + "/" + folder_name
            fp = fp.parent
        folder_name = "/" + folder_name
    return folder_name


def get_vms(pool: vim.ResourcePool, add_vm_templates: bool = False) -> Iterable[vim.VirtualMachine]:
    """Fetch list of VMs from vSphere."""
    vnames: List[Tuple[str, vim.VirtualMachine]] = []
    if add_vm_templates:
        print("Will add VM Templates in addition to VMs")
    for v in pool.vm:
        try:
            if v.config and (add_vm_templates or not v.config.template):
                vnames.append((v.name, v))
        except vmodl.fault.ManagedObjectNotFound as ex:
            print(f"VM object has been deleted: {ex.obj}")
            continue

    vnames = sorted(vnames, key=lambda x: x[0])
    return [vm[1] for vm in vnames]


def filter_by_prefix(
    vms: List[vim.VirtualMachine], prefix: str
) -> List[vim.VirtualMachine]:
    """Filter input list of VMs by name prefix."""

    ret_vms: List[vim.VirtualMachine] = []
    for v in vms:
        try:
            if v.name.startswith(prefix):
                ret_vms.append(v)
        except vmodl.fault.ManagedObjectNotFound as ex:
            print(f"VM object has been deleted: {ex.obj}")
            continue

    return ret_vms


def filter_by_names_list(
    vms: Iterable[vim.VirtualMachine], names: List[str]
) -> Iterable[vim.VirtualMachine]:
    """Filter VMs by names list."""

    def yes(vm: vim.VirtualMachine) -> bool:
        try:
            return vm.name in names
        except vmodl.fault.ManagedObjectNotFound as ex:
            print(f"VM object has been deleted: {ex.obj}")
            return False

    return filter(yes, vms)


def filter_by_uuids_list(
    vms: Iterable[vim.VirtualMachine], uuids: List[str]
) -> Iterable[vim.VirtualMachine]:
    """Filter VMs by uuids list."""

    def yes(vm: vim.VirtualMachine) -> bool:
        try:
            return vm.config.uuid in uuids
        except vmodl.fault.ManagedObjectNotFound as ex:
            print(f"VM object has been deleted: {ex.obj}")
            return False

    return filter(yes, vms)


def filter_by_folder_path(
    vms: Iterable[vim.VirtualMachine], folder_path: str
) -> Iterable[vim.VirtualMachine]:
    """Filter VMs by full folder path."""
    return [v for v in vms if get_vm_path(v) == folder_path]


def destroy_vms(
    vms: Iterable[vim.VirtualMachine], message: str = "Will destroy those VMs"
):
    """destroy VMs."""
    if vms:
        print(message)
        for one in vms:
            print("{}: {}".format(one.name, one.config.annotation))
        print("Press Ctrl-C to terminate. Continue in 5 sec")
        time.sleep(5)

        for one in vms:
            destroy_vm(one)
    else:
        print("No VMs found to destroy")


@retry(Exception, delay=1, tries=10)
def destroy_vm(vm: vim.VirtualMachine):
    vmname = vm.name
    try:
        task = vm.PowerOff()
        while task.info.state == "running":
            print("{} shuting down".format(vmname))
        print("{} shuted down".format(vmname))
        task = vm.Destroy()
        while task.info.state == "running":
            print("{} destroying".format(vmname))
        print("{} destroyed".format(vmname))
    except Exception as ex:
        template = "An exception of type {0} occurred. Message:\n{1!r}"
        message = template.format(type(ex).__name__, ex)
        print(message)
        raise


def get_pools_by_names(client: VConnector, pool_names: Iterable[str]) -> List[vim.ResourcePool]:
    pools = []
    for pool_name in pool_names:
        _pools = get_resource_pools(client, pool_name)
        if _pools:
            pools.extend(_pools)
        else:
            print(f"Can't find `{pool_name}` resource pool")
    return pools


def get_vms_from_pools(pools: Iterable[vim.ResourcePool], add_vm_templates: bool = False) -> List[vim.VirtualMachine]:
    vms = []
    for pool in pools:
        pool_vms = get_vms(pool, add_vm_templates)
        vms.extend(pool_vms)

    return vms


def run_list_vms(args: Namespace) -> None:
    """Operate list vms."""
    pool_names = args.resource_pool_name
    paths = args.machine_folder_path

    client = get_client(args.user, args.password, args.host)
    pools = get_pools_by_names(client, pool_names)

    if not pools:
        print("No valid pool names provided")
        return

    vms = get_vms_from_pools(pools, args.add_vm_templates)

    path_filtered = []
    for path in paths:
        path_filtered.extend(filter_by_folder_path(vms, path))
    vms = path_filtered

    for one in vms:
        if args.show_name:
            print(one.name)

        if args.show_id:
            print(one.config.uuid)


@retry(Exception, delay=1, tries=3)
def run_destroy_vms(args: Namespace) -> None:
    """Operate destroy vms."""
    pool_names = args.resource_pool_name
    paths = args.machine_folder_path

    client = get_client(args.user, args.password, args.host)
    pools = get_pools_by_names(client, pool_names)

    if not pools:
        print("No valid pool names provided")
        return

    vms = get_vms_from_pools(pools, args.add_vm_templates)

    path_filtered = []
    for path in paths:
        path_filtered.extend(filter_by_folder_path(vms, path))
    vms = path_filtered

    if args.destroy_all:
        destroy_vms(vms, "Will destroy those VMs")

    if args.names:
        vms = filter_by_names_list(vms, args.names)
        destroy_vms(vms, "Will destroy those VMs")

    if args.ids:
        vms = filter_by_uuids_list(vms, args.names)
        destroy_vms(vms, "Will destroy those VMs")


def main(cli_args: List[str]):
    """Run script."""
    cli = run_cli(cli_args)
    return cli.func(cli)


if __name__ == "__main__":
    sys.exit(main(sys.argv[1:]))
