#    Copyright 2024 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations

from si_tests import logger
from si_tests.managers.bootstrap_manager import BootstrapManager

LOG = logger.logger


class NTPManager:

    def __init__(self, bootstrap_manager: BootstrapManager, verbose: bool = True):
        self.verbose = verbose
        self.bootstrap_manager = bootstrap_manager
        self.remote = bootstrap_manager.remote_seed()
        self.seed_node_ip = self.bootstrap_manager.get_seed_ip()

    def install_ntp(self, ntp_pools=None, use_local_time=True):
        """Install NTP server on the Seed node

        This turns Seed node into a NTP server which can be used by clusters

        :param ntp_pools: list of strings, ntp pools names to sync time
        :param use_local_time: bool, if True, then ntp server will use local time and not sync from external sources
        """
        ntp_pools = ntp_pools or []

        def _form_config_cmd(pools, use_local):

            config = 'cat << EOF > /etc/ntp.conf\n'

            if use_local:
                config += "server 127.127.1.0\n"
                config += "fudge 127.127.1.0 stratum 10\n"

            if pools and not use_local:
                for line in pools:
                    config += f"pool {line} iburst\n"

            # Basic config
            config += ("driftfile /var/lib/ntp/ntp.drift\n"
                       "leapfile /usr/share/zoneinfo/leap-seconds.list\n"
                       "statistics loopstats peerstats clockstats\n"
                       "filegen loopstats file loopstats type day enable\n"
                       "filegen peerstats file peerstats type day enable\n"
                       "filegen clockstats file clockstats type day enable\n"
                       "restrict -4 default kod notrap nomodify nopeer noquery limited\n"
                       "restrict -6 default kod notrap nomodify nopeer noquery limited\n"
                       "restrict 127.0.0.1\n"
                       "restrict ::1\n"
                       "restrict source notrap nomodify\n"
                       )
            config += "EOF\n"
            return config

        create_config_cmd = _form_config_cmd(pools=ntp_pools, use_local=use_local_time)
        restart_ntp_cmd = "systemctl restart ntp"
        sync_hw_clock_cmd = "hwclock --systohc"
        install_cmds = ["apt install -y ntp ntpstat", create_config_cmd, sync_hw_clock_cmd, restart_ntp_cmd]

        for cmd in install_cmds:
            with self.remote.sudo(enforce=True):
                self.remote.check_call(cmd, verbose=self.verbose)

    def set_datetime(self, date):
        """
        Set date and time on seed node
        :param date: str, date in format 'YYYY-MM-DD HH:MM:SS'
        """
        LOG.info(f"Setting date to {date}")
        cmd = f'timedatectl set-time \"{date}\"'
        with self.remote.sudo(enforce=True):
            self.remote.check_call(cmd, verbose=self.verbose)

    def set_time(self, time):
        """
        Set time on seed node
        :param time: str, time in format 'HH:MM:SS'
        """
        LOG.info(f"Setting time to {time}")
        cmd = f"timedatectl set-time {time}"
        with self.remote.sudo(enforce=True):
            self.remote.check_call(cmd, verbose=self.verbose)

    def add_time(self, days=0, hours=0, minutes=0, seconds=0, sync_hw_clock=True):
        time_parts = []

        if days:
            time_parts.append(f'+{days} days')
        if hours:
            time_parts.append(f'+{hours} hours')
        if minutes:
            time_parts.append(f'+{minutes} minutes')
        if seconds:
            time_parts.append(f'+{seconds} seconds')

        cmd_date = f'date -s "{" ".join(time_parts)}"'
        with self.remote.sudo(enforce=True):
            self.remote.check_call(cmd_date, verbose=self.verbose)
        if sync_hw_clock:
            self.sync_hw_clock()

    def sync_hw_clock(self):
        with self.remote.sudo(enforce=True):
            self.remote.check_call('hwclock --systohc', verbose=self.verbose)

    @property
    def ntptime(self):
        return self.remote.check_call("ntptime", verbose=self.verbose).stdout_str

    @property
    def date(self):
        return self.remote.check_call("date", verbose=self.verbose).stdout_str
