import exec_helpers
import yaml

from exec_helpers import CalledProcessError
from retry import retry
from si_tests import logger
from si_tests.utils import utils
from si_tests.deployments.utils import commons

LOG = logger.logger


class Helm3Manager(object):
    binary = utils.get_binary_path("helm")

    def __init__(self,
                 namespace: str = 'openstack',
                 kubeconfig_path: str = None,
                 verbose=False):
        self.namespace = namespace
        self.kubeconfig_path = kubeconfig_path
        self.verbose = verbose

    @retry(CalledProcessError, delay=60, tries=10, logger=LOG)
    def run_cmd(self, cmd, cwd=None):
        cmd = " ".join([self.binary, *cmd])
        res = exec_helpers.Subprocess().execute(cmd, cwd=cwd, verbose=self.verbose)
        if res.exit_code:
            raise CalledProcessError(result=res, expected=(0,))
        return (res.stdout_str, res.stderr_str)

    def dependency_update(self, path):
        executor = commons.get_local_executor()
        result = executor.execute(f"{self.binary} dependency update", cwd=path)
        assert result.exit_code == 0, result

    def list(self, args=None):
        args = args or []
        cmd = [
            "list",
            "--namespace",
            self.namespace,
            "-o",
            "json",
            *args,
        ]
        stdout, stderr = self.run_cmd(cmd)
        if isinstance(res := yaml.safe_load(stdout), list):
            return res
        else:
            LOG.error("Helm return unexpected answer " + stdout)
            return []

    def get_release_values(self, name, args=None):
        args = args or []
        cmd = [
            "get",
            "values",
            "--namespace",
            self.namespace,
            name,
            "-o",
            "json",
            *args,
        ]
        stdout, stderr = self.run_cmd(cmd)
        return yaml.safe_load(stdout)

    def get_releases_values(self, args=None):
        args = args or []
        res = {}
        for release in self.list(args):
            name = release['name']
            res[name] = self.get_release_values(name)
        return res

    def delete_chart(self, kubeconfig_path, namespace, chart_name, timeout="5m"):
        _kubeconfig_path = kubeconfig_path or self.kubeconfig_path
        cmd = [
            f"--kubeconfig {_kubeconfig_path}",
            "uninstall",
            chart_name,
            "--namespace",
            namespace,
            "--timeout",
            timeout
        ]
        LOG.debug(f"Helm command: {cmd}")
        stdout, stderr = self.run_cmd(cmd)
        LOG.info(stdout)
        if stderr:
            LOG.error(stderr)
            return False
        return True

    def install_chart(self, kubeconfig_path, namespace, chart_name,
                      chart_path, timeout="5m", values_path=None):
        _kubeconfig_path = kubeconfig_path or self.kubeconfig_path
        LOG.info(f'Installing chart {chart_name} to ns:{namespace}')
        cmd = [
            f"--kubeconfig {_kubeconfig_path}",
            "install",
            chart_name,
            chart_path,
            "--namespace",
            namespace,
            "--wait",
            "--timeout",
            timeout
        ]
        if values_path is not None:
            for value_path in values_path:
                cmd.extend(["-f", value_path])

        LOG.debug(f"Helm command:\n{cmd}")
        stdout, stderr = self.run_cmd(cmd)
        LOG.info(stdout)
        if stderr:
            LOG.error(stderr)
            return False
        return True
