import yaml

from si_tests import settings
from si_tests import logger
from si_tests.managers import openstack_manager
from si_tests.utils import waiters

LOG = logger.logger
timeout = 600


class Policy(object):

    def __init__(self, os_service, os_pod):
        self.os_service = os_service
        osdpl_name = settings.OSH_DEPLOYMENT_NAME
        self.os_manager = openstack_manager.OpenStackManager()
        self.osdpl = self.os_manager.get_openstackdeployment(osdpl_name)
        self.os_pods = self.get_policy_pods(os_pod)

    def get_policy_pod(self, os_pod):
        api_pod = self.get_policy_pods(os_pod)
        return api_pod[0]

    def get_policy_pods(self, os_pod):
        api_pod = [
            pod for pod in self.os_manager.get_os_pods() if
            os_pod in pod.name]
        assert len(api_pod) > 0, f"POD {os_pod} was not found"
        return api_pod

    def update_policy_osdpl(self, policy, step):
        osdpl_before_update = self.osdpl.read()
        fp_before_update = osdpl_before_update.status['fingerprint']
        common = {
            'spec': {
                'features': {
                    'policies': {
                        self.os_service: policy},
                }
            }
        }
        self.osdpl.patch(common)
        LOG.info(f"[Step {step}]. Waiting for OSDPL fingerprint updating.")
        timeout_msg = (f"[Step {step}]OpenStack deployment didn't reach"
                       " `deployed=True` status")
        waiters.wait(self.os_manager.check_osdpl_status_value,
                     predicate_args=('fingerprint',
                                     fp_before_update,
                                     False),
                     timeout=timeout,
                     timeout_msg=timeout_msg)
        LOG.info(f"[Step {step}] Policy was successfully "
                 f"updated to {policy} in OSDPL")

    def wait_policy_updated_on_pod(self, check_policy_file, policy,
                                   step):
        timeout_msg = (f"[Step {step}] Policy was not updated "
                       f"to {policy} ")
        if not policy:
            timeout_msg = (f"[Step {step}] Policy was not updated to empty policy")
        waiters.wait(self.check_policy_updated,
                     predicate_args=(check_policy_file,
                                     policy),
                     timeout=timeout,
                     timeout_msg=timeout_msg)
        LOG.info(f"[Step {step}] Policy was successfully "
                 f"updated to {policy} ")

    def check_policy_updated(self, check_policy_file, policy):
        """Wait policy updated
        Args:
            :param check_policy_file: (list) bash script to get policy, like:
             ['/bin/sh', '-c', ' cat /etc/glance/policy.d/02-custom.yaml']
            :param policy: (dict|None) dict with expected policy values to check
        """
        if not policy:
            policy = dict()
        for pod in self.os_pods:
            updated_policy = yaml.safe_load(pod.exec(
                check_policy_file))
            if not updated_policy:
                updated_policy = dict()
            if updated_policy != policy:
                return False

        return True

    def check_policy_absent_before_update(self, get_policy_file_before_update):

        for pod in self.os_pods:
            policy_before_update = pod.exec(
                get_policy_file_before_update)
            assert policy_before_update == '', "Policy should be absent"
