#    Copyright 2023 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations

import pytest
import time

from si_tests import logger, settings
from si_tests.deployments.deploy_refapp import Refapp
from si_tests.managers.bootstrap_manager import BootstrapManager
from si_tests.managers.load_tools_manager import LoadToolsManager
from si_tests.managers.openstack_manager import OpenStackManager
from si_tests.managers.openstack_client_manager import OpenStackClientManager
from si_tests.utils import templates as templates_utils

LOG = logger.logger


def store_statistics(statistic_data,
                     statistic_data_source='load_manager',  # Constant for the fixtures in this module
                     statistic_endpoint_name='refapp',    # Constant for the fixtures in this module
                     statistic_endpoint_url='',
                     test_name='',
                     cluster_name='',
                     cluster_namespace='',
                     cluster_provider_name='',
                     version_before_test='',
                     version_after_test='',
                     openstack_version_before_test='',
                     openstack_version_after_test='',
                     tf_version_before_test='',
                     tf_version_after_test='',
                     release_key='',  # To use as a key in 'expected_downtimes' dictionary
                     start_time_str='',
                     start_time_timestamp=0.0,
                     end_time_str='',
                     end_time_timestamp=0.0,
                     jenkins_build_url=settings.JENKINS_BUILD_URL,
                     statistic_raw_data='',
                     ):
    """Save statistic data. Appends statistic_data to an existing YAML"""
    data = {
        'statistic_data': statistic_data,
        'statistic_data_source': statistic_data_source,
        'statistic_endpoint_name': statistic_endpoint_name,
        'statistic_endpoint_url': statistic_endpoint_url,
        'test_name': test_name,
        'cluster_name': cluster_name,
        'cluster_namespace': cluster_namespace,
        'cluster_provider_name': cluster_provider_name,
        'version_before_test': version_before_test,
        'version_after_test': version_after_test,
        'openstack_version_before_test': openstack_version_before_test,
        'openstack_version_after_test': openstack_version_after_test,
        'tf_version_before_test': tf_version_before_test,
        'tf_version_after_test': tf_version_after_test,
        'release_key': release_key,
        'start_time_str': start_time_str,
        'start_time_timestamp': int(start_time_timestamp),
        'end_time_str': end_time_str,
        'end_time_timestamp': int(end_time_timestamp),
        'jenkins_build_url': jenkins_build_url,
        'statistic_raw_data': statistic_raw_data,
    }

    statistic_file_path = settings.DOWNTIME_STATISTIC_FILE_PATH
    LOG.info(f"Save '{statistic_endpoint_name}' availability statistic to {statistic_file_path}")
    with templates_utils.YamlEditor(file_path=statistic_file_path) as editor:
        current_content = editor.content
        current_content.setdefault('collected_data', []).append(data)
        editor.content = current_content


def get_cluster_name_params():
    """Guess which variables were used to pass the cluster name/namespace"""
    TARGET_CLUSTER = settings.get_var('TARGET_CLUSTER')
    TARGET_NAMESPACE = settings.get_var('TARGET_NAMESPACE')
    if TARGET_CLUSTER and TARGET_NAMESPACE:
        LOG.info("Use TARGET_CLUSTER and TARGET_NAMESPACE as the cluster parameters")
        return TARGET_CLUSTER, TARGET_NAMESPACE

    KAAS_NAMESPACE = settings.get_var('KAAS_NAMESPACE')
    KAAS_CHILD_CLUSTER_NAME = settings.get_var('KAAS_CHILD_CLUSTER_NAME')
    if KAAS_CHILD_CLUSTER_NAME and KAAS_NAMESPACE:
        LOG.info("Use KAAS_CHILD_CLUSTER_NAME and KAAS_NAMESPACE as the cluster parameters")
        return KAAS_CHILD_CLUSTER_NAME, KAAS_NAMESPACE

    raise Exception("No cluster name/namespace environment variables found")


@pytest.fixture(scope='function')
def mos_tf_api_loadtest(os_manager, tf_manager, func_name):
    """
    Check tf-api availability via public endpoint
    Use for MOS clusters (KUBECONFIG_PATH contains the target cluster config)
    """
    if not settings.SI_LOADTOOLS_CHECK_TF_API_WORKLOAD:
        LOG.info("Skip TF api endpoint check, please set SI_LOADTOOLS_CHECK_TF_API_WORKLOAD in True")
        yield
        return

    bootstrap_manager = BootstrapManager.get_si_config_bootstrap_manager()

    load_manager = LoadToolsManager(bootstrap_manager=bootstrap_manager, os_manager=os_manager)

    load_manager.run_loadtools_on_seed_node(run_class="TungstenFabric")
    load_manager.start_swarm()

    tf_version_before_test = tf_manager.get_tf_version()
    start_time_str = time.strftime("%Y%m%d%H%M%S")
    start_time_timestamp = time.time()

    yield

    end_time_str = time.strftime("%Y%m%d%H%M%S")
    end_time_timestamp = time.time()
    tf_version_after_test = tf_manager.get_tf_version()
    release_key = 'mos'  # use downtimes for MOS in expected_downtimes dictionary

    try:
        load_manager.stop_swarm()
        statistics = load_manager.get_full_history()

        # Save statistic data
        store_statistics(statistic_data=statistics,
                         statistic_endpoint_name='tungstenfabric',
                         test_name=func_name,
                         cluster_name='',  # Leave empty for MOS-only clusters
                         cluster_namespace='',
                         cluster_provider_name='',
                         tf_version_before_test=tf_version_before_test,
                         tf_version_after_test=tf_version_after_test,
                         release_key=release_key,
                         start_time_str=start_time_str,
                         start_time_timestamp=start_time_timestamp,
                         end_time_str=end_time_str,
                         end_time_timestamp=end_time_timestamp)

    except Exception as e:
        raise e
    finally:
        load_manager.save_docker_logs(f"loadtools_{end_time_str}.log")
        load_manager.exit()


@pytest.fixture(scope='function')
def mos_loadtest_os_refapp(os_manager, openstack_client_manager, func_name):
    """
    Deploy RefApp to OpenStack Cluster (if not exist) as a workload
    Use for MOS clusters (KUBECONFIG_PATH contains the target cluster config)

    """
    if not settings.ALLOW_WORKLOAD:
        LOG.info("Skip RefApp load")
        yield
        return

    osdpl = os_manager.get_osdpl_deployment()

    LOG.info("Start RefApp load")
    refapp = Refapp(openstack_client_manager)
    refapp.deploy()
    LOG.info(f"RefApp URL = {refapp.url}")

    bootstrap_manager = BootstrapManager.get_si_config_bootstrap_manager()

    load_manager = LoadToolsManager(bootstrap_manager=bootstrap_manager, os_manager=os_manager,
                                    refapp_address=refapp.url)
    # load_manager.run_smoke()
    refapp.check()

    load_manager.run_loadtools_on_seed_node()
    load_manager.start_swarm()

    openstack_version_before_test = osdpl.read().spec["openstack_version"]
    start_time_str = time.strftime("%Y%m%d%H%M%S")
    start_time_timestamp = time.time()

    yield

    end_time_str = time.strftime("%Y%m%d%H%M%S")
    end_time_timestamp = time.time()
    openstack_version_after_test = osdpl.read().spec["openstack_version"]
    release_key = 'mos'  # use downtimes for MOS in expected_downtimes dictionary

    try:
        load_manager.stop_swarm()
        load_manager.wait_loadtools_available(run_refapp_smoke=True)
        statistics = load_manager.get_full_history()

        # Save statistic data
        store_statistics(statistic_data=statistics,
                         statistic_endpoint_url=refapp.url,
                         test_name=func_name,
                         cluster_name='',  # Leave empty for MOS-only clusters
                         cluster_namespace='',
                         cluster_provider_name='',
                         openstack_version_before_test=openstack_version_before_test,
                         openstack_version_after_test=openstack_version_after_test,
                         release_key=release_key,
                         start_time_str=start_time_str,
                         start_time_timestamp=start_time_timestamp,
                         end_time_str=end_time_str,
                         end_time_timestamp=end_time_timestamp)

        LOG.info("Check RefApp after test")
        # load_manager.run_smoke()
        refapp.check()
    except Exception as e:
        raise e
    finally:
        load_manager.save_docker_logs(f"loadtools_{end_time_str}.log")
        load_manager.exit()
        refapp.collect_logs()


@pytest.fixture(scope='function')
def mcc_loadtest_os_refapp(kaas_manager, func_name):
    """
    Deploy RefApp to OpenStack Cluster (if not exist) as a workload
    Use for MCC-based clusters (KUBECONFIG_PATH contains Management cluster config)

    """
    if not settings.ALLOW_WORKLOAD:
        LOG.info("Skip RefApp load")
        yield
        return

    cluster_name, cluster_namespace = get_cluster_name_params()
    managed_ns = kaas_manager.get_namespace(cluster_namespace)
    cluster = managed_ns.get_cluster(cluster_name)
    if cluster.is_management:
        LOG.info("Skip OpenStack RefApp load for mgmt cluster")
        yield
        return
    clusterrelease_before_test = cluster.clusterrelease_version
    if "mos" not in clusterrelease_before_test:
        LOG.info(f"Skipping fixture 'mcc_loadtest_os_refapp': expected MOSK cluster, but provided "
                 f"{cluster_namespace}/{cluster_name} "
                 f"with clusterrelease {cluster.clusterrelease_version}")
        yield
        return

    LOG.info("Start RefApp load")
    child_kubeconfig_name, child_kubeconfig = cluster.get_kubeconfig_from_secret()
    with open(child_kubeconfig_name, 'w') as f:
        f.write(child_kubeconfig)
    cluster_os_manager = OpenStackClientManager(kubeconfig=child_kubeconfig_name)

    refapp = Refapp(cluster_os_manager, kubeconfig=child_kubeconfig_name)
    refapp.deploy()
    LOG.info(f"RefApp URL = {refapp.url}")

    bootstrap_manager = BootstrapManager.get_si_config_bootstrap_manager()
    os_manager = OpenStackManager(kubeconfig=child_kubeconfig_name)
    load_manager = LoadToolsManager(bootstrap_manager=bootstrap_manager, os_manager=os_manager,
                                    refapp_address=refapp.url)
    # load_manager.run_smoke()
    refapp.check()

    load_manager.run_loadtools_on_seed_node()
    load_manager.start_swarm()

    start_time_str = time.strftime("%Y%m%d%H%M%S")
    start_time_timestamp = time.time()

    yield

    end_time_str = time.strftime("%Y%m%d%H%M%S")
    end_time_timestamp = time.time()
    clusterrelease_after_test = cluster.clusterrelease_version

    # use downtimes for MCC LCM actions in expected_downtimes dictionary
    is_patchrelease = cluster.is_patchrelease_upgrade(clusterrelease_version_before=clusterrelease_before_test,
                                                      clusterrelease_version_after=clusterrelease_after_test)
    if is_patchrelease:
        # Patchreleases have a reduced LCM actions in comparing with major releases
        # So, there may be expected reduced downtimes
        release_key = 'patch_release'
    else:
        release_key = 'major_release'

    try:
        load_manager.stop_swarm()
        load_manager.wait_loadtools_available(run_refapp_smoke=True)
        statistics = load_manager.get_full_history()

        # Save statistic data
        store_statistics(statistic_data=statistics,
                         statistic_endpoint_url=refapp.url,
                         test_name=func_name,
                         cluster_name=cluster.name,
                         cluster_namespace=cluster.namespace,
                         cluster_provider_name=cluster.provider.provider_name,
                         version_before_test=clusterrelease_before_test,
                         version_after_test=clusterrelease_after_test,
                         release_key=release_key,
                         start_time_str=start_time_str,
                         start_time_timestamp=start_time_timestamp,
                         end_time_str=end_time_str,
                         end_time_timestamp=end_time_timestamp)

        LOG.info("Check RefApp after test")
        # load_manager.run_smoke()
        refapp.check()
    except Exception as e:
        raise e
    finally:
        load_manager.save_docker_logs(f"loadtools_{end_time_str}.log")
        load_manager.exit()
        refapp.collect_logs()


@pytest.fixture(scope='function')
def mcc_loadtest_k8s_api(kaas_manager, func_name):
    # TODO(ddmitriev): add loadtest for kube-api endpoint in the TARGET_CLUSTER
    yield


@pytest.fixture(scope='function')
def mcc_loadtest_mke_api(kaas_manager, func_name):
    # TODO(ddmitriev): add loadtest for mke api endpoint in the TARGET_CLUSTER
    yield


@pytest.fixture(scope='function')
def collect_downtime_statistics(mcc_loadtest_os_refapp, mcc_loadtest_k8s_api, mcc_loadtest_mke_api):
    """Aggregation fixture to run several other fixtures which collect different downtime statistics"""
    yield


def run_load_test(kaas_manager, func_name, service):
    cluster_name, cluster_namespace = get_cluster_name_params()
    managed_ns = kaas_manager.get_namespace(cluster_namespace)
    cluster = managed_ns.get_cluster(cluster_name)
    clusterrelease_before_test = cluster.clusterrelease_version
    keycloak_ip = kaas_manager.get_keycloak_ip()
    keycloak_url = f"https://{keycloak_ip}"
    iam_password = kaas_manager.si_config.get_keycloak_user_password(settings.LMA_DOWNTIME_USER)

    LOG.info(f"Run {service} load")
    kubeconfig_name, kubeconfig = cluster.get_kubeconfig_from_secret()
    with open(kubeconfig_name, 'w') as f:
        f.write(kubeconfig)

    bootstrap_manager = BootstrapManager.get_si_config_bootstrap_manager()
    os_manager = OpenStackManager(kubeconfig=kubeconfig_name)
    load_manager = LoadToolsManager(bootstrap_manager=bootstrap_manager, os_manager=os_manager)

    load_manager.run_loadtools_on_seed_node_mcc_only(tag=service)
    getattr(load_manager, "start_swarm_for_lma")(**{service + '_url': cluster.__getattribute__(service + '_url'),
                                                    'keycloak_url': keycloak_url,
                                                    'iam_user': settings.LMA_DOWNTIME_USER,
                                                    'iam_password': iam_password})

    start_time_str = time.strftime("%Y%m%d%H%M%S")
    start_time_timestamp = time.time()

    yield

    end_time_str = time.strftime("%Y%m%d%H%M%S")
    end_time_timestamp = time.time()
    clusterrelease_after_test = cluster.clusterrelease_version
    if not cluster.is_management:
        is_patchrelease = cluster.is_patchrelease_upgrade(
            clusterrelease_version_before=clusterrelease_before_test,
            clusterrelease_version_after=clusterrelease_after_test)
    else:
        is_patchrelease = False
    release_key = 'patch_release' if is_patchrelease else 'major_release'

    try:
        load_manager.stop_swarm()
        load_manager.wait_loadtools_available()
        statistics = load_manager.get_full_history()

        # Save statistic data
        store_statistics(statistic_data=statistics,
                         statistic_endpoint_name=service,
                         statistic_endpoint_url=cluster.__getattribute__(service + '_url'),
                         test_name=func_name,
                         cluster_name=cluster.name,
                         cluster_namespace=cluster.namespace,
                         cluster_provider_name=cluster.provider.provider_name,
                         version_before_test=clusterrelease_before_test,
                         version_after_test=clusterrelease_after_test,
                         release_key=release_key,
                         start_time_str=start_time_str,
                         start_time_timestamp=start_time_timestamp,
                         end_time_str=end_time_str,
                         end_time_timestamp=end_time_timestamp)

    except Exception as e:
        raise e
    finally:
        load_manager.save_docker_logs(f"loadtools_{end_time_str}.log")
        load_manager.exit()


def run_load_test_keycloak(kaas_manager, func_name):
    service = 'keycloak'
    cluster = kaas_manager.get_mgmt_cluster()
    clusterrelease_before_test = cluster.clusterrelease_version
    keycloak_ip = kaas_manager.get_keycloak_ip()
    keycloak_url = f"https://{keycloak_ip}"
    iam_password = kaas_manager.si_config.get_keycloak_user_password(settings.LMA_DOWNTIME_USER)

    LOG.info("Run keycloak load")
    kubeconfig_name, kubeconfig = cluster.get_kubeconfig_from_secret()
    with open(kubeconfig_name, 'w') as f:
        f.write(kubeconfig)

    bootstrap_manager = BootstrapManager.get_si_config_bootstrap_manager()
    os_manager = OpenStackManager(kubeconfig=kubeconfig_name)
    load_manager = LoadToolsManager(bootstrap_manager=bootstrap_manager, os_manager=os_manager)

    load_manager.run_loadtools_on_seed_node_mcc_only(tag=service)
    getattr(load_manager, "start_swarm_for_keycloak")(**{'keycloak_url': keycloak_url,
                                                         'iam_user': settings.LMA_DOWNTIME_USER,
                                                         'iam_password': iam_password})

    start_time_str = time.strftime("%Y%m%d%H%M%S")
    start_time_timestamp = time.time()

    yield

    end_time_str = time.strftime("%Y%m%d%H%M%S")
    end_time_timestamp = time.time()
    clusterrelease_after_test = cluster.clusterrelease_version
    release_key = 'major_release'

    try:
        load_manager.stop_swarm()
        load_manager.wait_loadtools_available()
        statistics = load_manager.get_full_history()

        # Save statistic data
        store_statistics(statistic_data=statistics,
                         statistic_endpoint_name=service,
                         statistic_endpoint_url=keycloak_url,
                         test_name=func_name,
                         cluster_name=cluster.name,
                         cluster_namespace=cluster.namespace,
                         cluster_provider_name=cluster.provider.provider_name,
                         version_before_test=clusterrelease_before_test,
                         version_after_test=clusterrelease_after_test,
                         release_key=release_key,
                         start_time_str=start_time_str,
                         start_time_timestamp=start_time_timestamp,
                         end_time_str=end_time_str,
                         end_time_timestamp=end_time_timestamp)

    except Exception as e:
        raise e
    finally:
        load_manager.save_docker_logs(f"loadtools_{end_time_str}.log")
        load_manager.exit()


def run_mcc_loadtest_keystone(kaas_manager, func_name):
    """
    Check keystone public endpoint
    Use for MCC-based clusters (KUBECONFIG_PATH contains Management cluster config)
    """
    if not settings.ALLOW_WORKLOAD:
        LOG.info("Skip Keystone load")
        yield
        return
    mgmt_cluster = kaas_manager.get_mgmt_cluster()
    clusterrelease_before_test = mgmt_cluster.clusterrelease_version
    child_clusters = kaas_manager.get_child_clusters()
    containers_map = {}
    if not child_clusters:
        LOG.info('No child clusters found. Skipping Keystone load test')
        yield
        return
    for cluster in child_clusters:
        cluster_name = cluster.name
        cluster_namespace = cluster.namespace
        managed_ns = kaas_manager.get_namespace(cluster_namespace)
        cluster = managed_ns.get_cluster(cluster_name)
        child_clusterrelease = cluster.clusterrelease_version
        if "mos" not in child_clusterrelease:
            LOG.info(f"Skipping fixture 'mcc_loadtest_keystone': expected MOSK cluster, but provided "
                     f"{cluster_namespace}/{cluster_name} "
                     f"with clusterrelease {cluster.clusterrelease_version}")
            yield
            return
        LOG.info(f"Start Keystone load for cluster {cluster_name} in namespace {cluster_namespace}")
        child_kubeconfig_name, child_kubeconfig = cluster.get_kubeconfig_from_secret()
        with open(child_kubeconfig_name, 'w') as f:
            f.write(child_kubeconfig)
        bootstrap_manager = BootstrapManager.get_si_config_bootstrap_manager()
        os_manager = OpenStackManager(kubeconfig=child_kubeconfig_name)
        openstack_ingress = os_manager.get_ingress_svc_external_ip()
        load_manager = LoadToolsManager(bootstrap_manager=bootstrap_manager, os_manager=os_manager)
        container_name, loadtools_url = load_manager.run_loadtools_for_keystone_check(
            cluster_name=cluster_name, cluster_namespace=cluster_namespace)
        load_manager.start_swarm_for_keystone(endpoint_ip=openstack_ingress, user_count=1)
        start_time_str = time.strftime("%Y%m%d%H%M%S")
        start_time_timestamp = time.time()
        containers_map[container_name] = {'cluster_namespace': cluster_namespace,
                                          'cluster_name': cluster_name,
                                          'start_time_str': start_time_str,
                                          'start_time_timestamp': start_time_timestamp,
                                          'loadtools_url': loadtools_url}
    yield
    end_time_str = time.strftime("%Y%m%d%H%M%S")
    end_time_timestamp = time.time()
    clusterrelease_after_test = mgmt_cluster.clusterrelease_version
    failed_cases = []
    for container_name, data in containers_map.items():
        try:
            loadtools_url = data.get('loadtools_url')
            cluster_namespace = data.get('cluster_namespace')
            cluster_name = data.get('cluster_name')
            start_time = data.get('start_time_str')
            start_time_timestamp = data.get('start_time_timestamp')
            load_manager.stop_swarm(loadtools_url=loadtools_url)
            load_manager.wait_loadtools_available(loadtools_url=loadtools_url)
            statistics = load_manager.get_full_history(loadtools_url=loadtools_url)
            # Save statistic data
            store_statistics(statistic_data=statistics,
                             statistic_endpoint_name='keystone',
                             test_name=func_name,
                             cluster_name=cluster_name,
                             cluster_namespace=cluster_namespace,
                             cluster_provider_name='baremetal',
                             version_before_test=clusterrelease_before_test,
                             version_after_test=clusterrelease_after_test,
                             release_key='mos',
                             start_time_str=start_time,
                             start_time_timestamp=start_time_timestamp,
                             end_time_str=end_time_str,
                             end_time_timestamp=end_time_timestamp)
        except Exception as e:
            LOG.error(f"Failed to store statistic for cluster {cluster_name} in namespace {cluster_namespace}. "
                      f"Container name: {container_name}")
            failed_cases.append({'Message': e, 'cluster_name': cluster_name, 'cluster_namespace': cluster_namespace,
                                 'container_name': container_name})
        finally:
            load_manager.save_docker_logs(f"loadtools_{cluster_name}_{cluster_namespace}_{end_time_str}.log",
                                          container_name=container_name)
            load_manager.exit(loadtools_url=loadtools_url, container_name=container_name)
        if failed_cases:
            raise Exception(failed_cases)


@pytest.fixture(scope='function')
def mcc_loadtest_grafana(kaas_manager, func_name):
    if not settings.GRAFANA_DOWNTIME_CHECK:
        LOG.info("Skip Grafana load, please set GRAFANA_DOWNTIME_CHECK in True")
        yield
        return
    yield from run_load_test(kaas_manager, func_name, service='grafana')


@pytest.fixture(scope='function')
def mcc_loadtest_prometheus(kaas_manager, func_name):
    if not settings.PROMETHEUS_DOWNTIME_CHECK:
        LOG.info("Skip Prometheus load, please set PROMETHEUS_DOWNTIME_CHECK in True")
        yield
        return
    yield from run_load_test(kaas_manager, func_name, service='prometheus')


@pytest.fixture(scope='function')
def mcc_loadtest_alerta(kaas_manager, func_name):
    if not settings.ALERTA_DOWNTIME_CHECK:
        LOG.info("Skip Alerta load, please set ALERTA_DOWNTIME_CHECK in True")
        yield
        return
    yield from run_load_test(kaas_manager, func_name,  service='alerta')


@pytest.fixture(scope='function')
def mcc_loadtest_alertmanager(kaas_manager, func_name):
    if not settings.ALERTMANAGER_DOWNTIME_CHECK:
        LOG.info("Skip Alerta load, please set ALERTMANAGER_DOWNTIME_CHECK in True")
        yield
        return
    yield from run_load_test(kaas_manager, func_name,  service='alertmanager')


@pytest.fixture(scope='function')
def mcc_loadtest_kibana(kaas_manager, func_name):
    if not settings.KIBANA_DOWNTIME_CHECK:
        LOG.info("Skip Alerta load, please set KIBANA_DOWNTIME_CHECK in True")
        yield
        return
    yield from run_load_test(kaas_manager, func_name,  service='kibana')


@pytest.fixture(scope='function')
def mcc_loadtest_keycloak(kaas_manager, func_name):
    if not settings.KEYCLOAK_DOWNTIME_CHECK:
        LOG.info("Skip Keycloak load, please set KEYCLOAK_DOWNTIME_CHECK in True")
        yield
        return
    yield from run_load_test_keycloak(kaas_manager, func_name)


@pytest.fixture(scope='function')
def mcc_loadtest_keystone(kaas_manager, func_name):
    if not settings.SI_LOADTOOLS_CHECK_KEYSTONE_WORKLOAD:
        LOG.info("Skip Keystone endpoint check, please set SI_LOADTOOLS_CHECK_KEYSTONE_WORKLOAD in True")
        yield
        return
    yield from run_mcc_loadtest_keystone(kaas_manager, func_name)
