import enum
import json
import re
from collections import namedtuple

import yaml
import pytest

from deepdiff import DeepDiff
from tabulate import tabulate

from si_tests import settings
from si_tests import logger
from si_tests.utils.utils import Provider
from si_tests.utils import utils
from si_tests.managers import introspect_manager


LOG = logger.logger


def _introspect_MKE_7914(openstack_client):
    """Check for UCP installer failures evidences"""
    JIRA = "MKE-7914"

    seed_ip = openstack_client.get_seed_ip()
    introspect = introspect_manager.IntrospectBootstrap(openstack_client, JIRA, seed_ip)
    introspect.init()
    bastion_ip = introspect.bastion_ip
    private_key_path = introspect.private_key_path
    PREF = introspect.PREF
    remote = introspect.remote

    collected_data = {}

    for mgmt_ip in introspect.mgmt_ips:
        ssh_cmd = introspect.get_ssh_cmd(mgmt_ip, private_key_path, bastion_ip)
        details = introspect.get_openstack_instance_details(ssh_cmd, mgmt_ip)
        if not details:
            continue
        hostname = details['hostname']
        collected_data[mgmt_ip] = details

        LOG.debug(f"{PREF} Show docker containers on the mgmt node")
        try:
            res = remote.check_call(f"{ssh_cmd} 'docker ps --all || true'")
            collected_data[mgmt_ip]['containers'] = res.stdout_str.splitlines()
        except Exception as e:
            LOG.error(f"{PREF} Unable to get docker containers list "
                      f"from {mgmt_ip}: {e}")
            continue

        LOG.debug(f"{PREF} Check for the bug evidence")
        try:
            res = remote.execute(
                f"{ssh_cmd} \"sudo fgrep -R -B3 "
                f"'expected response status code 200 but got 500' "
                f"/var/log/lcm/runners/\"")

            res1 = remote.execute(
                f"{ssh_cmd} \"sudo fgrep 'OCI runtime create failed' "
                f"/var/log/syslog\"")
            if not res.stdout and not res1.stdout:
                continue
            collected_data[mgmt_ip]["/var/log/lcm/runners/"] = res.stdout_str
            collected_data[mgmt_ip]["/var/log/syslog"] = res1.stdout_str
            # Something happened
            if res.stdout_str:
                LOG.info(f"{PREF} Found evidence of the bug on the node "
                         f"{hostname} in LCM logs:\n{res.stdout_str}")
            if res1.stdout_str:
                LOG.info(f"{PREF} Found evidence of the bug on the node "
                         f"{hostname} in /var/log/syslog:\n{res1.stdout_str}")

                res2 = remote.execute(
                    f"{ssh_cmd} docker ps --all | fgrep 'Exited'")
                if res2.stdout_str:
                    collected_data[mgmt_ip][
                        "docker_ps_exited"] = res2.stdout_str
                    LOG.info(f"{PREF} Containers in 'Exited' status on "
                             f"the node {hostname}:\n{res2.stdout_str}")

        except Exception as e:
            LOG.error(f"{PREF} Unable to check the evidence of the bug "
                      f"on the node {hostname}: {e}")
            continue

    LOG.debug(f"{PREF} Save the collected data to artifacts")
    LOG.debug(yaml.dump(collected_data))
    with open(f'{settings.ARTIFACTS_DIR}/introspect_{JIRA}.yaml',
              mode='w') as f:
        f.write(yaml.dump(collected_data))


def _introspect_PRODX_9238(openstack_client):
    """Measure disk speed"""
    JIRA = "PRODX-9238"

    seed_ip = openstack_client.get_seed_ip()
    introspect = introspect_manager.IntrospectBootstrap(openstack_client, JIRA, seed_ip)
    introspect.init()
    bastion_ip = introspect.bastion_ip
    private_key_path = introspect.private_key_path
    PREF = introspect.PREF
    remote = introspect.remote

    collected_data = {}

    for mgmt_ip in introspect.mgmt_ips:
        ssh_cmd = introspect.get_ssh_cmd(mgmt_ip, private_key_path, bastion_ip)
        details = introspect.get_openstack_instance_details(ssh_cmd, mgmt_ip)
        if not details:
            continue
        hostname = details['hostname']
        hypervisor_hostname = details.get(
            'os', {}).get('hypervisor_hostname', '-')
        collected_data[mgmt_ip] = details

        collected_data[mgmt_ip]['disk_speed'] = {}

        LOG.debug(f"{PREF} Perform the local disk speed measurement "
                  f"on the mgmt node ({JIRA})")
        try:
            # Default write, as most of applications do
            res = remote.check_call(
                f"{ssh_cmd} 'sudo dd if=/dev/urandom of=/.dd_speedtest_file "
                f"bs=1024 count=100000 2>&1|tail -n1'")
            write_default = res.stdout_str
            collected_data[mgmt_ip]['disk_speed'][
                'write_default'] = res.stdout_str

            # Write with direct sync, as a database server may be configured
            res = remote.check_call(
                f"{ssh_cmd} 'sudo dd if=/dev/urandom of=/.dd_speedtest_file "
                f"bs=1024 count=1000 oflag=dsync 2>&1|tail -n1'")
            write_direct_sync = res.stdout_str
            collected_data[mgmt_ip]['disk_speed'][
                'write_direct_sync'] = res.stdout_str

            w_default = write_default.split(",")[-1]
            w_dsync = write_direct_sync.split(",")[-1]

            LOG.info(f"{PREF} "
                     f"{hypervisor_hostname}/{hostname}: "
                     f"{w_dsync} (dsync) / {w_default} (default)")

            remote.check_call(f"{ssh_cmd} 'sudo rm /.dd_speedtest_file'")
        except Exception as e:
            LOG.error(f"{PREF} Unable to collect the disk speed info "
                      f"from {mgmt_ip}: {e}")
            continue

    LOG.debug(f"{PREF} Save the collected data to artifacts")
    LOG.debug(yaml.dump(collected_data))
    with open(f'{settings.ARTIFACTS_DIR}/introspect_{JIRA}.yaml',
              mode='w') as f:
        f.write(yaml.dump(collected_data))


def _get_ipam_data(kaas_manager, clusters):
    data = {}  # resulting data

    for cluster in clusters:
        if cluster.provider not in (Provider.baremetal, Provider.equinixmetalv2):
            LOG.warning(f"Skip IPAM check for provider "
                        f"{cluster.provider} in {cluster.name}")
            continue

        data[cluster.name] = {}
        LOG.info(f"Collecting IPAM data from cluster {cluster.name}")

        ns = kaas_manager.get_namespace(cluster.namespace)
        ipamhosts = [host.read() for host in ns.get_ipam_hosts()]
        # ipaddrs = [addr.read() for addr in ns.get_ipam_ipaddrs()]
        # subnets = ns.get_ipam_subnets()

        for ipamhost in ipamhosts:
            nics_data = {
                'netconfig': []
            }
            nics_data['netconfig'] = utils.get_np_struct_from_netconfigfiles(ipamhost.status.get('netconfigFiles', []))

            ipamhost_name = ipamhost.metadata.name
            data[cluster.name][ipamhost_name] = nics_data

    LOG.debug(yaml.dump(data))
    return data


def _get_hosts_data(kaas_manager, clusters):
    data = {}  # resulting data
    for cluster in clusters:
        if cluster.provider not in (Provider.baremetal, Provider.equinixmetalv2):
            LOG.warning(f"Skip network config check for provider "
                        f"{cluster.provider} in {cluster.name}")
            continue

        data[cluster.name] = {
            'machines': {},
            'loadbalancer': '',
        }
        provider_status = cluster.data['status']['providerStatus']
        loadbalancer = provider_status.get('loadBalancerIP')
        data[cluster.name]['loadbalancer'] = loadbalancer
        LOG.info(f"Cluster: {cluster.name} with loadbalancer {loadbalancer}")

        machines = cluster.get_machines()
        for machine in machines:
            if machine.machine_status not in ['Ready', 'Prepare']:
                LOG.error(f"Skipping machine {machine.name} because of wrong "
                          f"status: {machine.machine_status}")
                continue

            nics_data = {
                'machine_types': machine.machine_types,
                'gateway': '',
                'vip': '',
                'ifaces': {},
            }

            # Get the addresses from interfaces
            # TODO(ddmitriev): consider to use a daemonset, or job, or a pod
            #     with enabled hostnetwork to get network configuration from
            #     the cluster without ssh access
            #     PRODX-11650
            private_key = settings.KAAS_CHILD_CLUSTER_PRIVATE_KEY_FILE
            username = settings.KAAS_CHILD_CLUSTER_SSH_LOGIN
            res = machine._run_cmd("ip -j a", verbose=False,
                                   ssh_key=private_key, ssh_login=username)
            ip_a = res.stdout_str
            ips = json.loads(ip_a)

            vip = ''
            for nic in ips:
                addresses_ipv4 = []
                addresses_ipv6 = []
                nic_name = nic.get('ifname')
                mac = nic.get('address')

                if 'addr_info' not in nic:
                    continue
                if len(nic['addr_info']) == 0:
                    continue
                if nic_name is None:
                    LOG.error(f"Interface with MAC {mac} on the machine "
                              f"{cluster.name}/{machine.name} don't have "
                              f"the 'ifname' attribute while has "
                              f"some addresses assigned to it, skipping")
                    continue

                for addr in nic['addr_info']:
                    # VIP properties: no broadcast, /32, unique
                    if addr['local'] == loadbalancer:
                        LOG.info(f"Found VIP {addr['local']} on the node "
                                 f"{cluster.name}/{machine.name} "
                                 f"iface {nic_name}")
                        if vip:
                            raise Exception(f"VIP {addr['local']} is found on "
                                            f"multiple interfaces on the node "
                                            f"{cluster.name}/{machine.name}")
                        vip = addr['local']
                    else:
                        address = f"{addr['local']}/{addr['prefixlen']}"
                        if addr['family'] == "inet6":
                            addresses_ipv6.append(address)
                        else:
                            addresses_ipv4.append(address)

                # assume that there is no two interfaces with the same name on
                # the same node
                nics_data['ifaces'][nic_name] = {
                    'addresses_ipv4': sorted(addresses_ipv4),
                    'addresses_ipv6': sorted(addresses_ipv6),
                    'mac': mac,
                }

            nics_data['vip'] = vip

            # Get the default gateway address
            res = machine._run_cmd(
                "route -n | grep ' UG ' | grep -e '^0.0.0.0'",
                verbose=False, ssh_key=private_key, ssh_login=username)
            ip_r = res.stdout_str
            default_gws = [route for route in ip_r.splitlines()]
            if len(default_gws) > 1:
                raise Exception(f"More than one default gateways found "
                                f"on the node {machine.name}: {default_gws}")
            elif len(default_gws) == 1:
                default_gw = default_gws[0].split()[1]
                nics_data['gateway'] = default_gw

            data[cluster.name]['machines'][machine.name] = nics_data

    LOG.debug(yaml.dump(data))
    return data


@pytest.fixture(scope='session')
def introspect_MKE_7914(openstack_client):
    """Check the result of management cluster bootstrap test

    Collect the data specific to the bug:
    https://mirantis.jira.com/browse/MKE-7914
    """
    # pass the bootstrap test first
    yield

    try:
        _introspect_MKE_7914(openstack_client)
    except Exception as e:
        LOG.error(f"Exception in introspect_MKE_7914:\n{e}")
        # Do not fail the testcase because of fixture errors
        pass


@pytest.fixture(scope='session')
def introspect_PRODX_9238(openstack_client):
    """Check the result of management cluster bootstrap test

    Collect the data specific to the bug:
    https://mirantis.jira.com/browse/PRODX-9238
    """
    # pass the bootstrap test first
    yield

    try:
        _introspect_PRODX_9238(openstack_client)
    except Exception as e:
        LOG.error(f"Exception in introspect_PRODX_9238:\n{e}")
        # Do not fail the testcase because of fixture errors
        pass


class IfaceCheckFlags(enum.Enum):
    """Prefixes for interfaces that do not require some types of checks
    Prefix:
        iface_prefix: Interface name prefix to match
    Check flags:
        check_iface: Expect the presense of the same interface after upgrade
        check_mac: Expect that the interface has the same MAC after upgrade
        check_ipv4: Expect that the interface has the same IPv4 addresses after upgrade
        check_ipv6: Expect that the interface has the same IPv6 addresses after upgrade
        check_ipv6_link_local: Expect that the interface has the same IPv6 link-local address fe80:: after upgrade
    Message:
        text: String with comment to log
    """
    pkt = ("pkt", False, False, False, False, False,
           "[PRODX-17948] Ignore TungstenFabric virtual interface")
    vhost0 = ("vhost0", True, False, True, False, False,
              "[PRODX-26788] Ignore TungstenFabric virtual interface IPv6 and MAC addresses checks")
    br_phy = ("br-phy", True, False, True, False, False,
              "[PRODX-27083] Ignore OVS tunnel interface IPv6 and MAC addresses checks")
    k8s_ifaces = ("k8s-", True, False, True, False, False,
                  "Ignore IPv6 and MAC addresses checks for custom configured interfaces for k8s")
    ceph_ifaces = ("ceph-", True, False, True, False, False,
                   "Ignore IPv6 and MAC addresses checks for custom configured interfaces for ceph")
    docker = ("docker", False, False, False, False, False,
              "Skip Docker bridge interface")
    docker_br = ("br-", False, False, False, False, False,
                 "Skip Docker bridge interface")
    docker_veth = ("veth", False, False, False, False, False,
                   "Skip Docker veth interface")
    calico_tunnel = ("cali", False, False, False, False, False,
                     "Skip Calico tunnel interface")
    vxlan_tunnel = ("vxlan", False, False, False, False, False,
                    "Skip Calico vxlan tunnel interface")
    tap_iface = ("tap", False, False, False, False, False,
                 "Skip OpenStack 'tap' interface")
    _default = ("_default", True, True, True, True, False,
                "All checks are enabled for the interface")

    def __init__(self, iface_prefix, check_iface, check_mac, check_ipv4, check_ipv6, check_ipv6_link_local, text):
        self.iface_prefix = iface_prefix
        self.check_iface = check_iface
        self.check_mac = check_mac
        self.check_ipv4 = check_ipv4
        self.check_ipv6 = check_ipv6
        self.check_ipv6_link_local = check_ipv6_link_local
        self.text = text

    @classmethod
    def get_iface_check_flags(cls, iface_name):
        return next((e for e in cls if iface_name.startswith(e.iface_prefix)), cls._default)


@pytest.fixture(scope='session')
def introspect_PRODX_9696_mgmt(kaas_manager):
    """Check that the network configuration is not changed during the test"""

    # ------------------
    # 1. Get info from ipamhosts: MAC addresses, interfaces and IPs
    # 2. TODO(ddmitriev, PRODX-11595): Check that ipaddrs contain the same
    #    address as in imaphosts
    # 3. Check that BM hosts has the same IP address on the specified
    #    interfaces from ipamhosts
    # 4. Check that BM host has the 'default' route which match the gateway
    #    from ipamhosts
    # 5. Check that a BM host has the IP address for keepalived VIP:
    #    cluster.data['status']['providerStatus']['loadBalancerIP']
    #    which is unique and is *ONLY* on 'control' node types

    if settings.DISABLE_INTROSPECT_PRODX_9696:
        LOG.warning("[PRODX-9696] introspect_PRODX_9696 disabled")
        yield
        return

    clusters = kaas_manager.get_clusters()

    # Collect data before the test
    ipam_data1 = _get_ipam_data(kaas_manager, clusters)

    if not ipam_data1.keys():
        LOG.warning("[PRODX-9696] Skip network config check "
                    "because no clusters with suitable provider found")
        yield
        return

    if settings.DISABLE_IP_CONSISTENCY_CHECK:
        LOG.warning("PRODX_9696: Check for IP addresses on nodes is skipped")
        hosts_data1 = {}
    else:
        hosts_data1 = _get_hosts_data(kaas_manager, clusters)

    # Run the test
    yield

    # Collect data after the test
    ipam_data2 = _get_ipam_data(kaas_manager, clusters)
    if settings.DISABLE_IP_CONSISTENCY_CHECK:
        LOG.warning("PRODX_9696: Check for IP addresses on nodes is skipped")
        hosts_data2 = {}
    else:
        hosts_data2 = _get_hosts_data(kaas_manager, clusters)

    _check_introspect_PRODX_9696(ipam_data1, ipam_data2, hosts_data1, hosts_data2)


@pytest.fixture(scope='session')
def introspect_PRODX_9696_target_cluster(kaas_manager):
    """Check that the network configuration is not changed during the test for target and parent clusters"""

    # ------------------
    # 1. Get info from ipamhosts: MAC addresses, interfaces and IPs
    # 2. TODO(ddmitriev, PRODX-11595): Check that ipaddrs contain the same
    #    address as in imaphosts
    # 3. Check that BM hosts has the same IP address on the specified
    #    interfaces from ipamhosts
    # 4. Check that BM host has the 'default' route which match the gateway
    #    from ipamhosts
    # 5. Check that a BM host has the IP address for keepalived VIP:
    #    cluster.data['status']['providerStatus']['loadBalancerIP']
    #    which is unique and is *ONLY* on 'control' node types

    if settings.DISABLE_INTROSPECT_PRODX_9696:
        LOG.warning("[PRODX-9696] introspect_PRODX_9696 disabled")
        yield
        return

    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    cluster = ns.get_cluster(settings.TARGET_CLUSTER)
    parent_cluster = cluster.get_parent_cluster()
    mgmt_cluster = kaas_manager.get_mgmt_cluster()
    if mgmt_cluster.name == parent_cluster.name and mgmt_cluster.namespace == parent_cluster.namespace:
        clusters = [mgmt_cluster, cluster]
    else:
        clusters = [mgmt_cluster, parent_cluster, cluster]

    # Collect data before the test
    ipam_data1 = _get_ipam_data(kaas_manager, clusters)

    if not ipam_data1.keys():
        LOG.warning("[PRODX-9696] Skip network config check "
                    "because no clusters with suitable provider found")
        yield
        return

    if settings.DISABLE_IP_CONSISTENCY_CHECK:
        LOG.warning("PRODX_9696: Check for IP addresses on nodes is skipped")
        hosts_data1 = {}
    else:
        hosts_data1 = _get_hosts_data(kaas_manager, clusters)

    # Run the test
    yield

    # Collect data after the test
    ipam_data2 = _get_ipam_data(kaas_manager, clusters)
    if settings.DISABLE_IP_CONSISTENCY_CHECK:
        LOG.warning("PRODX_9696: Check for IP addresses on nodes is skipped")
        hosts_data2 = {}
    else:
        hosts_data2 = _get_hosts_data(kaas_manager, clusters)

    _check_introspect_PRODX_9696(ipam_data1, ipam_data2, hosts_data1, hosts_data2)


def _check_introspect_PRODX_9696(ipam_data1, ipam_data2, hosts_data1, hosts_data2):
    errors = []
    # 1. Compare IPAM data
    for cluster_name1 in ipam_data1.keys():
        if cluster_name1 not in ipam_data2:
            LOG.error(f"Cluster {cluster_name1} is disappeared after test,"
                      " skipping check for the cluster")
            continue
        ipam_hosts1 = ipam_data1[cluster_name1]
        ipam_hosts2 = ipam_data2[cluster_name1]
        for ipam_host1 in ipam_hosts1.keys():
            if ipam_host1 not in ipam_hosts2:
                errors.append(f"Ipamhost {cluster_name1}/{ipam_host1} "
                              f"disappeared from ipamhosts after test")
                continue
            try:
                diff = DeepDiff(
                    ipam_hosts1[ipam_host1]['netconfig'],
                    ipam_hosts2[ipam_host1]['netconfig'],
                    ignore_order=True,
                )
                assert diff == {}, "Netplan configuration has been changed"
            except AssertionError as e:
                errors.append(
                    f"Netconfig on the Ipamhost {cluster_name1}/"
                    f"{ipam_host1} has been changed: {e}")

    # 2. Compare hosts data
    for cluster_name1 in hosts_data1.keys():
        if cluster_name1 not in hosts_data2:
            # do not double the error message about missing cluster
            continue
        cluster_data1 = hosts_data1[cluster_name1]
        cluster_data2 = hosts_data2[cluster_name1]
        if cluster_data1['loadbalancer'] != cluster_data2['loadbalancer']:
            errors.append(f"Loadbalancer is changed on the cluster "
                          f"{cluster_name1} after test, was "
                          f"{cluster_data1['loadbalancer']}, "
                          f"became {cluster_data2['loadbalancer']}")

        machines_data1 = cluster_data1['machines']
        machines_data2 = cluster_data2['machines']
        vip1 = ''
        vip1_machine_name = ''
        vip2 = ''
        vip2_machine_name = ''
        for machine_name1 in machines_data1.keys():
            if machine_name1 not in machines_data2:
                errors.append(f"Machine {machine_name1} disappeared after "
                              f"test from the cluster {cluster_name1}")
                continue
            machine1 = machines_data1[machine_name1]
            machine2 = machines_data2[machine_name1]
            if machine1['gateway'] != machine2['gateway']:
                errors.append(
                    f"Default gateway on the machine {cluster_name1}/"
                    f"{machine_name1} has been changed from "
                    f"{machine1['gateway']} to {machine2['gateway']}")

            # Check VIP on machines before test
            if machine1['vip']:
                if vip1 != '':
                    errors.append(
                        f"Found VIP assigned to multiple machines in the "
                        f"cluster {cluster_name1} before test: {vip1} on the "
                        f"machine {vip1_machine_name} and {machine1['vip']} "
                        f"on the machine {machine_name1}")
                else:
                    vip1 = machine1['vip']
                    vip1_machine_name = machine_name1

                if 'control' not in machine1['machine_types']:
                    errors.append(
                        f"Found VIP assigned to the non-controller machine "
                        f"in the cluster {cluster_name1} before test: "
                        f"{machine1['vip']} on the machine {machine_name1} "
                        f"({machine1['machine_types']})")

            # Check VIP on machines after test
            if machine2['vip']:
                if vip2 != '':
                    errors.append(
                        f"Found VIP assigned to multiple machines in the "
                        f"cluster {cluster_name1} after test: {vip2} on the "
                        f"machine {vip2_machine_name} and {machine2['vip']} "
                        f"on the machine {machine_name1}")
                else:
                    vip2 = machine2['vip']
                    vip2_machine_name = machine_name1

                if 'control' not in machine2['machine_types']:
                    errors.append(
                        f"Found VIP assigned to the non-controller machine in "
                        f"the cluster {cluster_name1} after test: "
                        f"{machine2['vip']} on the machine {machine_name1} "
                        f"({machine2['machine_types']})")

            ifaces1 = machine1['ifaces']
            ifaces2 = machine2['ifaces']
            for iface1_name in ifaces1.keys():
                iface_check_flags = IfaceCheckFlags.get_iface_check_flags(iface1_name)
                if not iface_check_flags.check_iface:
                    LOG.debug(f"Skipping check for interface '{iface1_name}' presense "
                              f"on '{machine_name1}': {iface_check_flags.text}\n"
                              f"Before update: {ifaces1.get('iface1_name')}\n"
                              f"After update: {ifaces2.get('iface1_name')}")
                    continue

                if iface1_name not in ifaces2:
                    errors.append(
                        f"Interface {iface1_name} on the machine "
                        f"{cluster_name1}/{machine_name1} disappeared "
                        f"after test, was: {ifaces1[iface1_name]}")
                    continue

                ifmac1 = ifaces1[iface1_name]['mac']
                ifmac2 = ifaces2[iface1_name]['mac']
                if ifmac1 != ifmac2:
                    if iface_check_flags.check_mac:
                        errors.append(
                            f"Interface {iface1_name} on the machine "
                            f"{cluster_name1}/{machine_name1} changed "
                            f"the MAC: was {ifmac1}, became {ifmac2}")
                    else:
                        LOG.debug(f"Skipping check for interface '{iface1_name}' MAC address "
                                  f"on '{machine_name1}': {iface_check_flags.text}\n"
                                  f"Before update: {ifaces1.get('iface1_name', dict()).get('mac')}\n"
                                  f"After update: {ifaces2.get('iface1_name', dict()).get('mac')}")

                addresses1_ipv4 = ifaces1[iface1_name]['addresses_ipv4']
                addresses2_ipv4 = ifaces2[iface1_name]['addresses_ipv4']
                if addresses1_ipv4 != addresses2_ipv4:
                    if iface_check_flags.check_ipv4:
                        errors.append(
                            f"Interface {iface1_name} on the machine "
                            f"{cluster_name1}/{machine_name1} changed the IPv4 "
                            f"address configuration: was {addresses1_ipv4}, "
                            f"became {addresses2_ipv4}")
                    else:
                        LOG.debug(f"Skipping check for interface '{iface1_name}' IPv4 addresses "
                                  f"on '{machine_name1}': {iface_check_flags.text}\n"
                                  f"Before update: {ifaces1.get('iface1_name', dict()).get('addresses_ipv4')}\n"
                                  f"After update: {ifaces2.get('iface1_name', dict()).get('addresses_ipv4')}")

                addresses1_ipv6 = ifaces1[iface1_name]['addresses_ipv6']
                addresses2_ipv6 = ifaces2[iface1_name]['addresses_ipv6']
                if not iface_check_flags.check_ipv6_link_local:
                    LOG.debug(f"Skipping IPv6 link-local addresses for interface '{iface1_name}' "
                              f"on machine {machine_name1} if present:\n"
                              f"Before update: {addresses1_ipv6}\n"
                              f"After update: {addresses2_ipv6}")
                    addresses1_ipv6 = [addr for addr in addresses1_ipv6 if not addr.startswith('fe80::')]
                    addresses2_ipv6 = [addr for addr in addresses2_ipv6 if not addr.startswith('fe80::')]
                if addresses1_ipv6 != addresses2_ipv6:
                    if iface_check_flags.check_ipv6:
                        errors.append(
                            f"Interface {iface1_name} on the machine "
                            f"{cluster_name1}/{machine_name1} changed the IPv6 "
                            f"address configuration: was {addresses1_ipv6}, "
                            f"became {addresses2_ipv6}")
                    else:
                        LOG.debug(f"Skipping check for interface '{iface1_name}' IPv6 addresses "
                                  f"on '{machine_name1}': {iface_check_flags.text}\n"
                                  f"Before update: {ifaces1.get('iface1_name', dict()).get('addresses_ipv6')}\n"
                                  f"After update: {ifaces2.get('iface1_name', dict()).get('addresses_ipv6')}")

        if vip1 != vip2:
            errors.append(
                f"VIP address is changed on the cluster {cluster_name1}: "
                f"was {vip1} on the machine {vip1_machine_name}, "
                f"became {vip2} on the machine {vip2_machine_name}")

    # 3. TODO(ddmitriev, PRODX-11595): Compare IPAM and machines data

    # 4. Raise an exception if any error has been found
    if errors:
        msg = '\n'.join(errors)
        raise Exception(f"Found the following errors:\n{msg}")
    else:
        LOG.info("PRODX-9696: PASSED. Network settings in IPAM and machines "
                 "were not changed")


def _introspect_ceph(cluster):
    ceph_data = {'ceph_info': {}}
    if (cluster.is_management or cluster.is_regional)\
            and cluster.provider is Provider.equinixmetal:
        LOG.info("Equinix mgmt and regional clusters don't support Ceph. "
                 "Skipping data collection")
        pass
    elif cluster.provider in Provider.with_ceph():
        ceph_cwl_name = 'ceph-clusterworkloadlock'
        cwl_state = cluster.check.get_clusterworkloadlock_state(
            name=ceph_cwl_name)
        if cluster.workaround.skip_kaascephcluster_usage():
            miracephcluster_status = cluster.check.get_miraceph_phase()
            miraceph_health = cluster.check.get_miracephhealths_health_status()
            ceph_data['ceph_info']['miraceph_health'] = miraceph_health
            ceph_data['ceph_info']['miracephcluster_status'] =\
                miracephcluster_status
            ceph_data['ceph_info']['ceph_daemons_status'] =\
                cluster.check.get_miracephhealths_daemons_status()
            ceph_data['ceph_info']['clusterworkloadlock_state'] = cwl_state
            ceph_data['ceph_info']['ceph_version'] = cluster.get_miracephhealth_version()
        else:
            kaascephcluster_status = cluster.check.get_kaascephcluster_state()
            ceph_health = cluster.check.get_ceph_health_status()
            ceph_data['ceph_info']['ceph_health'] = ceph_health
            ceph_data['ceph_info']['kaascephcluster_status'] =\
                kaascephcluster_status
            ceph_data['ceph_info']['ceph_daemons_status'] =\
                cluster.check.get_ceph_daemons_status()
            ceph_data['ceph_info']['clusterworkloadlock_state'] = cwl_state
            ceph_data['ceph_info']['ceph_version'] = cluster.get_ceph_version()
            ceph_data['ceph_info']['kaas_cephstate_message'] = cluster.check.get_kaascephstate_message()
        LOG.info(f"Ceph cluster data: \n{yaml.dump(ceph_data)}")
        with open(
            f'{settings.ARTIFACTS_DIR}/introspect_ceph_{cluster.name}.yaml',
                mode='w') as f:
            f.write(yaml.dump(ceph_data))
    else:
        LOG.info(f"No Ceph data available for provider {cluster.provider}. "
                 f"Skipping data collection")
        pass


@pytest.fixture(scope='session')
def introspect_ceph_child(kaas_manager):
    """Collect Ceph data from child cluster"""

    # Wait for test is finished first
    yield
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)
    if cluster.is_ceph_deployed:
        _introspect_ceph(cluster)
    else:
        LOG.info("Child is deployed without Ceph support. Skip collection Ceph data from child cluster.")


@pytest.fixture(scope='session')
def collect_machines_timestamps(kaas_manager):
    """
    Collect lcmmachines phases timestamps before and after test
    :param kaas_manager:
    :return:
    """
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)
    dedicated_ctrlplane = cluster.data.get('spec').get(
        'providerSpec', {}).get('value', {}).get('dedicatedControlPlane', True)
    max_wkr_upgrade_cnt = cluster.max_worker_upgrade_count
    LOG.info("Collecting lcmmachines timestamps before test")
    timestamps_before = cluster.get_cluster_lcmmachines_timestamps()
    timestamps_map = {'before_test': timestamps_before,
                      'after_test': {},
                      'cluster_name': cluster_name,
                      'cluster_namespace': namespace_name,
                      'dedicated_control_plane': dedicated_ctrlplane,
                      'max_worker_upgrade_count': max_wkr_upgrade_cnt,
                      'is_ansible_version_changed': True}
    if not cluster.is_byo_child:
        ansible_version_before = cluster.get_cluster_machines_ansible_versions()

    yield

    max_wkr_upgrade_cnt_after_test = cluster.max_worker_upgrade_count
    if max_wkr_upgrade_cnt != max_wkr_upgrade_cnt_after_test:
        LOG.warning(f"Max worker upgrade count value was changed during test. "
                    f"Before test: maxWorkerUpgradeCount: {max_wkr_upgrade_cnt}"
                    f"After test: maxWorkerUpgradeCount: {max_wkr_upgrade_cnt_after_test}")
        timestamps_map.update({'max_worker_upgrade_count_after_test': max_wkr_upgrade_cnt_after_test})
    LOG.info("Collecting lcmmachines timestamps after test")
    timestamps_map['after_test'] = cluster.get_cluster_lcmmachines_timestamps()
    if not cluster.is_byo_child:
        ansible_version_after = cluster.get_cluster_machines_ansible_versions()
        if ansible_version_before == ansible_version_after:
            LOG.info(f"Ansible version before upgrade: {ansible_version_before} is the same "
                     f"as after: {ansible_version_after}, we can skip test_compare_indexes_timestamps "
                     f"test because we did nothing on machines during this update and for this purpose we "
                     f"add special flag into timestamps file")
            timestamps_map.update({'is_ansible_version_changed': False})

    with open(
            f'{settings.ARTIFACTS_DIR}/introspect_machines_timestamps.yaml',
            mode='w') as f:
        f.write(yaml.dump(timestamps_map))


@pytest.fixture(scope='session')
def introspect_child_deploy_events(kaas_manager):
    """Get events from a just deployed Cluster and it's parent"""
    yield
    cluster_name = settings.KAAS_CHILD_CLUSTER_NAME
    namespace_name = settings.KAAS_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)

    parent_events = cluster.get_parent_events()
    parent_cluster = cluster.get_parent_cluster()
    if parent_cluster.is_regional:
        parent_cluster_name = f"Regional cluster {parent_cluster.name}"
    else:
        parent_cluster_name = f"Management cluster {parent_cluster.name}"

    msg = f"Events from the {parent_cluster_name} for the Cluster {cluster.namespace}/{cluster.name}:"
    prefix = (f"\n\n"
              f"\n{'=' * len(msg)}"
              f"\n{msg}"
              f"\n{'=' * len(msg)}")
    parsed_parent_events = utils.parse_events(parent_events, filtered_events=True)
    events_msg = utils.create_events_msg(parsed_parent_events, prefix=prefix)
    LOG.info(events_msg)
    # Write all Parent events into artifacts
    all_parent_events = utils.parse_events(parent_events, filtered_events=False)
    all_events_msg = utils.create_events_msg(all_parent_events, prefix=prefix)
    with open(f'{settings.ARTIFACTS_DIR}/events_from_parent_cluster.txt', mode='w') as f:
        f.write(all_events_msg)

    if cluster.is_k8s_available():
        cluster_events = cluster.get_cluster_events()
        msg = f"Events from the Cluster {cluster.namespace}/{cluster.name}:"
        prefix = (f"\n\n"
                  f"\n{'=' * len(msg)}"
                  f"\n{msg}"
                  f"\n{'=' * len(msg)}")
        parsed_cluster_events = utils.parse_events(cluster_events, filtered_events=True)
        events_msg = utils.create_events_msg(parsed_cluster_events, prefix=prefix)
        LOG.info(events_msg)
        # Write all Cluster events into artifacts
        all_cluster_events = utils.parse_events(cluster_events, filtered_events=False)
        all_events_msg = utils.create_events_msg(all_cluster_events, prefix=prefix)
        with open(f'{settings.ARTIFACTS_DIR}/events_from_child_cluster.txt', mode='w') as f:
            f.write(all_events_msg)
    else:
        LOG.error("Cluster events are not available, please check if the k8s API is working")


@pytest.fixture(scope='session')
def introspect_openstack_seed_management_deploy_events(openstack_client):
    """Get events from a just deployed Management cluster"""
    yield
    JIRA = "PRODX-23759"
    seed_ip = openstack_client.get_seed_ip()
    if not seed_ip:
        LOG.warning("No seed IP address provided, skipping Management cluster events introspection")
        return
    openstack_client = None
    introspect = introspect_manager.IntrospectBootstrap(openstack_client, JIRA, seed_ip)

    introspect.introspect_management_deploy_events()


@pytest.fixture(scope='session')
def introspect_standalone_seed_management_deploy_events():
    """Get events from a just deployed Management cluster"""
    yield
    JIRA = "PRODX-23759"
    seed_ip = settings.SEED_STANDALONE_EXTERNAL_IP
    if not seed_ip:
        LOG.warning("No seed IP address provided, skipping Management cluster events introspection")
        return
    openstack_client = None
    introspect = introspect_manager.IntrospectBootstrap(openstack_client, JIRA, seed_ip)

    introspect.introspect_management_deploy_events()


def _introspect_lcm_operation_stuck(cluster, cluster_name, namespace_name):
    """
    Found cluster and machine(s) with lcmOperationStuck flag.

    """
    LOG.info(f"Check cluster: {cluster.name}")
    stalled_lcm_machines_list = []
    stalled_lcm_clusters_list = []
    stalled_machine = namedtuple('stalled_machine', 'lcm_machine_name stuck_flag lcm_state_name lcm_state_type')
    stalled_cluster = namedtuple('stalled_cluster', 'lcm_cluster_name stuck_flag lcm_state_name lcm_state_type')
    lcm_machines = cluster.get_lcmmachines()
    lcm_cluster = cluster.get_lcm_cluster(name=cluster_name, namespace=namespace_name)
    lcm_cluster_states = {s.data.get('spec', {}).get('arg'): s for s in
                          cluster.get_lcmcluster_states(namespace=namespace_name)}

    if lcm_cluster:
        lcm_cluster_name = lcm_cluster.name
        lcm_cluster_status = lcm_cluster.data.get('status') or {}
        lcm_cluster_flag = lcm_cluster_status.get('lcmOperationStuck', False)

        if stalled_state := lcm_cluster_states.get(lcm_cluster_name):
            stalled_lcm_clusters_list.append(stalled_cluster(lcm_cluster_name, lcm_cluster_flag,
                                                             stalled_state.name,
                                                             stalled_state.data.get('spec', {}).get('type')
                                                             ))
        else:
            stalled_lcm_clusters_list.append(stalled_cluster(lcm_cluster_name, lcm_cluster_flag, '', ''))

    for lcm_machine in lcm_machines:
        lcm_machine_name = lcm_machine.data.get('metadata', {}).get('name')
        lcm_machine_status = lcm_machine.data.get('status') or {}
        lcm_machine_flag = lcm_machine_status.get('lcmOperationStuck', False)

        if stalled_state := lcm_cluster_states.get(lcm_machine_name):
            stalled_lcm_machines_list.append(stalled_machine(lcm_machine_name, lcm_machine_flag,
                                                             stalled_state.name,
                                                             stalled_state.data.get('spec', {}).get('type')))
        else:
            stalled_lcm_machines_list.append(stalled_machine(lcm_machine_name, lcm_machine_flag, '', ''))

    LOG.error("LCM cluster state:")
    LOG.error('\n' + tabulate(stalled_lcm_clusters_list, tablefmt="presto",
                              headers=[item.replace('_', ' ') for item in stalled_cluster._fields]))

    LOG.error("LCM machine state:")
    LOG.error('\n' + tabulate(stalled_lcm_machines_list, tablefmt="presto",
                              headers=[item.replace('_', ' ') for item in stalled_machine._fields]))


@pytest.fixture(scope='session')
def introspect_mgmt_lcm_operation_stuck(request, kaas_manager):
    yield

    if request.session.testsfailed:
        cluster = kaas_manager.get_mgmt_cluster()
        _introspect_lcm_operation_stuck(cluster, cluster.name, cluster.namespace)
        regional_clusters = kaas_manager.get_regional_clusters()
        for regional_cluster in regional_clusters:
            _introspect_lcm_operation_stuck(regional_cluster, regional_cluster.name, regional_cluster.namespace)


@pytest.fixture(scope='session')
def introspect_child_lcm_operation_stuck(request, kaas_manager):
    yield

    if request.session.testsfailed:
        managed_ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
        cluster = managed_ns.get_cluster(settings.TARGET_CLUSTER)
        _introspect_lcm_operation_stuck(cluster,
                                        settings.TARGET_CLUSTER, settings.TARGET_NAMESPACE)


@pytest.fixture(scope='session')
def introspect_openstack_seed_management_deploy_objects(openstack_client):
    """Inspect objects conditions from a just deployed Management cluster using seed node from OpenStack"""
    yield
    JIRA = "PRODX-25544"
    seed_ip = openstack_client.get_seed_ip()
    if not seed_ip:
        LOG.warning("No seed IP address provided, skipping Management cluster events introspection")
        return
    openstack_client = None
    introspect = introspect_manager.IntrospectBootstrap(openstack_client, JIRA, seed_ip)

    introspect.introspect_management_deploy_objects(settings.CLUSTER_NAME,
                                                    settings.CLUSTER_NAMESPACE,
                                                    seed_ip)


@pytest.fixture(scope='session')
def introspect_standalone_seed_management_deploy_objects():
    """Inspect objects conditions from a just deployed Management cluster using standalone seed node"""
    yield
    JIRA = "PRODX-25544"
    seed_ip = settings.SEED_STANDALONE_EXTERNAL_IP
    if not seed_ip:
        LOG.warning("No seed IP address provided, skipping Management cluster events introspection")
        return
    openstack_client = None
    introspect = introspect_manager.IntrospectBootstrap(openstack_client, JIRA, seed_ip)

    introspect.introspect_management_deploy_objects(settings.CLUSTER_NAME,
                                                    settings.CLUSTER_NAMESPACE,
                                                    seed_ip)


@pytest.fixture(scope='session')
def introspect_openstack_seed_regional_deploy_objects(openstack_client, kaas_manager):
    """Inspect objects conditions from a just deployed Regional cluster using seed node from OpenStack"""
    yield
    JIRA = "PRODX-25544"
    seed_ip = openstack_client.get_seed_ip()
    if not seed_ip:
        LOG.warning("No seed IP address provided, skipping Management cluster events introspection")
        return
    openstack_client = None
    introspect = introspect_manager.IntrospectBootstrap(openstack_client, JIRA, seed_ip)

    introspect.introspect_regional_deploy_objects(settings.REGIONAL_CLUSTER_NAME,
                                                  settings.REGION_NAMESPACE,
                                                  kaas_manager,
                                                  seed_ip)


@pytest.fixture(scope='session')
def introspect_standalone_seed_regional_deploy_objects(kaas_manager):
    """Inspect objects conditions from a just deployed Regional cluster using standalone seed node"""
    yield
    JIRA = "PRODX-25544"
    seed_ip = settings.SEED_STANDALONE_EXTERNAL_IP
    if not seed_ip:
        LOG.warning("No seed IP address provided, skipping Management cluster events introspection")
        return
    openstack_client = None
    introspect = introspect_manager.IntrospectBootstrap(openstack_client, JIRA, seed_ip)

    introspect.introspect_regional_deploy_objects(settings.REGIONAL_CLUSTER_NAME,
                                                  settings.REGION_NAMESPACE,
                                                  kaas_manager,
                                                  seed_ip)


@pytest.fixture(scope='session')
def introspect_management_upgrade_objects(kaas_manager):
    """Inspect objects conditions from Management and Regional clusters after upgrade"""
    yield
    JIRA = "PRODX-25544"
    seed_ip = None
    openstack_client = None
    introspect = introspect_manager.IntrospectBootstrap(openstack_client, JIRA, seed_ip)
    introspect.introspect_management_upgrade_objects(kaas_manager)


@pytest.fixture(scope='session')
def introspect_child_deploy_objects(kaas_manager):
    """Inspect objects conditions from a just deployed Child cluster"""
    yield
    JIRA = "PRODX-25544"
    seed_ip = None
    openstack_client = None
    introspect = introspect_manager.IntrospectBootstrap(openstack_client, JIRA, seed_ip)

    introspect.introspect_child_deploy_objects(settings.KAAS_CHILD_CLUSTER_NAME,
                                               settings.KAAS_NAMESPACE,
                                               kaas_manager)


@pytest.fixture(scope='session')
def introspect_child_target_objects(kaas_manager):
    """Inspect objects conditions from a TARGET_CLUSTER Child cluster"""
    yield
    JIRA = "PRODX-25544"
    seed_ip = None
    openstack_client = None
    introspect = introspect_manager.IntrospectBootstrap(openstack_client, JIRA, seed_ip)

    introspect.introspect_child_deploy_objects(settings.TARGET_CLUSTER,
                                               settings.TARGET_NAMESPACE,
                                               kaas_manager)


def _introspect_machines_stages(cluster, kaas_manager):
    """
    Found cluster and machine(s) with lcmOperationStuck flag.

    """
    LOG.info(f"Check cluster: {cluster.name}")
    cluster_release = cluster.clusterrelease_version
    LOG.info(f"Cluster release: {cluster_release}")

    machines_stages = []
    all_machines = cluster.get_machines()
    actual_version = cluster.clusterrelease_actual_version
    if 'rc' in actual_version:
        to_release = actual_version.replace('-rc+', '-')
    else:
        to_release = actual_version.replace('+', '-')

    for machine in all_machines:
        statuses = kaas_manager.get_machine_upgrade_status_crd(name=f"{machine.name}-{to_release}",
                                                               namespace=cluster.namespace)
        machines_stages.append({machine.name: statuses})

    for machine in machines_stages:
        for v, k in machine.items():
            LOG.info('\n' + f"Machine name: {v} ")
            LOG.info('\n' + f"Stages: {k} ")
            LOG.info('\n' + "============================")


@pytest.fixture(scope='session')
def introspect_machines_stages(request, kaas_manager):
    yield

    if request.session.testsfailed:
        ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
        cluster = ns.get_cluster(settings.TARGET_CLUSTER)
        _introspect_machines_stages(cluster, kaas_manager)


@pytest.fixture(scope='session')
def introspect_machines_stages_mgmt(request, kaas_manager):
    yield

    if request.session.testsfailed:
        cluster = kaas_manager.get_mgmt_cluster()
        _introspect_machines_stages(cluster, kaas_manager)
        regional_clusters = kaas_manager.get_regional_clusters()
        for regional_cluster in regional_clusters:
            _introspect_machines_stages(regional_cluster, kaas_manager)


@pytest.fixture(scope='session')
def introspect_distribution_not_changed(request, kaas_manager):
    """Ensure that distribution upgrade is not performed during ordinary LCM operations"""
    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    cluster = ns.get_cluster(settings.TARGET_CLUSTER)
    is_distro_upgrade_enabled = cluster.is_postpone_distribution_upgrade_enabled

    yield

    if request.session.testsfailed:
        LOG.warning("Skip distribution upgrade check because the test was failed")
    else:
        if is_distro_upgrade_enabled:
            LOG.info("Check that distribution upgrade haven't been started while postponeDistributionUpdate=True")
            cluster.check.check_inplace_distribution_upgrade_not_started()
        else:
            LOG.info("Skip distribution upgrade check while postponeDistributionUpdate=False")


def _introspect_PRODX_51933(cluster_name, namespace, kaas_manager):
    """Check task .* blocked for more than pattern in dmesg"""
    ns = kaas_manager.get_namespace(namespace)
    cluster_under_check = ns.get_cluster(cluster_name)
    machines_list = cluster_under_check.get_machines(raise_if_empty=False)
    pattern = "task .* blocked for more than"
    cmd_dmseg = "dmesg"
    cmd_grep = "dmesg | grep -P 'task .* blocked for more than'"
    potentially_issued_nodes = {}

    for machine in machines_list:
        matched_lines = []
        result_dmesg = machine.exec_pod_cmd(cmd_dmseg, verbose=False)['logs']
        LOG.debug(f"Result of executing dmesg cmd on "
                  f"machine {machine.name}: {result_dmesg}")
        matches = re.findall(pattern, result_dmesg)
        if matches:
            LOG.info(f'matches found for pattern {pattern} in {result_dmesg} '
                     f'on machine {machine.name}')
            matched_lines.append(result_dmesg)
            potentially_issued_nodes.update({machine.name: matched_lines})
        result_grep = machine.exec_pod_cmd(cmd_grep, verbose=False)['logs']
        LOG.debug(f"Result of executing grep cmd on "
                  f"machine {machine.name}: {result_grep}")
        if result_grep.strip():
            matched_lines.append(result_grep.strip())
            LOG.info(f"Additional matches found in dmesg over grep on machine: {machine.name}")
            potentially_issued_nodes.update({machine.name: matched_lines})

    LOG.info(f"Potentially issued nodes: {potentially_issued_nodes}")
    if machines_list:
        assert not potentially_issued_nodes, (f"There are some potentially "
                                              f"affected nodes by "
                                              f"PRODX_51933: {potentially_issued_nodes}")


@pytest.fixture(scope='session')
def introspect_no_PRODX_51933_after_lcm(request, kaas_manager):
    yield
    if settings.SKIP_TO_CHECK_PRODX_51933:
        LOG.info("Check for issue PRODX-51933 is skipped")
    else:
        LOG.banner(f"Check if there issue "
                   f"PRODX_51933 on nodes for "
                   f"cluster {settings.TARGET_CLUSTER}/{settings.TARGET_NAMESPACE}")
        _introspect_PRODX_51933(
            settings.TARGET_CLUSTER, settings.TARGET_NAMESPACE, kaas_manager)
