import pytest
import random

from si_tests import logger
from si_tests import settings
from si_tests.utils import waiters

LOG = logger.logger


@pytest.fixture(scope='module')
def cleanup_hoc_and_machine_labels(kaas_manager, request):
    """Delete HostOSConfiguration object and machine labels before or after test starts"""
    # Run on tear-down (after test execution)
    if request.param != 'run_on_setup':
        yield
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)

    if ns.hostosconfiguration_is_present(name=settings.HOC_CPUSHIELD_TEST_HOC_NAME):
        delete_timeout = 600
        delete_interval = 10

        existing_config = ns.get_hostosconfiguration(name=settings.HOC_CPUSHIELD_TEST_HOC_NAME)
        LOG.info(f"Deleting HostOSConfiguration '{existing_config.name}'")
        existing_config.delete(async_req=True)
        timeout_msg = (f"HostOSConfiguration {settings.HOC_CPUSHIELD_TEST_HOC_NAME} "
                       f"was not deleted in {delete_timeout}s")
        waiters.wait(lambda: not bool(ns.hostosconfiguration_is_present(name=settings.HOC_CPUSHIELD_TEST_HOC_NAME)),
                     timeout=delete_timeout,
                     interval=delete_interval,
                     timeout_msg=timeout_msg)

    machines = cluster.get_machines()
    for machine in machines:
        machine.remove_machine_labels(list(settings.HOC_CPUSHIELD_REBOOT_LABEL.keys()))
    # Run on setup (prior test execution)
    if request.param == 'run_on_setup':
        yield


@pytest.fixture(scope='function')
def create_hoc_before_lcm_and_delete_after(kaas_manager, request):
    """Create HostOSConfiguration object and machine labels before test and delete after"""
    if not settings.CREATE_HOC_BEFORE_LCM_AND_DELETE_AFTER:
        LOG.info("Skipping create_hoc_before_lcm_and_delete_after fixture execution")
        yield
        return
    # Run on setup (prior test execution)
    test_name = request.node.originalname.replace('_', '-')
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)
    hoc_name = f"si-hoc-{test_name}-" + str(random.randint(10, 99))
    machines = cluster.get_machines()
    log_prefix = 'SETUP PHASE (before LCM tests):'
    assert machines, f"{log_prefix} No machines in cluster: {cluster_name}"

    day2_label = {"day2-custom-si-label" + str(random.randint(10, 99)): "enabled"}
    # traceroute package sample
    hostoscfg_config = [
        {
            "module": "package",
            "moduleVersion": settings.HOC_PACKAGE_MODULE_VERSION,
            "values": {"packages": [{"name": "traceroute", "state": "present"}]},
        }
    ]

    if hasattr(request, 'param') and isinstance(request.param, dict):
        if 'configs' in request.param and request.param['configs']:
            hostoscfg_config = request.param['configs']
            LOG.info("{log_prefix} User passed a custom configs for HostOSConfiguration object. "
                     f'Using {hostoscfg_config} values')
        if 'labels' in request.param and request.param['labels']:
            day2_label = request.param['labels']
            LOG.info("{log_prefix} User passed custom matchLabels for HostOSConfiguration object. "
                     f'Using {day2_label} values')

    # hoc sample
    hostoscfg_data = {
        "apiVersion": "kaas.mirantis.com/v1alpha1",
        "kind": "HostOSConfiguration",
        "metadata": {
            "name": hoc_name,
            "namespace": namespace_name,
        },
        "spec": {
            "configs": hostoscfg_config,
            "machineSelector": {
                "matchLabels": day2_label,
            }
        }
    }

    LOG.info(f"{log_prefix} Add label day-2-custom-si to each machine in child cluster: {cluster_name}")
    for machine in machines:
        machine.add_machine_labels(day2_label)

    LOG.info(f"{log_prefix} Create HOC object with module package version: {settings.HOC_PACKAGE_MODULE_VERSION}")
    # Remember LCMMachines timestamps before creating HostOSConfiguration
    lcmmachines_timestamps_before = cluster.get_cluster_lcmmachines_timestamps()
    hostoscfg = ns.create_hostosconfiguration_raw(hostoscfg_data)
    LOG.info(f"{log_prefix} Host config is {hostoscfg}")
    LOG.info(f"{log_prefix} Check that machines from hostosconfiguration status field "
             "have labels used for machineSelector")
    cluster.check.check_hostosconfig_machine_selector(hostoscfg)
    LOG.info(f"{log_prefix} Check that new items added into machineTypes in LCMCluster")
    cluster.check.wait_lcmcluster_day2_machinetypes(hostoscfg)
    LOG.info(f"{log_prefix} Check that new items added into stateItems in LCMMachine")
    cluster.check.wait_lcmmachine_day2_stateitems(hostoscfg, lcmmachines_timestamps_before)
    # Remember LCMMachines timestamps after creating HostOSConfiguration
    lcmmachines_timestamps_after = cluster.get_cluster_lcmmachines_timestamps()
    yield
    # Run on tear-down (after test execution)
    log_prefix = 'TEARDOWN PHASE (after LCM tests):'
    existing_config = ns.get_hostosconfiguration(name=hoc_name)
    # Re-verify machine labels, stateItems
    LOG.info(f"{log_prefix} Check that machines from hostosconfiguration status field "
             "have labels used for machineSelector")
    cluster.check.check_hostosconfig_machine_selector(existing_config)
    LOG.info(f"{log_prefix} Check that new items added into machineTypes in LCMCluster")
    cluster.check.wait_lcmcluster_day2_machinetypes(existing_config)
    LOG.info(f"{log_prefix} Check that new items added into stateItems in LCMMachine "
             "and were not re-executed after initial application")
    cluster.check.wait_lcmmachine_day2_stateitems(existing_config,
                                                  lcmmachines_timestamps_before=lcmmachines_timestamps_before,
                                                  lcmmachines_timestamps_after=lcmmachines_timestamps_after)
    LOG.info(f"{log_prefix} Deleting HostOSConfiguration '{existing_config.name}'")
    existing_config.delete(async_req=True)
    timeout_msg = f"HostOSConfiguration {hoc_name} was not deleted"
    waiters.wait(lambda: not bool(ns.hostosconfiguration_is_present(name=hoc_name)),
                 timeout=1200,
                 interval=10,
                 timeout_msg=timeout_msg)
    machines = cluster.get_machines()
    for machine in machines:
        machine.remove_machine_labels(list(day2_label.keys()))
