import time

from si_tests.utils import waiters
from si_tests.deployments.utils import kubectl_utils
from si_tests.deployments.utils import commons


class Waiter:
    def __init__(self, os_manager, timeout, namespace=None):
        self.timeout = timeout
        self.os_manager = os_manager
        self.namespace = namespace

    def statefulset(self, name, replicas=1, namespace=None):
        namespace = namespace or self.namespace
        self.pods(name, replicas, namespace)

        k8s_object = self.os_manager.api.statefulsets.get(name, namespace)

        def _wait():
            return k8s_object.read().status.ready_replicas

        self.wait(_wait)
        return self

    def pods(self, pod_name, replicas, namespace=None):
        namespace = namespace or self.namespace
        pods_api = self.os_manager.api.pods

        def _wait():
            return len(pods_api.list_starts_with(
                pod_name, namespace)) >= replicas

        self.wait(_wait)
        return self

    def daemonset(self, name, replicas=3, namespace=None):
        namespace = namespace or self.namespace

        self.pods(name, replicas, namespace)

        daemonset = self.os_manager.api.daemonsets.get(name, namespace)

        def _wait():
            return daemonset.read().status.number_ready >= replicas

        self.wait(_wait)
        return self

    def tf_vrouter_daemonsets(self, type, vrouter_nodes=1, namespace=None):
        namespace = namespace or self.namespace
        name_prefix = f"tf-vrouter-{type}"

        self.pods(name_prefix, vrouter_nodes, namespace)
        d_list = self.os_manager.api.daemonsets.list_starts_with(name_prefix)
        for d in d_list:
            def _wait():
                return d.read().status.number_ready == \
                    d.read().status.desired_number_scheduled

            self.wait(_wait)
        return self

    def deployment(self, name, replicas=1, namespace=None):
        namespace = namespace or self.namespace

        self.pods(name, replicas, namespace)

        k8s_object = self.os_manager.api.deployments.get(name, namespace)

        def _wait():
            return k8s_object.read().status.ready_replicas

        self.wait(_wait)
        return self

    def replicasets(self, name, namespace):
        replicasets_api = self.os_manager.api.replicasets

        def _wait():
            rs_list = [rs for rs in replicasets_api.list_all(
                namespace=namespace) if name in rs.name]
            wait_dict = []
            for rs in rs_list:
                ready_replicas = rs.ready_replicas or 0
                if rs.desired_replicas != ready_replicas:
                    wait_dict.append(rs)
            return len(wait_dict) == 0

        self.wait(_wait)
        return self

    def wait_pass(self, func, expected, *args, **kwargs):
        start_time = time.time()
        waiters.wait_pass(func, expected, timeout=self.timeout,
                          predicate_args=args, predicate_kwargs=kwargs)
        self.update_timeout(time.time() - start_time)
        return self

    def wait(self, func, *args, **kwargs):
        start_time = time.time()
        waiters.wait(func, timeout=self.timeout, predicate_args=args,
                     predicate_kwargs=kwargs)
        self.update_timeout(time.time() - start_time)
        return self

    @staticmethod
    def read_checker(k8s_object):
        k8s_object.read()
        return True

    def update_timeout(self, timedelta):
        self.timeout = self.timeout - timedelta

    def wait_for_dns_warmup(self, kubectl: kubectl_utils.Kubectl,
                            keystone_pod_name=None,
                            resolve='it.just.works',
                            resolved_expected_ip=None,
                            ns=None):
        """Run dig for few times, unless it's ok
        Must pass 5 times in a row"""

        def _wait():
            i = 1
            _with = resolved_expected_ip or 'any'
            while 5 >= i:
                commons.LOG.info(f"Check {i}/5 "
                                 f"'{resolve}' to be resolved with '{_with}' ip")
                rez = kubectl.exec(
                    keystone_pod_name,
                    f"bash -c 'timeout 2 dig +short {resolve}|head -n1'",
                    namespace=ns).result
                i += 1
                if resolved_expected_ip:
                    if rez.exit_code == 0 and rez.stdout_str == resolved_expected_ip:
                        continue
                    elif rez.exit_code == 0 and rez.stdout_str != resolved_expected_ip:
                        _m = rez.stdout_str or rez.stderr_str
                        commons.LOG.debug(f"resolved with {_m}")
                        return False
                    else:
                        return False
                elif rez.exit_code == 0 and not resolved_expected_ip:
                    commons.LOG.debug("wait_for_dns_warmup: success")
                    continue
                else:
                    commons.LOG.debug(f"wait_for_dns_warmup: {rez}")
                    return False
            return True

        self.wait(_wait)
        return self
