import os
import typing

from . import commons
from si_tests.utils import utils
from exec_helpers import SSHClient, Subprocess
from retry import retry
from paramiko import ssh_exception
from si_tests import logger

LOG = logger.logger


class _Kubectl():
    def __init__(self,
                 executor: typing.Union[SSHClient, Subprocess],
                 kubectl_path: str,
                 kubeconfig: str,
                 namespace: str = None,
                 verbose: bool = False):
        self.executor = executor
        self.namespace = namespace
        self.result = None
        self.kubeconfig = kubeconfig
        self.__kubectl = kubectl_path
        self.__verbose = verbose

    def _execute(self, command, **kwargs):
        command = "KUBECONFIG={} {} {} ".format(
            self.kubeconfig, self.__kubectl, command)
        if self.__verbose:
            LOG.info(f'Attempt to execute command {command}')
        result = self.executor.execute(command=command, verbose=self.__verbose, **kwargs)
        commons.LOG.debug("%s", result.stdout_str)
        assert result.exit_code == 0, result
        self.result = result
        return self

    def apply(self, apply_file_name="-", stdin=None, options=""):
        command = "apply -f {} ".format(apply_file_name) + options
        return self._execute(command, stdin=stdin)

    def exec(self, pod_name, command, namespace=None):
        namespace = namespace or self.namespace
        command = "exec -n {} {} -- {}".format(namespace, pod_name, command)
        return self._execute(command)

    @retry(AssertionError, tries=5, delay=5, logger=LOG)
    def get(self, resource_type, command, namespace=None, string=False):
        namespace = namespace or self.namespace
        command = "get {} -n {} {}".format(
            resource_type, namespace, command)
        if string:
            return self._execute(command).result_str
        else:
            return self._execute(command)

    @retry(AssertionError, tries=5, delay=5, logger=LOG)
    def get_crds(self):
        return self._execute("get crds").result_str

    def create(self, resource_type, command, namespace=None):
        namespace = namespace or self.namespace
        command = "create {} -n {} {}".format(
            resource_type, namespace, command)
        return self._execute(command)

    def wait(self, command, namespace=None):
        namespace = namespace or self.namespace
        command = "wait -n {} {}".format(namespace, command)
        return self._execute(command)

    @retry(AssertionError, delay=5, tries=5, logger=LOG)
    def cp(self, source, destination, container=""):
        container = "-c {}".format(container) if container else container
        command = "cp {} {} {}".format(source, destination, container)
        return self._execute(command)

    def delete(self, resource_type, command, namespace=None, **kwargs):
        namespace = namespace or self.namespace
        command = "delete {} -n {} {}".format(
            resource_type, namespace, command)
        return self._execute(command, **kwargs)

    @retry(exceptions=(ssh_exception.SSHException, EOFError), tries=5, delay=1.5, jitter=(0, 1.4), logger=logger.logger)
    def logs(self, pod_name: str, namespace: str = None) -> str:
        namespace = namespace or self.namespace
        command = "KUBECONFIG={} {} logs {} -n {} ".format(
            self.kubeconfig, self.__kubectl, pod_name, namespace)
        result = self.executor.execute(command=command, verbose=self.__verbose)
        assert result.exit_code == 0, result
        self.result = result
        return self.result_str

    @retry(AssertionError, tries=5, delay=5, logger=LOG)
    def label(self, resource_type: str, value: str, namespace: str = None,
              object_name: str = "", flags: str = "", **kwargs):
        namespace = namespace or self.namespace
        command = f"label -n {namespace} {resource_type} {object_name} {value} {flags}"
        return self._execute(command, **kwargs)

    @retry(AssertionError, tries=5, delay=5, logger=LOG)
    def patch(self, resource_type: str, object_name: str, patch: str,
              patch_type: str = "merge", namespace: str = None, **kwargs):
        namespace = namespace or self.namespace
        command = f"patch -n {namespace} {resource_type} {object_name} --type={patch_type} -p '{patch}'"
        return self._execute(command, **kwargs)

    @retry(AssertionError, tries=5, delay=5, logger=LOG)
    def get_node_name_by_label(self, label):
        return self.get('nodes',
                        f"-l {label} -ojsonpath='{{.items[*].metadata.name}}'"
                        ).result_str.split()

    @property
    def result_yaml(self):
        return self.result.stdout_yaml

    @property
    def result_str(self):
        return self.result.stdout_str


class Kubectl(_Kubectl):
    def __init__(self, namespace: str = None, kubeconfig: str = None, verbose: bool = False):
        super().__init__(
            executor=commons.get_local_executor(),
            kubectl_path=utils.get_binary_path("kubectl"),
            kubeconfig=os.path.abspath(kubeconfig or os.environ.get("KUBECONFIG")),
            namespace=namespace,
            verbose=verbose)


class RemoteKubectl(_Kubectl):
    def __init__(self, remote: SSHClient, binary_path: str,
                 namespace: str = None, kubeconfig: str = None,
                 verbose: bool = False):
        super().__init__(
            executor=remote,
            kubectl_path=remote.check_call(f"realpath $(command -v {binary_path})").stdout_str,
            kubeconfig=kubeconfig,
            namespace=namespace,
            verbose=verbose
        )
