import functools
import json
from http import client as httpconnection_client

import requests
import yaml

import si_tests.utils.waiters as helpers
from si_tests import logger
from si_tests.clients import http_client
from si_tests.clients.iam import keycloak_client

LOG = logger.logger


def HTTPConnectionLog(*args):
    message = ' '.join(args)
    LOG.debug(message)


def httpconnection_debug(func):
    @functools.wraps(func)
    def debugit(*args, **kwargs):
        # mock the print function in the http.client
        httpconnection_client.print = HTTPConnectionLog
        httpconnection_client.HTTPConnection.debuglevel = 1
        try:
            return func(*args, **kwargs)
        finally:
            httpconnection_client.print = print
            httpconnection_client.HTTPConnection.debuglevel = 0

    return debugit


class MKEDashboardClientPlain(object):
    """MKE client with plain text authentication"""

    def __init__(self, dashboard_url,
                 username="admin", password='topsecret',
                 token=None, verify=False):
        self.token = token
        self.verify = verify
        self.username = username
        self.password = password
        self.httpclient = http_client.HttpClient(base_url=dashboard_url)
        if not self.token:
            self.login()

    def compose_header(self, content_type="application/json"):
        header = {"Content-Type": content_type}
        if self.token:
            header.update({"Authorization": "Bearer {}".format(self.token)})
        return header

    @httpconnection_debug
    def login(self):
        """Authorize with username/password and get new token"""
        self.token = None
        url = "/auth/login"
        body = ('{{"username":"{0}","password":"{1}"}}'
                .format(self.username, self.password))
        res = self.httpclient.request(url=url, method='POST',
                                      headers=self.compose_header(),
                                      body=body,
                                      verify=self.verify)
        decoded_res = json.loads(res.content.decode('utf-8'))
        if "auth_token" not in decoded_res:
            raise ValueError("Authentication failed, UCP response: {0}"
                             .format(decoded_res))
        self.token = decoded_res["auth_token"]
        LOG.debug(f"UCP auth token have been refreshed for user: '{self.username}'")

    @httpconnection_debug
    def _get_resource(self, resource_name, raise_on_error=True,
                      content_type="application/json", deserialize=True):
        url = "/{0}".format(resource_name)
        try:
            res = self.httpclient.request(
                url=url, method='GET',
                headers=self.compose_header(content_type=content_type),
                verify=self.verify,
                raise_on_error=raise_on_error)
            if res.status_code == requests.codes.unauthorized:
                # Try to refresh the token
                self.login()
                res = self.httpclient.request(
                    url=url, method='GET',
                    headers=self.compose_header(),
                    verify=self.verify,
                    raise_on_error=raise_on_error)
        except requests.HTTPError as e:
            if e.response.status_code != requests.codes.unauthorized:
                raise
            LOG.warning(e)
            self.login()
            res = self.httpclient.request(
                url=url, method='GET',
                headers=self.compose_header(),
                verify=self.verify,
                raise_on_error=raise_on_error)
            # Try to refresh token again
        result = res.content.decode("utf-8")
        if deserialize:
            result = json.loads(result)
            LOG.debug(f"Response:\n{yaml.dump(result)}")
        return result

    @httpconnection_debug
    def _post_resource(self, resource_name, body, raise_on_error=True):
        url = "/{0}".format(resource_name)
        try:
            res = self.httpclient.request(
                url=url, method='POST',
                headers=self.compose_header(),
                body=body,
                verify=self.verify,
                raise_on_error=raise_on_error)
            if res.status_code == requests.codes.unauthorized:
                # Try to refresh the token
                self.login()
                res = self.httpclient.request(
                    url=url, method='POST',
                    headers=self.compose_header(),
                    body=body,
                    verify=self.verify,
                    raise_on_error=raise_on_error)
        except requests.HTTPError as e:
            if e.response.status_code != requests.codes.unauthorized:
                raise
            LOG.warning(e)
            self.login()
            res = self.httpclient.request(
                url=url, method='POST',
                headers=self.compose_header(),
                body=body,
                verify=self.verify,
                raise_on_error=raise_on_error)
        result = json.loads(res.content.decode('utf-8'))
        LOG.debug(f"Response:\n{yaml.dump(result)}")
        return result

    @httpconnection_debug
    def _delete_resource(self, resource_name, raise_on_error=True):
        url = "/{0}".format(resource_name)
        try:
            res = self.httpclient.request(
                url=url, method='DELETE',
                headers=self.compose_header(),
                verify=self.verify,
                raise_on_error=raise_on_error)
            if res.status_code == requests.codes.unauthorized:
                # Try to refresh the token
                self.login()
                res = self.httpclient.request(
                    url=url, method='DELETE',
                    headers=self.compose_header(),
                    verify=self.verify,
                    raise_on_error=raise_on_error)
        except requests.HTTPError as e:
            if e.response.status_code != requests.codes.unauthorized:
                raise
            LOG.warning(e)
            self.login()
            res = self.httpclient.request(
                url=url, method='DELETE',
                headers=self.compose_header(),
                verify=self.verify,
                raise_on_error=raise_on_error)
        result = json.loads(res.content.decode('utf-8'))
        LOG.debug(f"Response:\n{yaml.dump(result)}")
        return result

    def get_mke_config(self):
        return self._get_resource(
            "api/ucp/config-toml", content_type="application/toml",
            deserialize=False)

    def create_k8s_namespace(self, name, wait_result=True, raise_on_error=True):
        resource_name = "api/v1/namespaces"
        payload = {
            "apiVersion": "v1",
            "kind": "Namespace",
            "metadata": {
                "name": name
            }
        }
        result = self._post_resource(resource_name=resource_name, body=json.dumps(payload),
                                     raise_on_error=raise_on_error)
        if wait_result:
            LOG.info("Wait for namespace creation")
            helpers.wait(
                lambda: name in self.get_k8s_namespace_names(),
                timeout=60, interval=10, timeout_msg="Resource creation timeout"
            )
        return result

    def create_k8s_pod(self, namespace, body, phases=('Running', 'Succeeded'), wait_result=True, raise_on_error=True):
        resource_name = f"api/v1/namespaces/{namespace}/pods"
        result = self._post_resource(resource_name=resource_name, body=body, raise_on_error=raise_on_error)
        if wait_result:
            LOG.info("Wait for pod creation")
            pod_name = result.get('metadata', {}).get('name')
            helpers.wait(
                lambda: (status := self.get_k8s_pod(name=pod_name, namespace=namespace, raise_on_error=False)
                         .get('status', {})) != "Failure" and status.get('phase') in phases,
                timeout=120, interval=10, timeout_msg="Resource creation timeout"
            )
        return result

    def delete_k8s_namespace(self, name, wait_result=True, raise_on_error=True):
        resource_name = f"api/v1/namespaces/{name}"
        result = self._delete_resource(resource_name=resource_name, raise_on_error=raise_on_error)
        if wait_result:
            LOG.info("Wait for namespace to be removed")
            helpers.wait(
                lambda: name not in self.get_k8s_namespace_names(),
                timeout=60, interval=10, timeout_msg="Resource creation timeout"
            )
        return result

    def delete_k8s_pod(self, name, namespace, wait_result=True, raise_on_error=True):
        resource_name = f"api/v1/namespaces/{namespace}/pods/{name}"
        result = self._delete_resource(resource_name=resource_name, raise_on_error=raise_on_error)
        if wait_result:
            LOG.info("Wait for pod to be removed")
            helpers.wait(
                lambda: name not in self.get_k8s_pods(namespace=namespace),
                timeout=120, interval=10, timeout_msg="Resource creation timeout"
            )
        return result

    def get_nodes(self):
        nodes = self._get_resource("nodes")
        LOG.debug(f"All nodes from mke dashboard  are {nodes}")
        # Try to set the node IP address from "ManagerStatus" if exists
        # https://github.com/moby/moby/issues/35437#issuecomment-504104947
        for node in nodes:
            if "ManagerStatus" in node and "Addr" in node["ManagerStatus"]:
                ad_addr = node["ManagerStatus"]["Addr"]
                if type(ad_addr) is str and "unknown" not in node["Status"]["State"]:
                    addr = ad_addr.split(":")[0]
                    node["Status"]["Addr"] = addr
        return nodes

    def get_node_by_name(self, node_name):
        return self._get_resource(f"nodes/{node_name}")

    def get_ready_nodes(self):
        return [node for node in self.get_nodes() if 'ready' in node['Status']["State"]]

    def get_k8s_namespaces(self, raise_on_error=True):
        return self._get_resource("kubernetesNamespaces", raise_on_error=raise_on_error)["items"]

    def get_k8s_namespace_names(self, raise_on_error=True):
        namespaces = self.get_k8s_namespaces(raise_on_error=raise_on_error)
        names = [ns["metadata"]["name"] for ns in namespaces]
        return names

    def get_k8s_pod(self, name, namespace, raise_on_error=True):
        resource_name = f"api/v1/namespaces/{namespace}/pods/{name}"
        return self._get_resource(resource_name=resource_name, raise_on_error=raise_on_error)

    def get_k8s_pods(self, namespace):
        return self._get_resource(
            f"api/v1/namespaces/{namespace}/pods")["items"]

    def get_k8s_pod_names(self, namespace):
        pods = self.get_k8s_pods(namespace)
        names = [pod["metadata"]["name"] for pod in pods]
        return names

    def get_swarm_tasks(self):
        return self._get_resource("tasks")

    def get_k8s_services(self, namespace):
        return self._get_resource(
            f"api/v1/namespaces/{namespace}/services")["items"]

    def get_k8s_service_names(self, namespace):
        services = self.get_k8s_services(namespace)
        names = [service["metadata"]["name"] for service in services]
        return names

    def get_swarm_services(self):
        return self._get_resource("services?status=true&all=1")

    def get_swarm_service_names(self):
        services = self.get_swarm_services()
        names = [service["Spec"]["Name"] for service in services]
        return names

    def get_swarm_containers(self):
        containers = self._get_resource("containers/json?all=1")
        return containers

    def get_swarm_container_names(self, state="running"):
        containers = self.get_swarm_containers()
        names = [container["Names"] for container in containers
                 if container["State"] == state]
        return names

    def get_swarm_service_replicas(self):
        """Calculates desired and actual service replicas

        Desired replicas: count of tasks with non-"shutdown" state
        Running tasks: count of tasks with "running" state on
                       the active nodes
        Based on: https://github.com/docker/
            cli/blob/master/cli/command/service/list.go#L110

        Returns dict:
        {
            "<service_name>": {
                "DesiredTasks": <int>,
                "RunningTasks": <int>,
            },
            ...
        }
        """
        services = self.get_swarm_services()
        tasks = self.get_swarm_tasks()
        nodes = self.get_nodes()
        active_node = {}
        for node in nodes:
            nstate = node.get("Status", {}).get("State", '')
            # noqa https://godoc.org/github.com/docker/docker/api/types/swarm#NodeState
            active_node[node["ID"]] = (nstate == "ready")

        for service in services:
            # noqa https://godoc.org/github.com/docker/docker/api/types/swarm#ServiceStatus
            if "ServiceStatus" in service:
                if type(service["ServiceStatus"]) is dict:
                    if "RunningTasks" not in service["ServiceStatus"]:
                        service["ServiceStatus"]["RunningTasks"] = 0
                    if "DesiredTasks" not in service["ServiceStatus"]:
                        service["ServiceStatus"]["DesiredTasks"] = 0
                    continue

            service["ServiceStatus"] = {
                "RunningTasks": 0,
                "DesiredTasks": 0,
            }

            mode = service["Spec"].get("Mode", {})
            if "Replicas" in mode.get("Replicated", {}):
                rdesired = mode["Replicated"]["Replicas"]
                service["ServiceStatus"]["DesiredTasks"] = rdesired
            elif "Global" in mode:
                pass

            tasks_count = 0
            desired_count = 0
            running_count = 0
            for task in tasks:
                tserviceid = task.get("ServiceID", '')
                tdesiredstate = task.get("DesiredState", '')
                tstate = task.get("Status", {}).get("State", '')
                tnodeid = task.get("NodeID")
                if tserviceid != service["ID"]:
                    continue
                tasks_count += 1

                # noqa https://godoc.org/github.com/docker/docker/api/types/swarm#TaskState
                if tdesiredstate != "shutdown":
                    desired_count += 1
                if tnodeid and active_node[tnodeid] is True:
                    if tstate == "running":
                        running_count += 1

            if tasks_count > 0:
                service["ServiceStatus"]["DesiredTasks"] = desired_count
                # Overwrites count from "Replicated"
                service["ServiceStatus"]["RunningTasks"] = running_count

        result = {s["Spec"]["Name"]: s["ServiceStatus"] for s in services}

        return result

    def allow_monitoring_agents(self):
        LOG.info("Configure 'Allow all authenticated users, including service accounts to schedule on all nodes,"
                 " including UCP managers.'")
        url = "/collectionGrants/authenticated/swarm/scheduler"
        res = self.httpclient.request(url=url, method='PUT',
                                      headers=self.compose_header(),
                                      verify=self.verify)
        if res.status_code == requests.codes.created:
            LOG.info("Monitoring agents allowed")

    def get_license(self):
        """
        Get MKE license data
        Returns: dict with 'license_config' and 'license_details' data

        """
        return self._get_resource("api/config/license")


class MKEDashboardClientOpenid(MKEDashboardClientPlain):
    """MKE client with OpenID authentication in Keycloak"""

    def __init__(self, dashboard_url, keycloak_ip,
                 username, password,
                 client_id='k8s', token=None, verify=False):
        self.keycloak_ip = keycloak_ip
        self.client_id = client_id  # 'kaas 'for mgmt cluster, 'k8s' for child
        super().__init__(dashboard_url, username=username,
                         password=password, token=token,
                         verify=verify)

    @httpconnection_debug
    def login(self):
        """Authorize with username/password and get new token"""

        self.token = None
        client = keycloak_client.KeycloakUserClient(self.keycloak_ip,
                                                    self.username,
                                                    self.password,
                                                    client_id=self.client_id,
                                                    realm_name="iam")
        result = client.get_openid_token()
        LOG.debug(f"OpenID data for MKE UI:\n{result}")
        self.token = result['id_token']
        LOG.info(f"MKE UI auth token have been refreshed for user: '{self.username}'")
