import json

import requests
import yaml

import si_tests.utils.waiters as helpers
from si_tests import logger
from si_tests.clients import http_client

LOG = logger.logger


class MCCDashboardClient(object):
    token = None

    def __init__(self, dashboard_url,
                 username, password,
                 verify=False, token=None):
        self.username = username
        self.password = password
        self.httpclient = http_client.HttpClient(base_url=dashboard_url)
        self.verify = verify
        self.token = token
        if not self.token:
            self.login()

    def compose_req_header(self):
        return {"Content-Type": "application/json",
                "Accept": "application/json",
                "Referer": "http://172.19.118.240/projects",
                "Host": "172.19.118.240",
                "Authorization": "Bearer {}".format(self.id_token)}

    def get_resource(self, resource_name, decode=True, raise_on_error=True):
        url = "/{0}".format(resource_name)
        try:
            res = self.httpclient.request(
                url=url, method='GET',
                headers=self.compose_req_header(),
                verify=self.verify,
                raise_on_error=raise_on_error)
            if res.status_code == requests.codes.unauthorized:
                # Try to refresh the token
                self.login()
                res = self.httpclient.request(
                    url=url, method='GET',
                    headers=self.compose_req_header(),
                    verify=self.verify,
                    raise_on_error=raise_on_error)
        except requests.HTTPError as e:
            if e.response.status_code != requests.codes.unauthorized:
                raise
            LOG.warning(e)
            # Try to refresh token again
            self.login()
            res = self.httpclient.request(
                url=url, method='GET',
                headers=self.compose_req_header(),
                verify=self.verify,
                raise_on_error=raise_on_error)
        LOG.debug(f"Raw response:\n{res.content}")
        if decode:
            result = json.loads(res.content.decode('utf-8'))
            LOG.debug(f"Response:\n{yaml.dump(result)}")
            return result
        else:
            return res

    def _post_resource(self, resource_name, body, raise_on_error=True):
        url = "/{0}".format(resource_name)
        try:
            res = self.httpclient.request(
                url=url, method='POST',
                headers=self.compose_req_header(),
                body=body,
                verify=self.verify,
                raise_on_error=raise_on_error)
            if res.status_code == requests.codes.unauthorized:
                # Try to refresh the token
                self.login()
                res = self.httpclient.request(
                    url=url, method='POST',
                    headers=self.compose_req_header(),
                    body=body,
                    verify=self.verify,
                    raise_on_error=raise_on_error)
        except requests.HTTPError as e:
            if e.response.status_code != requests.codes.unauthorized:
                raise
            LOG.warning(e)
            # Try to refresh token again
            self.login()
            res = self.httpclient.request(
                url=url, method='POST',
                headers=self.compose_req_header(),
                body=body,
                verify=self.verify,
                raise_on_error=raise_on_error)
        result = json.loads(res.content.decode('utf-8'))
        LOG.debug(f"Response:\n{yaml.dump(result)}")
        return result

    def _delete_resource(self, resource_name, raise_on_error=True):
        url = "/{0}".format(resource_name)
        try:
            res = self.httpclient.request(
                url=url, method='DELETE',
                headers=self.compose_req_header(),
                verify=self.verify,
                raise_on_error=raise_on_error)
            if res.status_code == requests.codes.unauthorized:
                # Try to refresh the token
                self.login()
                res = self.httpclient.request(
                    url=url, method='DELETE',
                    headers=self.compose_req_header(),
                    verify=self.verify,
                    raise_on_error=raise_on_error)
        except requests.HTTPError as e:
            if e.response.status_code != requests.codes.unauthorized:
                raise
            LOG.warning(e)
            # Try to refresh token again
            self.login()
            res = self.httpclient.request(
                url=url, method='DELETE',
                headers=self.compose_req_header(),
                verify=self.verify,
                raise_on_error=raise_on_error)
        result = json.loads(res.content.decode('utf-8'))
        LOG.debug(f"Response:\n{yaml.dump(result)}")
        return result

    def compose_login_header(self):
        header = {"Content-Type": "application/x-www-form-urlencoded"}
        if self.token:
            header.update({"Authorization": "Bearer {}".format(self.token)})
        return header

    def login(self):
        """Authorize with username/password and get new token"""
        self.token = None
        url = "/auth/realms/iam/protocol/openid-connect/token"

        body_data = {
            'username': self.username,
            'password': self.password,
            'client_id': 'kaas',
            'grant_type': 'password',
            'response_type': 'id_token',
            'scope': 'openid',
        }
        body = '&'.join([f"{k}={v}" for k, v in body_data.items()])

        LOG.debug(f"Login body: {body}")

        res = self.httpclient.request(url=url, method='POST',
                                      headers=self.compose_login_header(),
                                      body=body,
                                      verify=self.verify)
        LOG.debug(f"Login response: {res}")
        decoded_res = json.loads(res.content.decode('utf-8'))
        if "access_token" not in decoded_res:
            raise ValueError("Authentication failed, MCC UI response: {0}"
                             .format(decoded_res))
        self.token = decoded_res["access_token"]
        self.id_token = decoded_res["id_token"]
        LOG.info(f"MCC UI auth token have been refreshed for user: '{self.username}'")

    def create_k8s_namespace(self, name, wait_result=True, raise_on_error=True):
        resource_name = "api/v1/namespaces"
        payload = {
            "apiVersion": "v1",
            "kind": "Namespace",
            "metadata": {
                "name": name
            }
        }
        result = self._post_resource(resource_name=resource_name, body=json.dumps(payload),
                                     raise_on_error=raise_on_error)
        if wait_result:
            LOG.info("Wait for namespace creation")
            helpers.wait(
                lambda: name in self.get_k8s_namespace_names(),
                timeout=60, interval=10, timeout_msg="Resource creation timeout"
            )
        return result

    def create_k8s_pod(self, namespace, body, phases=('Running', 'Succeeded'), wait_result=True, raise_on_error=True):
        resource_name = f"api/v1/namespaces/{namespace}/pods"
        result = self._post_resource(resource_name=resource_name, body=body, raise_on_error=raise_on_error)
        if wait_result:
            LOG.info("Wait for pod creation")
            pod_name = result.get('metadata', {}).get('name')
            helpers.wait(
                lambda: (status := self.get_k8s_pod(name=pod_name, namespace=namespace, raise_on_error=False)
                         .get('status', {})) != "Failure" and status.get('phase') in phases,
                timeout=120, interval=10, timeout_msg="Resource creation timeout"
            )
        return result

    def delete_k8s_namespace(self, name, wait_result=True, raise_on_error=True):
        resource_name = f"api/v1/namespaces/{name}"
        result = self._delete_resource(resource_name=resource_name, raise_on_error=raise_on_error)
        if wait_result:
            LOG.info("Wait for namespace to be removed")
            helpers.wait(
                lambda: name not in self.get_k8s_namespace_names(),
                timeout=60, interval=10, timeout_msg="Resource creation timeout"
            )
        return result

    def delete_k8s_pod(self, name, namespace, wait_result=True, raise_on_error=True):
        resource_name = f"api/v1/namespaces/{namespace}/pods/{name}"
        result = self._delete_resource(resource_name=resource_name, raise_on_error=raise_on_error)
        if wait_result:
            LOG.info("Wait for pod to be removed")
            helpers.wait(
                lambda: name not in self.get_k8s_pods(namespace=namespace),
                timeout=120, interval=10, timeout_msg="Resource creation timeout"
            )
        return result

    def get_k8s_pods(self, namespace):
        resource_name = f"api/v1/namespaces/{namespace}/pods/"
        return self.get_resource(resource_name=resource_name)["items"]

    def get_k8s_pod(self, name, namespace, raise_on_error=True):
        resource_name = f"api/v1/namespaces/{namespace}/pods/{name}"
        return self.get_resource(resource_name=resource_name, raise_on_error=raise_on_error)

    def get_k8s_namespaces(self, raise_on_error=True):
        resource_name = "api/v1/namespaces"
        return self.get_resource(resource_name=resource_name, raise_on_error=raise_on_error).get('items', [])

    def get_k8s_namespace_names(self, raise_on_error=True):
        namespaces = self.get_k8s_namespaces(raise_on_error=raise_on_error)
        names = [ns["metadata"]["name"] for ns in namespaces]
        return names
