# Author: https://gist.github.com/FBosler/be10229aba491a8c912e3a1543bbc74e
# Updated to fit current framework by Alex Savatieiev (a.savex@gmail.com)
from functools import wraps
import time

from cfg_checker.common import logger, logger_cli
from cfg_checker.common.exception import KubeException


def retry(exceptions, total_tries=5, initial_wait=1, backoff_factor=2):
    """
    calling the decorated function applying an exponential backoff.
    Args:
        exceptions: Exception(s) that trigger a retry, can be a tuple
        total_tries: Total tries
        initial_wait: Time to first retry
        backoff_factor: Backoff multiplier (e.g. value of 2 will double
                        the delay each retry).
        logger: logger to be used, if none specified print
    """
    def retry_decorator(f):
        @wraps(f)
        def func_with_retries(*args, **kwargs):
            _tries, _delay = total_tries + 1, initial_wait
            while _tries > 1:
                try:
                    return f(*args, **kwargs)
                except exceptions as e:
                    _tries -= 1
                    print_args = args if args else "no args"
                    if _tries == 1:
                        msg = "... {} failed after {} tries".format(
                                # dirty hack to get name
                                str(f).split(" ")[1],
                                total_tries
                            )
                        logger_cli.info(msg)
                        logger.debug(
                            msg + "args: {}, kwargs: {}".format(
                                print_args,
                                kwargs
                            )
                        )
                        _tries = 0
                        raise KubeException(msg)
                    msg = "... {}; Exception: {}.\n" \
                          "... retrying in {} seconds!".format(
                              str(f).split(" ")[1],
                              e,
                              _delay
                          )
                    logger_cli.info(msg)
                    logger.debug(
                        msg + "args: {}, kwargs: {}\n".format(
                            print_args,
                            kwargs
                        )
                    )
                    time.sleep(_delay)
                    _delay *= backoff_factor

        return func_with_retries
    return retry_decorator
