#    Author: Alex Savatieiev (osavatieiev@mirantis.com; a.savex@gmail.com)
#    Copyright 2019-2022 Mirantis, Inc.
import os
import sys
from logging import DEBUG, INFO

from cfg_checker.cli.arguments import add_global_arguments
from cfg_checker.cli.command import execute_command, helps, parsers, \
    parsers_inits
from cfg_checker.common.settings import CheckerConfiguration
from cfg_checker.common import logger, logger_cli
from cfg_checker.helpers.args_utils import MyParser

pkg_dir = os.path.dirname(__file__)
pkg_dir = os.path.normpath(pkg_dir)


def config_check_entrypoint():
    """
    Main entry point. Uses nested parsers structure
    with a default function to execute the comand

    :return: - no return value
    """
    # Main entrypoint
    parser = MyParser(prog="# Mirantis Cloud configuration checker")
    add_global_arguments(parser)
    subparsers = parser.add_subparsers(dest='command')

    # create parsers
    for _command in helps.keys():
        _parser = subparsers.add_parser(
            _command,
            help=helps[_command]
        )
        parsers[_command] = parsers_inits[_command](_parser)

    # parse arguments
    try:
        args, unknown = parser.parse_known_args()
    except TypeError:
        logger_cli.info("\n# Please, check arguments")
        sys.exit(1)

    if unknown:
        logger_cli.error(
            "# Unexpected arguments: {}".format(
                ", ".join(["'{}'".format(a) for a in unknown])
            )
        )
        sys.exit(1)

    if not args.command:
        parser.print_help()
        logger_cli.info("\n# Please, type a command listed above")
        sys.exit(1)

    # Handle debug option before config init
    if args.debug:
        logger_cli.setLevel(DEBUG)
    else:
        logger_cli.setLevel(INFO)

    # Init the config and pass externally configured values
    config = CheckerConfiguration(args)

    # Execute the command
    result = execute_command(args, args.command, config)
    logger.debug(result)
    sys.exit(result)


if __name__ == '__main__':
    config_check_entrypoint()
