#    Copyright 2019 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

import base64
import enum
import exec_helpers
import ipaddress
import json
import os
import paramiko
import pytz
import random
import re
import requests
import secrets
import shutil
import string
import time
import traceback
import yaml
from contextlib import contextmanager
from datetime import datetime, timezone, timedelta
from functools import wraps
from io import StringIO
from kubernetes.client.rest import ApiException
from operator import itemgetter
from tabulate import tabulate

from si_tests.clients.k8s.base import K8sNamespacedResource
from si_tests.managers.si_config_manager import SIConfigManager
from si_tests.utils import packaging_version as version
from requests import exceptions as rexc
from retry import retry
from threading import Thread, Event

from si_tests import logger
from si_tests.utils import templates as templates_utils
from si_tests.settings import ARTIFACTS_DIR, ENV_NAME
from si_tests import settings

LOG = logger.logger

# kaas node-scope labeling
# Most labels hardcoded in kaas/core/frontend/src/consts.js or around
MACHINE_TYPES = {
    'cluster.sigs.k8s.io/control-plane': 'control',
    'hostlabel.bm.kaas.mirantis.com/controlplane': 'control',
    'hostlabel.bm.kaas.mirantis.com/worker': 'worker',
    'hostlabel.bm.kaas.mirantis.com/storage': 'storage',
}

EQUINIX_METRO_IN_REGIONS = {
    'AMER': ['ch', 'sp', 'sv', 'la', 'tr', 'dc', 'da', 'ny'],
    'EMEA': ['am', 'md', 'ld', 'fr', 'pa'],
    'APAC': ['ty', 'sg', 'sy', 'sl', 'hk'],
}


class LogTime(object):
    def __init__(self, name, file_path):
        self.name = name
        self.file_path = file_path
        self.start_time = 0

    def __enter__(self):
        self.start_time = time.time()
        return self

    def __exit__(self, x, y, z):
        end_time = time.time()
        total_time = end_time - self.start_time
        days = int(total_time // 86400)
        hours = int(total_time // 3600 % 24)
        minutes = int(total_time // 60 % 60)
        seconds = round(total_time % 60, 2)
        human_format = "{days}{hours}{minutes}{seconds}s".format(
            days=str(days) + "d " if days > 0 else "",
            hours=str(hours) + "h " if hours > 0 else "",
            minutes=str(minutes) + "m " if minutes > 0 else "",
            seconds=seconds)
        yaml_data = {
            "environment": ENV_NAME,
            "raw_duration": total_time,  # raw seconds
            "duration": human_format  # human readable format
        }
        with templates_utils.YamlEditor(
                file_path=self.file_path) as editor:
            current_content = editor.content
            current_content[self.name] = yaml_data
            editor.content = current_content


class Tail(Thread):
    """
    Tail thread allows to watch remote file continuously in parallel
    with running main thread. Typical use case is watching remote log
    file (from bootstrap.sh) while main thread runs ansible script.

    Tail will read all available lines from the file during each _read()
    iteration and will try to read everything else after stop() is called
    to ensure the entire file was read.

    NOTE: It is not possible to interrupt the thread while it is in _read().
          Entire file will be read to its end if no exception occurs.
    """

    def __init__(self, filename, *args, **kwargs):
        self.filename = filename
        self.prefix = kwargs.pop('prefix', '')
        self.remote = kwargs.pop('remote', None)
        self.logger = kwargs.pop('logger', LOG)
        self.interval = float(kwargs.pop('interval', 10))
        self._stop_event = Event()
        self._position = 0
        super(Tail, self).__init__(*args, **kwargs)

    def __enter__(self):
        self.start()

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.stop()
        self.join()

    def __str__(self):
        return "Tail (filename={}, interval={}, remote={})" \
            .format(self.filename, self.interval, self.remote is not None)

    def _open(self):
        if self.remote is None:
            return open(self.filename)
        else:
            return self.remote.open(self.filename)

    def _read(self):
        file_handler = None
        try:
            file_handler = self._open()
            file_handler.seek(self._position)

            for line in file_handler:
                self.logger.info(self.format(line.strip()))

            self._position = file_handler.tell()
        except OSError:
            self.logger.warning(
                self.format("Not found {}".format(self.filename)))
        except:  # noqa
            self.logger.error(self.format("Got Exception:",
                                          traceback.format_exc()))
            self.stop()
        finally:
            if file_handler:
                file_handler.close()

    def run(self):
        self.logger.info(self.format("Started {}".format(str(self))))
        try:
            while not self._stop_event.is_set():
                self._read()
                self._stop_event.wait(self.interval)
            self._read()
        finally:
            self.logger.info(self.format("Stopped {}".format(str(self))))

    def stop(self):
        self.logger.info(self.format("Stopping {} ...".format(str(self))))
        self._stop_event.set()

    def format(self, *args):
        return '{} {}'.format(self.prefix, '\n'.join(args))


class Provider(enum.Enum):
    aws = ("aws", "AWSClusterProviderSpec", "v1alpha1", "AWSMachineProviderSpec")
    azure = ("azure", "AzureClusterProviderSpec", "v1alpha1", "AzureMachineProviderSpec")
    baremetal = ("baremetal", "BaremetalClusterProviderSpec", "v1alpha1", "BareMetalMachineProviderSpec")
    equinixmetal = ("equinixmetal", "EquinixMetalClusterProviderSpec", "v1alpha1", "EquinixMetalMachineProviderSpec")
    equinixmetalv2 = ("equinixmetalv2", "EquinixMetalClusterProviderSpec", "v1alpha2",
                      "EquinixMetalMachineProviderSpec")
    openstack = ("openstack", "OpenstackClusterProviderSpec", "v1alpha1", "OpenstackMachineProviderSpec")
    vsphere = ("vsphere", "VsphereClusterProviderSpec", "v1alpha1", "VsphereMachineProviderSpec")
    byo = ("byo", "BYOClusterProviderSpec", "v1alpha1", "BYOMachineProviderSpec")

    def __init__(self, provider_name, cluster_spec, api_version, machine_spec):
        self.provider_name = provider_name
        self.cluster_spec = cluster_spec
        self.api_version = api_version
        self.machine_spec = machine_spec

    @classmethod
    def get_all_cluster_spec(cls):
        """
        Get all Cluster Provider Spec

        Returns: List of Cluster Provider Spec

        """
        return [e.cluster_spec for e in cls]

    @classmethod
    def get_all_machine_spec(cls):
        """
        Get all Machine Provider Spec

        Returns: List of Machine Provider Spec

        """
        return [e.machine_spec for e in cls]

    @classmethod
    def with_ceph(cls):
        """
        Get Providers with Ceph

        Returns: List of Provider object

        """
        return [cls.baremetal, cls.equinixmetal, cls.equinixmetalv2]

    @classmethod
    def get_provider_by_name(cls, provider_name):
        """
        Get Provider object by Provider name
        Args:
            provider_name: provider name

        Returns: Provider object

        """
        return next((e for e in cls if e.provider_name == provider_name), None)

    @classmethod
    def get_provider_by_cluster(cls, cluster_spec, api_version='v1alpha1'):
        """
        Get Provider object by Cluster Spec

        Args:
            cluster_spec: Cluster Spec
            api_version:  API version

        Returns: Provider object

        """
        return next((e for e in cls if e.cluster_spec == cluster_spec
                     and e.api_version == api_version), None)

    @classmethod
    def get_provider_by_machine(cls, machine_spec, api_version='v1alpha1'):
        """
        Get Provider object by Machine Spec

        Args:
            machine_spec: Machine Spec
            api_version: API version

        Returns:

        """
        return next((e for e in cls if e.machine_spec == machine_spec
                     and e.api_version == api_version), None)

    def __str__(self):
        return self.provider_name


class NodeLabel(enum.Enum):
    os_control_plane = 'openstack-control-plane=enabled'
    os_compute = 'openstack-compute-node=enabled'
    os_compute_dpdk = 'openstack-compute-node-dpdk=enabled'
    os_gateway = "openstack-gateway=enabled"
    ceph_role_mgr = 'ceph_role_mgr=true'
    ceph_role_mon = 'ceph_role_mon=true'
    ceph_osd = 'ceph_role_osd=true'
    kaas_master = 'node-role.kubernetes.io/master='
    kaas_worker = 'worker=worker'
    kaas_storage = 'storage=storage'
    kaas_stacklight = 'stacklight=enabled'
    kaas_keepalive = 'keepalive'
    tf_analytics = "tfanalytics=enabled"
    tf_control = "tfcontrol=enabled"

    @classmethod
    def get_by_label(cls, label):
        """
        Get item by label value
        Args:
            label: label name

        Returns: NodeLabel object

        """
        return next((e for e in cls if e.value == label), None)

    @classmethod
    def get_by_name(cls, label):
        """
        Get item by label name
        Args:
            label: label name

        Returns: NodeLabel object

        """
        return next((e for e in cls if e.name == label), None)


@contextmanager
def pushd(path):
    current_dir = os.getcwd()
    try:
        os.chdir(os.path.expanduser(path))
        yield
    finally:
        os.chdir(current_dir)


def reduce_occurrences(items, text):
    """ Return string without items(substrings)
        Args:
            items: iterable of strings
            test: string
        Returns:
            string
        Raise:
            AssertionError if any substing not present in source text
    """
    for item in items:
        LOG.debug(
            "Verifying string {} is shown in "
            "\"\"\"\n{}\n\"\"\"".format(item, text))
        assert text.count(item) != 0
        text = text.replace(item, "", 1)
    return text


def generate_keys():
    file_obj = StringIO()
    key = paramiko.RSAKey.generate(1024)
    key.write_private_key(file_obj)
    public = key.get_base64()
    private = file_obj.getvalue()
    file_obj.close()
    return {'private': private,
            'public': public}


def load_keyfile(file_path):
    with open(file_path, 'r') as private_key_file:
        private = private_key_file.read()
    key = paramiko.RSAKey(file_obj=StringIO(private))
    public = key.get_base64()
    return {'private': private,
            'public': public}


def get_rsa_key(private_key):
    f = StringIO(private_key)
    return paramiko.rsakey.RSAKey.from_private_key(f)


def dump_keyfile(file_path, key):
    key = paramiko.RSAKey(file_obj=StringIO(key['private']))
    key.write_private_key_file(file_path)
    os.chmod(file_path, 0o644)


def clean_dir(dirpath):
    shutil.rmtree(dirpath)


def backup_file(filepath, postfix=None, remote=None):
    if not postfix:
        postfix = time.strftime("_%Y%m%d_%H%M%S") + ".bkp"
    new_file = filepath + postfix

    ops_area = "[local]"
    if remote:
        ops_area = "[remote]"
        remote.check_call(f"cp {filepath} {new_file}")
    else:
        shutil.copyfile(filepath, new_file)

    LOG.info(f"{ops_area}: '{filepath}' backup was saved as '{new_file}'")


@retry(exec_helpers.CalledProcessError, delay=5, tries=5, jitter=1, logger=LOG)
def internal_check_docker_call(remote):
    _msg = 'Perform docker connectivity selfcheck for internal resources'
    if not settings.KAAS_BM_CI_ON_EQUINIX:
        LOG.info(_msg)
        remote.check_call(
            'docker run --rm '
            'mirantiseng.docker.mirantis.net/library/alpine:3.12.0 '
            'sh -c "wget {} -O -"'.format(settings.OS_AUTH_URL),
            verbose=True,
            error_info=f'Unable to perform access check to IC:{settings.OS_AUTH_URL}')
    else:
        LOG.warning(f"Skipping: {_msg}")


def check_test_result(request, test_results):
    """Function to check whether test has expected result

    :param mark: pytest request object
    :param test_results: expected test results list
    :rtype: boolean
    """
    for test_result in test_results:
        if hasattr(request.node, 'rep_call') and getattr(request.node.rep_call, test_result):
            LOG.debug(f"Test result is {test_result}")
            return True
    return False


def extract_name_from_mark(mark, info='name'):
    """Simple function to extract name from pytest mark

    :param mark: pytest.mark.MarkInfo
    :param info: Kwarg with information
    :rtype: string or None
    """
    if mark:
        if len(mark.args) > 0:
            return mark.args[0]
        elif info in mark.kwargs:
            return mark.kwargs[info]
    return None


def get_top_fixtures_marks(request, mark_name):
    """Order marks according to fixtures order

    When a test use fixtures that depend on each other in some order,
    that fixtures can have the same pytest mark.

    This method extracts such marks from fixtures that are used in the
    current test and return the content of the marks ordered by the
    fixture dependences.
    If the test case have the same mark, than the content of this mark
    will be the first element in the resulting list.

    :param request: pytest 'request' fixture
    :param mark_name: name of the mark to search on the fixtures and the test

    :rtype list: marks content, from last to first executed.
    """

    fixtureinfo = request.session._fixturemanager.getfixtureinfo(
        request.node, request.function, request.cls)

    top_fixtures_names = []
    for _ in enumerate(fixtureinfo.name2fixturedefs):
        parent_fixtures = set()
        child_fixtures = set()
        for name in sorted(fixtureinfo.name2fixturedefs):
            if name in top_fixtures_names:
                continue
            parent_fixtures.add(name)
            child_fixtures.update(
                fixtureinfo.name2fixturedefs[name][0].argnames)
        top_fixtures_names.extend(list(parent_fixtures - child_fixtures))

    top_fixtures_marks = []

    if mark_name in request.function.func_dict:
        # The top priority is the 'revert_snapshot' mark on the test
        top_fixtures_marks.append(
            extract_name_from_mark(
                request.function.func_dict[mark_name]))

    for top_fixtures_name in top_fixtures_names:
        fd = fixtureinfo.name2fixturedefs[top_fixtures_name][0]
        if mark_name in fd.func.func_dict:
            fixture_mark = extract_name_from_mark(
                fd.func.func_dict[mark_name])
            # Append the snapshot names in the order that fixtures are called
            # starting from the last called fixture to the first one
            top_fixtures_marks.append(fixture_mark)

    LOG.debug("Fixtures ordered from last to first called: {0}"
              .format(top_fixtures_names))
    LOG.debug("Marks ordered from most to least preffered: {0}"
              .format(top_fixtures_marks))

    return top_fixtures_marks


def gen_random_string(size):
    """Generate a random string of fixed length """
    letters = string.ascii_lowercase
    return ''.join(random.choice(letters) for i in range(size))


def gen_random_password(size):
    """Generate a random password of fixed length """
    return secrets.token_urlsafe(size)


def convert_to_bytes(size_str):
    """
    Units are case-insensitive.
    :param size_str: can be 1KB, 2mb, 3.2gB, 4.5TB. If 100, 100B format
                     is used, then assume value already in bytes
    :return: int value in bytes
    """
    if not size_str:
        return 0
    units = {'TB': 2 ** 40, 'GB': 2 ** 30, 'MB': 2 ** 20, 'KB': 2 ** 10}
    size_str = size_str.replace(" ", "")
    last_index = [x.end() for x in re.finditer(r'\d', size_str)][-1]
    value = float(size_str[:last_index])
    if value < 0:
        raise ValueError("Wrong size. Should be greater then 0")
    unit = size_str[last_index:].upper()
    if not unit or unit == 'B':
        return int(value)
    return int(value * units[unit])


def print_pods_status(pods):
    """Print pods status in table format

    :param pods: list of K8sPod objects which produced
                 by kubectl_client.pods.list_all()
                 or kubectl_client.pods.list(namespace=...)
    """
    rows = [['NAMESPACE', 'NAME', 'READY', 'STATUS',
             'RESTARTS', 'AGE', 'IP', 'NODE', 'REASON']]
    for pod in pods:
        restarts = 0
        container_statuses = pod['status']['container_statuses'] or []
        total_containers = len(container_statuses)
        runnint_containers = sum([
            1 for x in container_statuses if x['ready'] is True
        ])
        restarts = sum([
            x['restart_count'] for x in container_statuses
        ])
        ready = "{0}/{1}".format(runnint_containers, total_containers)

        age = ''
        if pod['status']['start_time']:
            age_delta = datetime.now(timezone.utc) - \
                        pod['status']['start_time']
            m, s = divmod(age_delta.seconds, 60)
            h, m = divmod(m, 60)
            days = "{0}d ".format(age_delta.days) if age_delta.days else ''
            hours = "{0}h ".format(h) if h else ''
            minutes = "{0}m ".format(m) if (m and not days) else ''
            seconds = "{0}s".format(s) if (not days and not hours) else ''
            age = "{days}{hours}{minutes}{seconds}".format(
                days=days,
                hours=hours,
                minutes=minutes,
                seconds=seconds
            )

        rows.append([
            pod['metadata']['namespace'],
            pod['metadata']['name'],
            ready,
            pod['status']['phase'],
            str(restarts),
            str(age),
            pod['status']['pod_ip'] or '',
            pod['spec']['node_name'],
            pod['status']['reason'] or ''
        ])

    LOG.debug("Raw rows: {0}".format(rows))
    cols = zip(*rows)

    col_widths = [max(len(value or 'None') for value in col) for col in cols]

    format_str = '  '.join(['{{{0}!s:<{1}}}'.format(n, width)
                            for n, width in enumerate(col_widths)])

    LOG.debug("Format string for rows: {0}".format(format_str))
    LOG.info('\n' + '\n'.join([format_str.format(*row) for row in rows]))
    return rows


def make_export_env_strting(envs):
    envs_string = '; '.join(["export {}='{}'".format(k, envs[k]) for k in envs])
    return envs_string


def merge_dicts(src, dst, path=None):
    path = path or []
    for key in dst:
        if key in src:
            if isinstance(src[key], dict) and isinstance(dst[key], dict):
                merge_dicts(src[key], dst[key], path + [str(key)])
            elif src[key] == dst[key]:
                pass  # same leaf value
            else:
                src[key] = dst[key]
        else:
            src[key] = dst[key]
    return src


def log_method_time():
    def log(func):
        @wraps(func)
        def wrapped(*args, **kwargs):
            with LogTime(func.__name__,
                         file_path=ARTIFACTS_DIR + 'time_spent.yaml'):
                result = func(*args, **kwargs)
                return result

        return wrapped

    return log


def get_expected_pods(path, target_nss=[]):
    LOG.info("Fetching lists of expected pods")
    template = templates_utils.render_template(path)
    expected_pods_file = yaml.load(template, Loader=yaml.SafeLoader)['ucp']
    expected_pods = {}
    for ns, pod_dict in expected_pods_file.items():
        if target_nss and ns not in target_nss:
            continue
        # we are in trouble if we have identical pod names
        # in different ns
        expected_pods.update(pod_dict)
    return expected_pods


def generate_list_pods(kaas_manager, target_nss=[]):
    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    cluster = ns.get_cluster(settings.TARGET_CLUSTER)
    if not cluster.is_existed():
        return {}
    LOG.info("Make sure we have all needed pods")
    cluster.check.check_actual_expected_pods()
    LOG.info("Generating list of pods for {0} cluster "
             "in {1} namespace".format(cluster.name, ns.name))
    kubectl_client = cluster.k8sclient
    ep = cluster.expected_pods
    expected_pods = {}
    for ns in ep:
        if not target_nss or ns in target_nss:
            pod_names = [x.split("/")[0] for x in ep[ns].keys()]
            for pod_name in sorted(pod_names, key=lambda x: len(x),
                                   reverse=True):
                pod_num = ep[ns][pod_name] if pod_name in ep[ns].keys() else ep[ns][f"{pod_name}/no_owner"]
                # order is guaranteed
                # https://stackoverflow.com/questions/39980323/are-dictionaries-ordered-in-python-3-6
                # https://docs.python.org/3.7/library/stdtypes.html#typesmapping
                expected_pods.update({pod_name: pod_num})
    actual_pods = []
    if target_nss:
        for ns in target_nss:
            actual_pods += kubectl_client.pods.list(
                namespace=ns,
                field_selector="status.phase=Running"
            )
    else:
        actual_pods = kubectl_client.pods.list_all(
            field_selector="status.phase=Running"
        )
    result = {}

    for pod_name, pod_num in expected_pods.items():
        pods = [x for x in actual_pods if x.name.startswith(pod_name)]
        # order is guaranteed
        result[pod_name] = (pod_name, pod_num, pods)
        if not target_nss:
            actual_pods = [x for x in actual_pods
                           if not x.name.startswith(pod_name)]
        else:
            actual_pods = [x for x in actual_pods if x.namespace in target_nss
                           and not x.name.startswith(pod_name)]

    LOG.debug("Generated list: {}".format(result))
    return result


@retry((rexc.Timeout, rexc.ConnectTimeout, rexc.ReadTimeout, rexc.ConnectionError), delay=15, tries=5, logger=LOG)
def get_latest_k8s_image_version(url, major_version):
    LOG.info("Fetching list of available k8s conformance "
             "images from {}".format(url))
    resp = requests.get(url, verify=False)
    if not resp.ok:
        raise rexc.ConnectionError("Request failed {}".format(resp.text))
    lst = []
    for txt in resp.text.split("\n"):
        if major_version in txt and 'att' not in txt:
            lst.append(
                txt.split('<a href="v')[1].split('/">v')[0])
    LOG.info("Available versions: {0}. "
             "Filtered by {1}".format(lst, major_version))
    return str(max([version.parse(x) for x in lst])).replace(".post", "-")


def get_labels_by_type(machine_type):
    """
    labels only in skope of MACHINE_TYPES
    terms kaas
    return: formatted dict with items
    """
    labels = dict()
    for label, m_type in MACHINE_TYPES.items():
        if m_type == machine_type:
            labels[label] = m_type
    return labels


def get_type_by_labels(labels):
    """
    labels only in scope of MACHINE_TYPES
    terms kaas.
    Types only for scope k8s node.(worker/control)
    """
    # Set default type to "worker" because worker nodes
    # don't have the related label in the openstack provider
    _machine_type = 'worker'
    # For bm case we may have multiple types for machine
    multiple_types = get_types_by_labels(labels)
    if 'worker' in multiple_types and 'storage' in multiple_types:
        _machine_type = 'worker'
    elif multiple_types:
        _machine_type = multiple_types[0]
    else:
        LOG.warning(f"Using default machine type:{_machine_type}")
    return _machine_type


def get_types_by_labels(labels):
    """
    labels and types in scope of MACHINE_TYPES
    terms kaas.
    return: list MACHINE_TYPES.values() item/items
    """
    # For bm case we may have multiple types for machine
    multiple_types = []
    for label, machine_type in MACHINE_TYPES.items():
        # label value might be in ['machinetype', 'true', str()]
        if label in labels and (any([labels[label] == machine_type,
                                     labels[label] == 'true',
                                     len(str(labels[label])) >= 1])):
            multiple_types.append(MACHINE_TYPES[label])
    if not multiple_types:
        # Workaround for missing labels for node role on worker nodes
        multiple_types = ['worker']
        LOG.warning(f"Looks like passed 'machine' dont have "
                    f"expected 'labels' for kaas. "
                    f"Using by default: {multiple_types}")
    return sorted(set(multiple_types))


def render_bmh_name(name, cname, uefi, labels, si_roles=False):
    """
    just to re-use one predictable name across all tests
    """
    if not si_roles:
        si_roles = []
    return '-'.join([name,
                     cname,
                     *get_types_by_labels(labels),
                     'efi' if uefi else 'noefi',
                     *si_roles,
                     ])


def get_available_equinix_metro(api_token,
                                machine_type,
                                machines_count,
                                metro_scoped='',
                                region_scoped=''):
    """Get random equinix metro with available resources for machines

    :param region_scoped: select only the metros from the specified Equinix
    region,
                          one of: AMER, EMEA, APAC
    """

    metros = [metro_scoped] if metro_scoped else \
        [metro["code"] for metro in discover_equinix_metros(api_token)]
    LOG.info(f"Found metros <metros>: {metros}")
    if region_scoped:
        metros = filter_equinix_metros_by_region(metros,
                                                 region_scoped)
        LOG.info(f"Region {region_scoped} scoped <metros>: {metros}")
    # simple load balancing
    random.shuffle(metros)

    LOG.info("Fetching available Equinix Metal metro "
             "to create {} servers".format(machines_count))
    desired_capacity = {
        'servers': [
            {
                'plan': machine_type,
                'quantity': machines_count
            }
        ]
    }

    for metro in metros:
        try:
            equinix_region = get_equinix_region_by_metro(metro)
            if equinix_region is None:
                LOG.warning(f"Equinix region for metro {metro} is undefined "
                            f"in get_equinix_region_by_metro(), skipping this metro")
                continue
            if equinix_metro_capable(api_token, metro, desired_capacity):
                LOG.info("{} metro suitable for {} servers"
                         .format(metro, machines_count))
                return metro
            else:
                LOG.info("{} metro not suitable for {} servers"
                         .format(metro, machines_count))
        except Exception as err:
            LOG.warning("{} metro unavailable: {}, skipping"
                        .format(metro, err))

    raise Exception("Failed to get available Equinix metro for {}"
                    " instances".format(machines_count))


def equinixmetalv2_get_network_config(cluster_namespace,
                                      cluster_name,
                                      mcc_network_config,
                                      machines_amount):
    cluster_network_config = mcc_network_config[f"{cluster_namespace}/{cluster_name}"]

    # find suitable metro for cluster chained with predefined equinixmetalv2 infra setup
    if settings.EQUINIX_SKIP_FACILITY_SELECTION:
        metro = settings.EQUINIX_FACILITY
        LOG.info(f"Metro selection will be skipped, "
                 f"metro {metro} will be used for management deploy")
    else:
        if cluster_network_config['metro']:
            metro = cluster_network_config['metro']
        else:
            metro = get_available_equinix_metro(
                settings.KAAS_EQUINIX_USER_API_TOKEN,
                settings.KAAS_EQUINIX_MACHINE_TYPE,
                machines_amount)

    return {"network_config": cluster_network_config,
            "metro": metro}


@retry(Exception, delay=3, tries=3, logger=LOG)
def equinix_metro_capable(api_token, metro_code, servers_meta):
    LOG.info("Checking hardware availability in metro {}"
             .format(metro_code))
    servers_meta['servers'][0]['metro'] = metro_code
    headers = {'Accept': 'application/json', 'X-Auth-Token': api_token,
               'Content-Type': 'application/json'}

    resp = requests.post('https://api.equinix.com/metal/v1/capacity/metros',
                         headers=headers, json=servers_meta)
    if not resp.ok:
        raise Exception("Request failed: {}".format(resp.text))

    result = resp.json()['servers'][0]
    return result['available']


@retry(Exception, delay=3, tries=3, logger=LOG)
def discover_equinix_metros_and_facilities(api_token):
    LOG.info("Gathering information about metros and matching them facilities")

    headers = {'Accept': 'application/json', 'X-Auth-Token': api_token,
               'Content-Type': 'application/json'}

    resp = requests.get('https://api.equinix.com/metal/v1/facilities', headers=headers)

    if not resp.ok:
        raise Exception("Request failed: {}".format(resp.text))

    LOG.debug(f"Equinix facilities:\n{yaml.dump(resp.json())}")
    facilities = resp.json()['facilities']
    metro_scoped_facilities = {}
    for facility in facilities:
        if not metro_scoped_facilities.get(facility['metro']['code']):
            metro_scoped_facilities[facility['metro']['code']] = []
        metro_scoped_facilities[facility['metro']['code']].append(facility['code'])

    LOG.info("Facilities/metros information gathered. Result: {}".format(metro_scoped_facilities))

    return metro_scoped_facilities


@retry(Exception, delay=3, tries=3, logger=LOG)
def discover_equinix_metros(api_token):
    LOG.info("Gathering information about metros")

    headers = {'Accept': 'application/json', 'X-Auth-Token': api_token,
               'Content-Type': 'application/json'}

    resp = requests.get('https://api.equinix.com/metal/v1/locations/metros', headers=headers)

    if not resp.ok:
        raise Exception("Request failed: {}".format(resp.text))

    LOG.debug(f"Equinix metros:\n{yaml.dump(resp.json())}")
    return resp.json()['metros']


def filter_equinix_metros_by_region(metros, region_scoped):
    """Filter the list <metros> and leave only the metros from <region_scoped> equinix region"""
    new_metros = [
        metro for metro in metros
        if get_equinix_region_by_metro(metro) == region_scoped]
    return new_metros


def get_equinix_region_by_metro(metro):
    """Return Equinix region for the provided metro
    return: str, one of 'AMER', 'EMEA', 'APAC', or None
    """
    equinix_region = [r for r, m in EQUINIX_METRO_IN_REGIONS.items() if metro in m]
    if not equinix_region:
        LOG.warning(f"Not found equinix region for metro {metro}, please update <metros_in_regions> dict")
        return None
    return equinix_region[0]


def get_equinix_metro_by_facility(facility):
    for metros in EQUINIX_METRO_IN_REGIONS.values():
        for metro in metros:
            if facility.startswith(metro.lower()):
                return metro


def vsphere_set_vm_template_list():
    vm_templates = []
    if settings.KAAS_VSPHERE_RHEL8_TEMPLATE_PATH:
        vm_templates.append(settings.KAAS_VSPHERE_RHEL8_TEMPLATE_PATH)
    if settings.KAAS_VSPHERE_UBUNTU_TEMPLATE_PATH:
        vm_templates.append(settings.KAAS_VSPHERE_UBUNTU_TEMPLATE_PATH)
    return vm_templates


def vsphere_set_rhel_license(template, rhel_license_name):
    if template == settings.KAAS_VSPHERE_UBUNTU_TEMPLATE_PATH:
        return ""
    else:
        return rhel_license_name


def get_docker_version(machine):
    cmd = "docker version --format '{{ json . }}'"
    try:
        result = machine.exec_pod_cmd(cmd, get_events=False, verbose=False)
        data = yaml.safe_load(result["logs"])
        versions = [f"Server Version: {data['Server']['Version']}"]
        if "Components" in data["Server"]:
            for component in data["Server"]["Components"]:
                versions.append(f"{component['Name']}: {component['Version']}")
        return '   '.join(versions)
    except Exception as e:
        msg = f"Unable to read docker data from {machine.name}: {e}"
        LOG.error(f"Unable to read docker data from {machine.name} , see debug log")
        LOG.debug(msg)
        return msg


def get_kernel_version(machine):
    cmd = "uname -rvi"
    try:
        result = machine.exec_pod_cmd(cmd, get_events=False, verbose=False)
        return result["logs"].strip()
    except Exception as e:
        msg = f"Unable to read kernel version from {machine.name}: {e}"
        LOG.error(f"Unable to read kernel version from {machine.name} , see debug log")
        LOG.debug(msg)
        return msg


def get_system_version(machine):
    cmd = ("set -a;"
           "cat /etc/redhat-release 2>/dev/null ||"
           "cat /etc/centos-release 2>/dev/null ||"
           " . /etc/os-release && echo $NAME $VERSION")
    try:
        result = machine.exec_pod_cmd(cmd, get_events=False, verbose=False)
        return result["logs"].strip()
    except Exception as e:
        msg = f"Unable to read system version from {machine.name}: {e}"
        LOG.error(f"Unable to read system version from {machine.name} , see debug log")
        LOG.debug(msg)
        return msg


def verify(expression, failure_msg, success_msg=None):
    """
    Function to perform soft assertions
    Args:
        :param expression: Expression that must be verified to be True
        :param failure_msg: Message to add failure if expression is False
        :param success_msg: Message to log if expression is True
    """
    if expression:
        if success_msg is not None:
            LOG.info(success_msg)
    else:
        assert False, failure_msg


def get_binary_path(binary):
    """
    Function to get binary full path. Searches in directory picked from
    configured as settings.SI_BINARIES_DIR, if not found fallback to
    directories specified in $PATH environment

    :param binary: The name of binary to look for
    :returns: full path to binary or None
    :raises Exception: when requested binary is not found
    """
    bin_path = os.path.join(settings.SI_BINARIES_DIR, binary)
    if os.path.isfile(bin_path):
        return bin_path
    result = exec_helpers.Subprocess().execute(
        f"which {binary}", verbose=False)
    if result.exit_code == 0:
        return result.stdout_str

    return '/binary/not/found'


def generate_upgrade_indexes(random_order=True, index_range=100):
    all_indexes = [i + 1 for i in range(index_range)]
    if random_order:
        random.shuffle(all_indexes)
    i = 0
    while True:
        yield all_indexes[i % index_range]
        i += 1


def read_event(event, read_event_data=True):
    if read_event_data:
        LOG.debug(f"Read event {event.name} from API")
        try:
            return event.read(cached=True)
        except ApiException:
            return None
    else:
        # Assume that 'event' already contains all required fields
        return event


def parse_events(events, event_prefix=None, sort=True, read_event_data=True, filtered_events=False,
                 truncate_events_num=10):
    grouped = {}
    for event in events:

        if event_prefix and not event.name.startswith(event_prefix):
            continue

        # Init 'data' variable with an Event object
        data = read_event(event, read_event_data)
        if data is None:
            continue

        if data.first_timestamp:
            data_first_date = str(data.first_timestamp.date())
            data_first_time = str(data.first_timestamp.time())
        else:
            data_first_date = ''
            data_first_time = ''
        if data.last_timestamp:
            data_last_date = str(data.last_timestamp.date())
            data_last_time = str(data.last_timestamp.time())
            data_last_timestamp = str(data.last_timestamp.timestamp())
        else:
            data_last_date = ''
            data_last_time = ''
            data_last_timestamp = ''

        parsed_data = {
            'data': data,
            'namespace': data.metadata.namespace,
            'event_date': data_first_date,
            'event_start': data_first_time,
            'event_end_date': data_last_date,
            'event_end_time': data_last_time,
            'event_end_timestamp': data_last_timestamp,
            'message': data.message,
            'reason': data.reason,
            'event_type': data.type,
            'component': data.source.component,
            'host': data.source.host or '-',
            'object_kind': data.involved_object.kind,
            'object_name': data.involved_object.name,
            'object_namespace': data.involved_object.namespace,
            'object_uid': data.involved_object.uid,
        }

        object_uid = data.involved_object.uid
        grouped.setdefault(object_uid, []).append(parsed_data)

    result = {}
    for group, events_data in grouped.items():
        # events_data.sort(key=itemgetter('event_end_time'))
        events_data.sort(key=itemgetter('event_end_timestamp'))
        fevent = events_data[-1]
        group_msg = (f"{fevent['object_kind']} {fevent['object_namespace']}/{fevent['object_name']}"
                     f" [Events from: {fevent['namespace']}/{fevent['component']}/{fevent['host']}]"
                     f" [Object ID: {group}]")

        if filtered_events:
            # Leave events which latest status is not 'Normal'
            if events_data[-1]['event_type'] != 'Normal':
                if truncate_events_num:
                    # Leave only {truncate_events_num} latest events
                    result[group_msg] = events_data[-truncate_events_num:]
                else:
                    result[group_msg] = events_data
            else:
                event = events_data[-1]
                LOG.debug(f"### Skipping log for events for {event['object_kind']}"
                          f" {event['object_namespace']}/{event['object_name']}"
                          f" from {event['namespace']}/{event['component']}/{event['host']}")
        else:
            if truncate_events_num:
                # Leave only {truncate_events_num} latest events
                result[group_msg] = events_data[-truncate_events_num:]
            else:
                result[group_msg] = events_data

    return result


def create_events_msg(events, prefix="  Events:", header=""):
    """Print events

    :param events: list of dicts, prepared in self.get_events()
    :param prefix: str, the forst lines in the output
    :param header: format string to show it before each group of events
    """
    message = []
    for group_msg, events_data in sorted(events.items()):
        if header:
            msg = header.format(group_msg=group_msg, events_data=events_data)
        else:
            msg = f"|| {group_msg} ||"
            msg = f"\n{'-' * len(msg)}\n{msg}\n{'-' * len(msg)}"
        message.append(msg)
        for event in events_data:
            message.append(f"  >> {event['event_end_date']} {event['event_end_time']} "
                           f"{event['event_type']} {event['reason']} "
                           f"{event['message']}")

    return f"{prefix}" + "\n".join(message) + "\n"


def get_credential_type_by_provider(provider_name: str) -> str:
    return {
        settings.AWS_PROVIDER_NAME: 'awscredentials',
        settings.AZURE_PROVIDER_NAME: 'azurecredentials',
        settings.OPENSTACK_PROVIDER_NAME: 'openstackcredentials',
        settings.VSPHERE_PROVIDER_NAME: 'vspherecredentials',
        settings.EQUINIXMETALV2_PROVIDER_NAME: 'equinixmetalcredentials',
        settings.BAREMETAL_PROVIDER_NAME: 'baremetalhostcredentials',
        settings.BYO_PROVIDER_NAME: 'byocredentials'
    }.get(provider_name)


def is_valid_ip(chk_str):
    """Check if passed str is valid ip or not

    :param chk_str: String to determine is IP valid or not
    :return:
    """
    LOG.info(f"Check IP address '{chk_str}'")
    try:
        ipaddress.ip_address(chk_str)
        LOG.info("IP address is correct")
        return True
    except ipaddress.AddressValueError:
        LOG.info(f"value for check: {chk_str}")
        LOG.error("Incorrect IP address")
        return False
    except ipaddress.NetmaskValueError:
        LOG.info(f"value for check: {chk_str}")
        LOG.error("Incorrect netmask address")
        return False
    except ValueError:
        LOG.info(f"It's not ip, looks like hostname: {chk_str}")
        return False


def get_provider_specific_path(base: str, provider_name: str, extra_path: str = None) -> str:
    """
    Function returns a provider-specific path depending on its type.

    I.e. for `openstack` literal it will return `base/extra_path`,

    for all other provider literals it will return `base/provider_name/extra_path`.
    """
    ret = base \
        if provider_name == settings.OPENSTACK_PROVIDER_NAME \
        else os.path.join(base, provider_name)
    if extra_path:
        ret = os.path.join(ret, extra_path)
    return ret


def save_cluster_name_artifact(namespace, name):
    cluster_name_path = "{0}/cluster_name".format(settings.ARTIFACTS_DIR)
    LOG.info(f"Save cluster namespace/name to '{cluster_name_path}'")
    with open(cluster_name_path, 'w') as f:
        f.write(f"{namespace}/{name}")


def is_reboot_condition(condition_type: string) -> bool:
    return condition_type is not None and \
        (condition_type == 'Reboot' or condition_type == 'RebootMachines')


def get_np_yaml_from_netconfigfiles(netconfigfiles):
    """Simple function to extract Netplan yaml from IPAM netconfigFiles list

    :netconfigfiles list: netconfigFiles field from IpamHost status
    :rtype: bytes
    """
    netplanPath = "/etc/netplan/60-kaas-lcm-netplan.yaml"
    encodedNetplan = ""
    for section in netconfigfiles:
        if section.get("path", "") == netplanPath:
            encodedNetplan = section.get("content", "")
            break
    return base64.b64decode(encodedNetplan)


def get_np_struct_from_netconfigfiles(netconfigfiles):
    """Simple function to extract Netplan struct from IPAM netconfigFiles list

    :netconfigfiles list: netconfigFiles field from IpamHost status
    :rtype: dict
    """
    netplanYaml = get_np_yaml_from_netconfigfiles(netconfigfiles)
    return yaml.safe_load(netplanYaml)


def remove_k8s_obj_annotations(si_clientobject, annotations_remove=None):
    """
    Remove annotation from k8s object.
    Example:
    >> remove_k8s_obj_annotations(bmh1, annotations_remove=['annotation1', 'annotation2'])

    :param K8sNamespacedResource si_clientobject: modified object
    :param list annotations_remove: list of annotations for remove
    """

    if not annotations_remove:
        LOG.warning("Annotations to remove are not set. Skipping")
        return

    assert isinstance(si_clientobject, K8sNamespacedResource), \
        f"si_clientobject must be instance of K8sNamespacedResource, type:{type(si_clientobject)}"
    assert isinstance(annotations_remove, list), "Please, use list of annotations keys"

    obj_typename = f"objectType:{type(si_clientobject)} with name {si_clientobject.name}"
    existing_annotations = si_clientobject.data.get('metadata', {}).get('annotations', {})
    annotations_patch = {}
    for annotation in annotations_remove:
        if annotation in existing_annotations:
            LOG.info(
                f"Annotation {annotation}: {existing_annotations[annotation]} will be removed from {obj_typename}"
            )
            annotations_patch[annotation] = None
        else:
            LOG.info(
                f"Annotation {annotation} not found in existing annotations for {obj_typename}. Skipping"
            )

    si_clientobject.patch({'metadata': {'annotations': annotations_patch}})


def generate_batch(lst, batch_size):
    """  Yields batch of specified size """
    if batch_size <= 0 or batch_size is None:
        LOG.warning(f"generate_batch: batch size is {batch_size}")
        return
    for i in range(0, len(lst), batch_size):
        yield lst[i: i + batch_size]


def check_downtime(downtime,
                   expected_downtime: int = settings.REFAPP_EXPECTED_DOWNTIME,
                   max_downtimes: int = 0,
                   raise_on_error: bool = True):
    """
    Check is existing downtime in list more than expected
    Args:
        downtime: list of downtime objects
        expected_downtime: maximum time in seconds for a single downtime period
        max_downtimes: maximum time in seconds of all downtimes found during the test
                       default = 0, which disables the check for total downtimes
    Returns: True when downtime is found and raise_on_error=False, False is no dowtime
    Raises Exception: when a downtime exceed expected_downtime, or total downtimes exceed max_downtime
    """
    assert expected_downtime >= 0, f"Expected single downtime is not positive integer {expected_downtime}"
    assert max_downtimes >= 0, f"Expected max downtime is not positive integer {max_downtimes}"

    if downtime:
        LOG.info("The following downtime ranges(s) found:\n" + tabulate(downtime, tablefmt="presto",
                                                                        headers="keys"))
    not_expected_downtime = list(filter(lambda x: x['duration'].total_seconds() > expected_downtime, downtime))
    total_downtimes = sum([x['duration'].total_seconds() for x in downtime])

    err_msg = ''
    if not_expected_downtime:
        err_msg += (f"The following downtime ranges(s) more that expected ({expected_downtime} s.)\n"
                    + tabulate(not_expected_downtime, tablefmt="presto", headers="keys") + "\n")
    if max_downtimes:
        if total_downtimes > max_downtimes:
            err_msg += f"Total downtimes is {total_downtimes}s which exceeds the expected maximum {max_downtimes}s\n"
    else:
        LOG.warning("Check for <max_downtimes> is disabled, please consider to enable it by set an expected value")

    if err_msg:
        LOG.error(err_msg)
        if raise_on_error:
            raise Exception(f"Current downtime during testing is longer than expected:\n{err_msg}")
        else:
            return err_msg != ''

    # Check passed
    LOG.info("Current downtimes are within expected limits")
    return False


def check_mosk_workloads_downtime(statistics, workload_type, single_downtime, max_downtimes):
    """
    Check whether each workload statistics satisfies requirements for single downtime period
    duration, and for total downtime duration.
    Args:
        statistics: dictionary of workloads statistics in format:
            <workload_name>:
              downtime:
                periods: <list of dictionaries>
                duration_total: <integer>
        workload_type: type of workload for example - openstack_instances_icmp
        single_downtime: integer duration of single downtime period in seconds
                         for workload
        max_downtimes: integer duration of all downtime periods in seconds
                       for workload
    Returns: None
    Raises AssertionError: when at least one of requirements isn't met.
    """
    assert single_downtime >= 0, f"Expected single downtime is not positive integer {single_downtime}"
    assert max_downtimes >= 0, f"Expected max downtime is not positive integer {max_downtimes}"

    check_succeeded = True
    for workload_name, data in statistics.items():
        try:
            downtime = data["downtime"]
            if downtime:
                LOG.info((f"The following downtime ranges(s) found for workload {workload_name}:\n" +
                          tabulate(downtime["periods"], tablefmt="presto", headers="keys")))
            unexpected_periods = list(filter(lambda x: x['duration'] > single_downtime, downtime["periods"]))
            downtime_total = downtime["duration_total"]
            assert not unexpected_periods, \
                (f"Found downtime periods longer than {single_downtime} seconds:\n"
                 + tabulate(unexpected_periods, tablefmt="presto", headers="keys"))
            assert max_downtimes >= downtime_total, \
                f"Total downtime {downtime_total} duration is bigger than {max_downtimes} seconds\n"
        except AssertionError as e:
            LOG.error(f"Workload {workload_name} of type {workload_type} failed downtime check.\n{e}")
            check_succeeded = False
    assert check_succeeded, f"MOSK {workload_type} downtime check is failed"


def get_distribution_relevance(clusterrelease_data):
    """
    :param clusterrelease_data: dict with clusterrelease data
    :return: dict with distribution relevance
    """
    release_name = clusterrelease_data.get('metadata', {}).get('name', '')
    allowed = clusterrelease_data.get('spec', {}).get('allowedDistributions', [])
    default = [d for d in allowed if d.get('default', False)]
    assert default, (f"No default distribution found for clusterrelease {release_name} "
                     f"in allowedDistributions: \n{yaml.dump(allowed)}")
    assert len(default) == 1, (f"More then one default distribution found for clusterrelease {release_name} "
                               f"in allowedDistributions: \n{yaml.dump(allowed)}")
    default = default[0]
    allowed_sorted = sorted(allowed, key=lambda d: d['version'])
    latest = allowed_sorted[-1]
    oldest = allowed_sorted[0]
    return {'default': default, 'oldest': oldest, 'latest': latest}


def get_distribution_for_node(kaas_manager, node, release_name):
    distribution = node.get('distribution', None)
    if distribution:
        LOG.info(f"Distribution {distribution} is taken from node data")
        return distribution
    if settings.DISTRIBUTION_RELEVANCE != 'default':
        d_rel = settings.DISTRIBUTION_RELEVANCE
        allowed = kaas_manager.get_allowed_distributions(release_name)
        assert d_rel in allowed.keys(), (f"Wrong DISTRIBUTION_RELEVANCE value set. Should be one of "
                                         f"['default', 'oldest', 'latest'], but got {d_rel}")
        distribution = allowed[d_rel]['id']
        LOG.info(f"DISTRIBUTION_RELEVANCE is set to {d_rel}. Will be used {distribution} for machines")
        return distribution
    else:
        LOG.info("DISTRIBUTION_RELEVANCE is set to default. Will skip setting distribution to machine spec directly")
        return None


def kaas_greater_than_2_26_0(kaas_manager) -> bool:
    actual_kaas_version = kaas_manager.get_mgmt_cluster().get_kaasrelease_version()
    return version_greater_than_2_26_0(actual_kaas_version)


def version_greater_than_2_26_0(version_str) -> bool:
    parsed_version = version.parse(version_str)
    cap_version = version.parse("kaas-2-26-0-rc")
    return parsed_version >= cap_version


def version_greater_or_equal_2_28_0_rc(version_str) -> bool:
    parsed_version = version.parse(version_str)
    cap_version = version.parse("kaas-2-28-0-rc")
    return parsed_version >= cap_version


def clusterrelease_version_greater_than_or_equal_to_kaas_2_30_0(cluster_release_version) -> bool:
    """Ceck if clusterrelease version is greater than or equal to version
    introduced in kaas 2.30.0
    :param cluster_release_version: str, version of clusterrelease
    """
    if not cluster_release_version or cluster_release_version == "":
        return False
    parsed_version = version.parse(cluster_release_version)
    # Remove pre-release information
    clean_version = version.parse(parsed_version.base_version)
    if version.parse("20.0.0") <= clean_version < version.parse("21.0.0"):
        return True
    if clean_version >= version.parse("21.0.0"):
        return True
    return False


def clusterrelease_version_greater_than_or_equal_to_kaas_2_31_0(cluster_release_version) -> bool:
    """Ceck if clusterrelease version is greater than or equal to version
    introduced in kaas 2.31.0
    :param cluster_release_version: str, version of clusterrelease
    """
    if not cluster_release_version or cluster_release_version == "":
        return False
    parsed_version = version.parse(cluster_release_version)
    # Remove pre-release information
    clean_version = version.parse(parsed_version.base_version)
    if version.parse("20.1.0") <= clean_version < version.parse("21.0.0"):
        return True
    if clean_version >= version.parse("21.1.0"):
        return True
    return False


def get_datetime_utc(datetime_str, format_str="%Y-%m-%dT%H:%M:%SZ"):
    return datetime.strptime(datetime_str, format_str).astimezone(pytz.UTC)


def parse_time_value(time_string):
    """Parse time string into h m s format and return dict of values

    :param: string with time in format: 1h4m5.98787698s
    :return: hours, minutes, seconds
    """
    pattern = r'(?:(\d+)h)*(?:(\d+)m)*(?:(\d+\.*\d*)s)*'
    match = re.match(pattern, time_string)
    parsed_time = {}
    if match:
        hours = int(match.group(1)) if match.group(1) else 0
        minutes = int(match.group(2)) if match.group(2) else 0
        seconds = int(float(match.group(3))) if match.group(3) else 0
        parsed_time = {"hours": hours, "minutes": minutes, "seconds": seconds}
    return parsed_time


# As we have npTemplate in l2template as string, then it requires some magic to append some keys correctly
def add_section_to_str_np_template(template_str: str, net_block: str, net_type: str) -> str:
    """
    :param template_str: npTemplate data as str. Could be obtained via l2template['spec']['npTemplate']
    :param net_type: Could be any existed type. Like 'vlans:', 'bridges:', 'ethernets:', 'bonds:'
    :param net_block: Block to add. For example, need to add new vlan to vlan section, then net_type="vlans"
                        net_block='si-fake-vlan:
                                     id: 999
                                     link: bond0
                                     addresses:
                                       - {{ ip "si-fake-vlan:si-test-fake-subnet" }}'
    :return: string with added section
    """
    lines = template_str.strip().splitlines()
    output_lines = []
    inserted = False

    for idx, line in enumerate(lines):
        output_lines.append(line)

        if not inserted and line.strip().startswith(f"{net_type}:"):
            base_indent = len(line) - len(line.lstrip())
            net_type_indent = base_indent + 2

            for j in range(idx + 1, len(lines)):
                next_line = lines[j]
                if (len(next_line) - len(next_line.lstrip())) <= base_indent:
                    break
            indented_net_block = [
                " " * net_type_indent + line
                for line in net_block.strip().splitlines()
            ]
            output_lines.extend(indented_net_block)
            inserted = True

    if not inserted:
        raise ValueError(f"Section {net_type} is not found in template")

    return "\n".join(output_lines)


def filter_ansible_log(message):
    """Filter Ansible log output

    :param message: str, multiline string with Ansible log
    :return: list of str, which contains:
             - PLAY, TASK and RUNNING HANDLER names
             - TASK and RUNNING HANDLER content, if it doesn't include
               a status keyword like 'ok:', 'changed:' or 'skipping:'
             - Couple of edge cases at the end of parsing the message
    """
    try:
        occurrences = yaml.safe_load(message)
    except Exception as e:
        # Unexpected message format
        LOG.error(f"Error while decoding YAML message: {e}")
        return str(message).splitlines()
    if not occurrences:
        return []
    first_occurrence = sorted(occurrences, key=lambda x: x.get('lastOccurrenceDate'))[0].get('content', '')
    if "\nTASK [" not in first_occurrence:
        # Looks like it is not an Ansible log. Unknown content, return as-is
        return first_occurrence.splitlines()

    # Separate different tasks in list of lines lists
    results = []
    current_lines = []
    for line in first_occurrence.splitlines():
        new_block = line.startswith('PLAY [') or line.startswith('RUNNING HANDLER [') or line.startswith('TASK [')
        if new_block and current_lines:
            results.append(current_lines)
            current_lines = []
        current_lines.append(line)
    if current_lines:
        results.append(current_lines)

    if not results:
        return []

    filtered_tasks = []
    for result in results:
        if result[0].startswith('PLAY ['):
            # Don't contain any statuses, just log the name
            filtered_tasks.append(result[0])
        elif result[0].startswith('RUNNING HANDLER [') or result[0].startswith('TASK ['):
            if any(res.startswith("skipping: ") or
                   res.startswith("ok: ") or
                   res.startswith("changed: ")
                   for res in result):
                # Task with non-error status
                filtered_tasks.append(result[0])
            else:
                # Task don't contain any good status, log the task content
                filtered_tasks.extend(result)

    if result[0].startswith('PLAY ['):
        # if the latest task is PLAY with no tasks after, then something wrong happened on preparing
        filtered_tasks.extend(result[1:])

    if not filtered_tasks:
        # If no any "good" task logged, then log the latest result
        filtered_tasks.extend(results[-1])
    return filtered_tasks


def is_k8s_res_exist(k8s_res: K8sNamespacedResource):
    try:
        k8s_res.read()
    except ApiException as ex:
        try:
            body = json.loads(ex.body)
        except (TypeError, json.decoder.JSONDecodeError):
            body = {}
        if (str(ex.status) == "404" and ex.reason == "Not Found") or \
                (body.get('reason') == "NotFound" and str(body.get('code')) == "404"):
            LOG.info(f"Resource {k8s_res} is not found.")
        else:
            LOG.error(f"Got api error {ex.body} while finding {k8s_res}.")
        return False
    LOG.info(f"Resource {k8s_res} is present.")
    return True


def get_schedule_for_cronjob(time_delta):
    """Get schedule in Cron-like manner for future run.

    :param time_delta: Seconds between now() and planned time
    :return: Cron-like schedule for future
    """
    schedule = datetime.now() + timedelta(seconds=time_delta)
    return schedule.astimezone(pytz.utc).strftime("%M %H %d %m *")


def ssh_k8s_node(hostname):
    with open(settings.NODES_INFO) as f:
        dictionary = yaml.safe_load(f)
    private_key = dictionary.get(hostname, {}).get('ssh')['private_key']
    assert private_key, "Private key is not defined"
    pkey = get_rsa_key(private_key)
    node_ip = dictionary.get(hostname, {}).get('ip')['address']
    username = dictionary.get(hostname, {}).get('ssh')['username']
    auth = exec_helpers.SSHAuth(username=username, password='', key=pkey)
    ssh = exec_helpers.SSHClient(host=node_ip, port=22, auth=auth)
    ssh.logger.addHandler(logger.console)
    ssh.sudo_mode = True
    return ssh


@retry(paramiko.ssh_exception.NoValidConnectionsError, delay=10, tries=4, logger=LOG)
def basic_ssh_command(cmd, host, user, password='', private_key='', port=22, verbose=True, timeout=120):
    assert password or private_key, "Auth method is not defined!"
    if private_key:
        auth = exec_helpers.SSHAuth(username=user, password='', key=private_key)
    else:
        auth = exec_helpers.SSHAuth(username=user, password=password)
    ssh = exec_helpers.SSHClient(host=host, port=port, auth=auth)
    ssh.logger.addHandler(logger.console)
    result = ssh.execute(cmd, verbose=verbose, timeout=timeout)
    return result


def get_dpdk_driver_package(distro):
    res = None
    if settings.OPENSTACK_DPDK_DRIVER_PACKAGE:
        res = settings.OPENSTACK_DPDK_DRIVER_PACKAGE
    else:
        res = settings.DEFAULT_OPENSTACK_DPDK_DRIVER_PACKAGE_BY_DISTRO.get(distro)
    assert res, "DPDK package is not specified."
    return res


def get_services_pids(machine, services, ssh_key):
    """Return dict with pids: service name for services

    :params
        machine: machine object on which services work
        services: list of services
        ssh_key: key for access on machine
    :return:
        dict [ service_name : pid ]
    """
    pids = {}
    result = machine._run_cmd(
        "ps -xaeo pid,command", verbose=False,
        ssh_key=ssh_key).stdout
    for line in result:
        line = line.decode().strip()
        for service in services:
            if service in line:
                pid = line.split()[0]
                if pid:
                    pids[service] = pid
    return pids


def render_child_data(si_config: SIConfigManager, render_opts=None, extra_data=None):
    if extra_data is None:
        extra_data = dict()
    if render_opts is None:
        render_opts = dict()
    # hack, required only for -bm virtual env.
    # Bz of specific logic, half of params required for child render,
    # rendered during bootstrap process,
    j2_extra_vars = si_config.get_ansible_state_env_config()

    default_render_opts = {
        "mgmt_version": 'RENDERED_mgmt_version',
        "target_region": 'RENDERED_region',
        "target_cluster": 'RENDERED_cluster_name',
        "target_namespace": 'RENDERED_namespace_name',
        "machine_distribution": 'RENDERED_machine_distribution',
        "pub_key_content": 'RENDERED_pub_key_content',
    }
    render_opts_data = {**default_render_opts, **render_opts}
    LOG.debug(f"render_child_data resulted render_opts_data:\n{render_opts_data}")
    # Gather nodes information from yaml file
    _rendered_data = templates_utils.render_template(settings.BM_CHILD_SETTINGS_YAML_PATH,
                                                     options=render_opts_data,
                                                     extra_env_vars=j2_extra_vars)
    child_data = yaml.safe_load(_rendered_data)[0]

    LOG.debug(f"Rendered child_data from file:\n"
              f"{settings.BM_CHILD_SETTINGS_YAML_PATH}:\n"
              f"{child_data}")
    if extra_data:
        # for some cases, we might need to provide extra data.
        # This is completely free-form dict.
        child_data['extra_data'] = extra_data
        LOG.debug(f"Final rendered child_data with extra_data, file:\n"
                  f"{settings.BM_CHILD_SETTINGS_YAML_PATH}:\n"
                  f"{child_data}")
    # save for debug purposes only
    with open(os.path.join(settings.ARTIFACTS_DIR, 'si_debug_rendered_child_data.txt'), "w") as f:
        f.write(_rendered_data)
    # .yaml actually could be reused in tests
    with open(os.path.join(settings.ARTIFACTS_DIR, 'si_debug_rendered_child_data.yaml'), "w") as f:
        yaml.dump(child_data, f, default_flow_style=False)
    return child_data


def get_string_as_bool(data, default: bool) -> bool:
    """
    Parse input string|bool into bool
    """
    boolean_states = settings.get_boolean_states_dict()
    return boolean_states.get(data.lower(), default)


def is_rockoon_used(k8s_api):
    for pod in k8s_api.pods.list_starts_with(
            "rockoon", "osh-system"
    ):
        return True
    return False
