import base64
import copy
import idna
import os
import yaml

from OpenSSL import SSL
from OpenSSL.SSL import SysCallError, Error
from cryptography import x509
from socket import socket, gaierror
from cryptography.x509 import Certificate

from si_tests import logger, settings
from si_tests.managers.kaas_manager import Manager, Cluster
from si_tests.utils import waiters, utils
from si_tests.utils import exceptions as timeout_exceptions

LOG = logger.logger


def get_certificate_from_host(hostname, port=443):
    hostname_idna = idna.encode(hostname)
    sock = socket()
    crypto_cert = None

    try:
        sock.connect((hostname, port))
        ctx = SSL.Context(SSL.SSLv23_METHOD)
        ctx.check_hostname = False
        ctx.verify_mode = SSL.VERIFY_NONE

        sock_ssl = SSL.Connection(ctx, sock)
        sock_ssl.set_connect_state()
        sock_ssl.set_tlsext_host_name(hostname_idna)
        sock_ssl.do_handshake()
        cert = sock_ssl.get_peer_certificate()
        crypto_cert = cert.to_cryptography()
        sock_ssl.close()
        sock.close()
    except (gaierror, timeout_exceptions.TimeoutError, TimeoutError, SysCallError, Error) as e:
        LOG.error(f"Socket connection error!\n{e}")

    return crypto_cert


def app_to_tlsspec(app):
    temp = app.split('-')
    tlsSpec = temp[0] + ''.join(ele.title() for ele in temp[1:])
    tlsSpec = tlsSpec.replace("Alertmanager", "AlertManager")
    return tlsSpec


def get_tls_status(cluster: Cluster, app_tlsspec):
    """Get Cluster status.providerStatus.tls.{app_tlsspec}"""
    tls_statuses = cluster.get_tls_statuses()
    tls_status = tls_statuses.get(app_tlsspec, {})
    return tls_status


def check_old_tlsstatus_changed(cluster: Cluster, app_tlsspec, old_tlsstatus):
    new_tls_status = get_tls_status(cluster, app_tlsspec)
    if get_tls_status(cluster, app_tlsspec) == old_tlsstatus:
        LOG.info(f"status.providerStatus.tls.{app_tlsspec} is not changed yet")
        return False
    else:
        LOG.info(f"status.providerStatus.tls.{app_tlsspec} has been changed")
        LOG.info(f"Old tls status:\n{yaml.dump(old_tlsstatus)}")
        LOG.info(f"New tls status:\n{yaml.dump(new_tls_status)}")
        return True


def get_tls_externalconfigapplied(cluster, app_tlsspec):
    """Get the flag externalConfigApplied from the Cluster status for an application configured with TLSConfig

    Example:

      status:
        providerStatus:
          tls:
            ...
            iamProxyAlerta:
              expirationTime: "2024-07-24T16:56:14Z"
              externalConfigApplied: true
              hostname: alerta.stacklight.child-cl.child-ns-ovs.local
    """
    tls_status = get_tls_status(cluster, app_tlsspec)
    is_applied = tls_status.get('externalConfigApplied')
    LOG.info(f"{app_tlsspec}.externalConfigApplied = {is_applied}")
    return is_applied


def get_tls_externalconfigdisappeared(cluster, app_tlsspec):
    """Check the flag externalConfigApplied is not present in the Cluster status

    Check the flag externalConfigApplied is not present in the Cluster status for
        an application configured with TLSConfig

    Example:

      status:
        providerStatus:
          tls:
            ...
            iamProxyAlerta:
              expirationTime: "2024-07-24T16:56:14Z"
              externalConfigApplied: true
              hostname: alerta.stacklight.child-cl.child-ns-ovs.local
    """
    tls_status = get_tls_status(cluster, app_tlsspec)
    if not tls_status:
        return False
    if tls_status.get('externalConfigApplied') is None:
        LOG.info(f"{app_tlsspec}.externalConfigApplied disappeared")
        return True
    return False


def wait_for_cert_applied(cert_pem, hostname, cluster, app_tlsspec, old_tlsstatus):
    # If TLSConfig was applied for this service before, then wait until 'externalConfigApplied' is cleared
    LOG.info(f"Wait until old TLSConfig for '{app_tlsspec}' "
             f"is changed in the Cluster status.providerStatus.tls.{app_tlsspec}")
    waiters.wait(
        lambda: check_old_tlsstatus_changed(cluster, app_tlsspec, old_tlsstatus),
        interval=30, timeout=600,
        timeout_msg=f"TLSConfig was not applied for the '{app_tlsspec}', previous config is still active")

    # Wait until TLSConfig will be applied in the Cluster status
    LOG.info(f"Wait until new TLSConfig for '{app_tlsspec}' is applied")
    waiters.wait(
        lambda: get_tls_externalconfigapplied(cluster, app_tlsspec),
        interval=10, timeout=600,
        timeout_msg=f"TLSConfig was not applied for the '{app_tlsspec}'")

    new_tls_status = get_tls_status(cluster, app_tlsspec)
    LOG.info(f"Applied tls_status:\n{yaml.dump(new_tls_status)}")

    # Check that application actually has the new certificate
    LOG.info(f"Wait until new certificate will be available on '{hostname}'")
    x509_cert = x509.load_pem_x509_certificate(cert_pem.encode())
    waiters.wait(
        lambda: get_certificate_from_host(hostname) == x509_cert,
        interval=10, timeout=600,
        timeout_msg=f"Certificate not changed for '{hostname}'")

    LOG.info(f"TLSConfig for '{app_tlsspec}' was successfuly applied, new certificate is available on '{hostname}'")


def wait_for_default_tls_applied(app: str, cluster: Cluster, old_x509_cert: Certificate):
    """Wait until TLSConfig will be removed and default certificate will be applied for the application"""
    experation_time = get_tls_status(cluster, app).get("expirationTime")

    waiters.wait(
        lambda: get_tls_externalconfigdisappeared(cluster, app_to_tlsspec(app)),
        interval=10, timeout=600,
        timeout_msg=f"Usage of TLSConfig was not removed for the '{app}'")

    def check_esperation_time(new_time):
        if new_time is None:
            return False

        return new_time != experation_time

    waiters.wait(
        lambda: check_esperation_time(get_tls_status(cluster, app).get("expirationTime")),
        interval=10, timeout=600,
        timeout_msg=f"Expiration date hasn't been changed for the '{app}'")

    waiters.wait(
        lambda: get_tls_status(cluster, app).get("hostname") is not None,
        interval=10, timeout=600,
        timeout_msg=f"Hostname hasn't been set for the '{app}'")

    hostname = get_tls_status(cluster, app).get("hostname")

    waiters.wait(
        lambda: get_certificate_from_host(hostname) != old_x509_cert,
        interval=10, timeout=600,
        timeout_msg=f"A new certificate hasn't been set for the '{app}'"
        )


def update_seed_kubeconfig(bootstrap, cluster, new_ca_pem=None, skip_tls_verify=False, new_server_url=None):
    """Update the KUBECONFIG with the provided CA, or disable TLS certificate verify

    :param new_ca_pem: multistring, PEM with the new CA for the KUBECONFIG
    :param skip_tls_verify: boolean, skip TLS certificate verity if True
    :param new_server_url: string, new URL for the KUBECONFIG cluster: 'https://{server_ip or server_hostname}:443'

    Only one of 'new_ca_pem' or 'skip_tls_verify' can be used, not both.

    :rtype string: local path to the updated KUBECONFIG file
    """
    if new_ca_pem and skip_tls_verify:
        raise Exception(f"Cannot update KUBECONFIG for the cluster '{cluster.namespace}/{cluster.name}': "
                        f"'certificate-authority-data' and 'insecure-skip-tls-verify' cannot be configured for the "
                        f"same cluster at the same time, please use just one of these options")

    if cluster.is_management:
        kubeconfig_name = "management_kubeconfig"
        kubeconfig_path = os.path.join(settings.ARTIFACTS_DIR, kubeconfig_name)
    elif cluster.is_regional:
        kubeconfig_name = "regional_kubeconfig"
        kubeconfig_path = os.path.join(settings.ARTIFACTS_DIR, kubeconfig_name)
    else:
        kubeconfig_name = f"{cluster.name}-kubeconfig"
        kubeconfig_path = os.path.join(settings.ARTIFACTS_DIR, kubeconfig_name)
        # raise Exception("Only MGMT or Regional cluster available")

    LOG.info("Store kubeconfig")
    _, kubeconfig = cluster.get_kubeconfig_from_secret()
    kubeconfig = yaml.load(kubeconfig, Loader=yaml.SafeLoader)

    if new_ca_pem:
        ca_bytes = new_ca_pem.encode("utf-8")
        kubeconfig['clusters'][0]['cluster'].pop('insecure-skip-tls-verify', None)
        kubeconfig['clusters'][0]['cluster']['certificate-authority-data'] = base64.b64encode(ca_bytes).decode("utf-8")
    if skip_tls_verify:
        kubeconfig['clusters'][0]['cluster'].pop('certificate-authority-data', None)
        kubeconfig['clusters'][0]['cluster']['insecure-skip-tls-verify'] = True
    if new_server_url:
        # new_server_url = 'https://{server_ip or server_hostname}:443'
        kubeconfig['clusters'][0]['cluster']['server'] = new_server_url

    with open(kubeconfig_path, 'w') as kubeconfig_file:
        yaml.dump(kubeconfig, kubeconfig_file)
    LOG.info(f"New kubeconfig data stored to: {kubeconfig_path}")

    target_kubeconfig_name = "kubeconfig" if cluster.is_management else f"kubeconfig-{cluster.name}"
    target_kubeconfig_path = bootstrap.get_remote_kubeconfig_path(kubeconfig_remote_name=target_kubeconfig_name)
    if target_kubeconfig_path:
        LOG.info("Update kubeconfig on seed node")
        remote = bootstrap.remote_seed()
        remote.execute(f"cp {target_kubeconfig_path} {target_kubeconfig_name}_prev_cert")
        remote.upload(kubeconfig_path, target_kubeconfig_path)
        LOG.info(f"Updated kubeconfig path: {target_kubeconfig_path}")
    else:
        LOG.info(f"Kubeconfig with name '{target_kubeconfig_name}' not found on seed node, skipping upload")
    return kubeconfig_path


class CertManager:
    def __init__(self, kaas_manager: Manager):
        self._kaas_manager = kaas_manager

    @property
    def kaas_manager(self) -> Manager:
        return self._kaas_manager

    def apply_cert_for_app(self, app, cluster: Cluster, hostname,
                           cert_pem, key_pem, ca_pem=None, app_tlsspec=None):
        app_tlsspec = app_tlsspec or app_to_tlsspec(app)
        old_tlsstatus = get_tls_status(cluster, app_tlsspec)
        rnd = utils.gen_random_string(3)
        tlsconfig = self.kaas_manager.create_tlsconfig(
            cluster=cluster,
            name=f"{app}-{rnd}-{cluster.name}", namespace=cluster.namespace,
            hostname=hostname, cert_pem=cert_pem, key_pem=key_pem, ca_pem=ca_pem)
        cluster.update_tlsconfigref_for_app(app=app_tlsspec, tlsconfigref=tlsconfig.name)
        wait_for_cert_applied(cert_pem, hostname, cluster, app_tlsspec, old_tlsstatus)

    def remove_cert_for_app(self, app: str, cluster: Cluster, app_tlsspec: str = None, old_cert: Certificate = None):
        app_tlsspec = app_tlsspec or app_to_tlsspec(app)

        tls_body: dict = copy.deepcopy(cluster.spec["providerSpec"]["value"]["tls"])
        LOG.info("Current Cluster '%s' spec tls config '%s'", cluster.name, tls_body)

        tlsconfig_name = tls_body[app_tlsspec]["tlsConfigRef"]
        tls_body[app_tlsspec] = None

        body = {
            "spec": {
                "providerSpec": {
                    "value": {
                        "tls": tls_body
                    }
                }
            }
        }
        cluster.patch(body)
        wait_for_default_tls_applied(app_tlsspec, cluster, old_cert)

        tlsconfigs = self.kaas_manager.get_tls_configs(namespace=cluster.namespace)
        app_tlsconfigs = [tlsconfig for tlsconfig in tlsconfigs if tlsconfig.name == tlsconfig_name]

        if not app_tlsconfigs:
            LOG.error("No TLSConfigs found for the '%s'", app)
            return

        for config in app_tlsconfigs:
            LOG.info("Delete TLSConfig '%s'", config.name)
            config.delete()
